<?php
/*
Plugin Name: ACF Polylang
Description: Polylang integration for ACF.
Version:     1.1.5
Author:      Le Web simple <pascal@lewebsimple.ca>
Author URI:  https://lewebsimple.ca
Text Domain: acf-polylang
Domain Path: /languages
*/

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'ACF_Polylang' ) ) :

	class ACF_Polylang {

		/** @var string The plugin version number. */
		var $version = '1.1.2';

		/**
		 * ACF_Polylang constructor.
		 */
		function __construct() {
			// Do nothing.
		}

		/**
		 * Initialize ACF Polylang
		 */
		function initialize() {

			// Bail early if already initialized
			if ( defined( 'ACF_POLYLANG_VERSION' ) ) {
				return;
			}

			// Define constants
			$this->define( 'ACF_POLYLANG_VERSION', $this->version );

			// Configure ACF for translation without duplicating field groups
			add_filter( 'acf/settings/l10n_textdomain', array( $this, 'l10n_textdomain' ) );
			add_filter( 'pll_get_post_types', array( $this, 'pll_get_post_types' ), 100, 2 );

			// Enhance ACF / Polylang
			add_action( 'publish_acf-field-group', array( $this, 'sanitize_field_group_key' ), 10, 2 );
			add_filter( 'pll_the_language_link', array( $this, 'pll_the_language_link' ), 10, 2 );
			if ( self::is_acf_admin() ) {
				add_action( 'pre_get_posts', array( $this, 'field_groups_orderby_title' ) );
			}

			// Export field group / field translations automatically
			add_action( 'save_post', array( $this, 'export_translations' ) );

			// Translate field types
			add_filter( 'acf/translate_field/type=date_picker', array( $this, 'translate_datetime_field' ) );
			add_filter( 'acf/translate_field/type=date_time_picker', array( $this, 'translate_datetime_field' ) );
			add_filter( 'acf/translate_field/type=time_picker', array( $this, 'translate_datetime_field' ) );
			add_filter( 'acf/translate_field/type=number', array( $this, 'translate_number_field' ) );
			add_filter( 'acf/translate_field/type=page', array( $this, 'translate_page_field' ) );
			add_filter( 'acf/translate_field', array( $this, 'translate_field_placeholder' ) );

		}

		/**
		 * Override ACF textdomain to translate only when needed
		 *
		 * @param $textdomain
		 *
		 * @return array|false|string
		 */
		function l10n_textdomain( $textdomain ) {
			// Skip textdomain in ACF admin to prevent translating twice
			if ( ACF_Polylang::is_acf_admin() ) {
				return $textdomain;
			}
			return apply_filters( 'acf_polylang_textdomain', wp_get_theme()->get( 'TextDomain' ) ?: $textdomain );
		}

		/**
		 * Disable ACF field group translation
		 *
		 * @param $post_types
		 * @param $is_settings
		 *
		 * @return mixed
		 */
		function pll_get_post_types( $post_types, $is_settings ) {
			if ( isset( $post_types['acf-field-group'] ) ) {
				unset( $post_types['acf-field-group'] );
			}

			return $post_types;
		}

		/**
		 * Determine field group key from sanitized title
		 *
		 * @param $post_id
		 * @param $post
		 */
		function sanitize_field_group_key( $post_id, $post ) {
			// Check ACF field group for existing key
			if ( strpos( $post->post_name, 'group_' ) === 0 ) {
				return;
			}
			$key = 'group_' . str_replace( '-', '_', sanitize_title( $post->post_title ) );

			$_POST['acf_field_group']['key']    = $key;
			$_REQUEST['acf_field_group']['key'] = $key;
		}

		/**
		 * Preserve query arguments in language switcher
		 *
		 * @param $url
		 * @param $slug
		 *
		 * @return string
		 */
		function pll_the_language_link( $url, $slug ) {
			return add_query_arg( $_GET, $url );
		}

		/**
		 * Order field groups by title in ACF admin
		 *
		 * @param WP_Query $query
		 */
		function field_groups_orderby_title( WP_Query $query ) {
			if ( $query->is_main_query() && $query->get( 'post_type' ) === 'acf-field-group' ) {
				$query->set( 'orderby', 'title' );
			}
		}

		/**
		 * Export field translations automatically to acf-json/acf-translations.php
		 *
		 * @param $post_id
		 */
		function export_translations( $post_id ) {
			if ( get_post_type( $post_id ) !== 'acf-field-group' || get_post_status( $post_id ) !== 'publish' ) {
				return;
			}
			if ( ! file_exists( $export_path = acf_get_setting( 'save_json' ) ) ) {
				return;
			}
			$textdomain = apply_filters( 'acf_polylang_textdomain', wp_get_theme()->get( 'TextDomain' ) ?: $textdomain )
			$output     = '';
			foreach ( self::get_all_translations() as $translation ) {
				$output .= '__( "' . addcslashes( $translation, '"' ) . '", \'' . $textdomain . '\' );' . PHP_EOL;
			}
			file_put_contents( $export_path . '/acf-translations.php', '<?php' . PHP_EOL . $output );
		}

		/**
		 * Get all translations from ACF field groups and fields
		 *
		 * @return array
		 */
		private function get_all_translations() {
			$translations = array();
			foreach ( acf_get_field_groups() as $field_group ) {
				$translations = array_merge( $translations, $this->get_field_group_translations( $field_group ) );
			}
			$translations = array_unique( $translations );
			sort( $translations );

			return $translations;
		}

		/**
		 * Get translations from a specific field group
		 *
		 * @param $field_group
		 *
		 * @return array
		 */
		private function get_field_group_translations( $field_group ) {
			if ( empty( $field_group['ID'] ) || get_post_type( $field_group['ID'] ) !== 'acf-field-group' ) {
				return array();
			}
			$translations = array( get_the_title( $field_group['ID'] ) );
			foreach ( acf_get_fields( $field_group['key'] ) as $field ) {
				$translations = array_merge( $translations, $this->get_field_translations( $field ) );
			}

			return $translations;
		}

		/**
		 * Get translations from a specific field
		 *
		 * @param $field
		 *
		 * @return array
		 */
		private function get_field_translations( $field ) {
			$translations = array();
			if ( empty( $field['type'] ) ) {
				return $translations;
			}

			// Field properties
			$properties = array(
				'label',
				'instructions',
				'placeholder',
				'message',
				'button_label',
				'next_text',
				'previous_text',
				'ui_on_text',
				'ui_off_text'
			);
			foreach ( $properties as $property ) {
				if ( ! empty( $field[ $property ] ) ) {
					$translations[] = $field[ $property ];
				}
			}

			// Number field
			if ( $field['type'] === 'number' ) {
				if ( ! empty( $field['prepend'] ) ) {
					$translations[] = $field['prepend'];
				}
				if ( ! empty( $field['append'] ) ) {
					$translations[] = $field['append'];
				}
			}

			// Date field
			if ( in_array( $field['type'], array( 'date_picker', 'date_time_picker', 'time_picker' ) ) ) {
				$translations[] = $field['return_format'];
			}

			// Field choices
			if ( ! empty( $field['choices'] ) ) {
				foreach ( $field['choices'] as $choice_value => $choice_label ) {
					$translations[] = $choice_label;
				}
			}

			// Sub-fields
			if ( ! empty( $field['sub_fields'] ) ) {
				foreach ( $field['sub_fields'] as $sub_field ) {
					$translations = array_merge( $translations, $this->get_field_translations( $sub_field ) );
				}
			}

			// Layouts
			if ( ! empty( $field['layouts'] ) ) {
				foreach ( $field['layouts'] as $layout ) {
					$translations[] = $layout['label'];
					if ( ! empty( $layout['sub_fields'] ) ) {
						foreach ( $layout['sub_fields'] as $sub_field ) {
							$translations = array_merge( $translations, $this->get_field_translations( $sub_field ) );
						}
					}
				}
			}

			return $translations;
		}

		/**
		 * Translate date_picker / date_time_picker / time_picker fields
		 *
		 * @param $field
		 *
		 * @return mixed
		 */
		function translate_datetime_field( $field ) {
			$textdomain             = apply_filters( 'acf/settings/l10n_textdomain', 'acf-polylang' );
			$field['return_format'] = __( $field['return_format'], $textdomain );

			return $field;
		}

		/**
		 * Translate number fields
		 *
		 * @param $field
		 *
		 * @return mixed
		 */
		function translate_number_field( $field ) {
			$textdomain = apply_filters( 'acf/settings/l10n_textdomain', 'kaliroots' );
			if ( ! empty( $field['prepend'] ) ) {
				$field['prepend'] = __( $field['prepend'], $textdomain );
			}
			if ( ! empty( $field['append'] ) ) {
				$field['append'] = __( $field['append'], $textdomain );
			}

			return $field;
		}

		/**
		 * Translate page fields
		 *
		 * @param $field
		 *
		 * @return mixed
		 */
		function translate_page_field( $field ) {
			$textdomain = apply_filters( 'acf/settings/l10n_textdomain', 'kaliroots' );
			if ( ! empty( $field['next_text'] ) ) {
				$field['next_text'] = __( $field['next_text'], $textdomain );
			}
			if ( ! empty( $field['previous_text'] ) ) {
				$field['previous_text'] = __( $field['previous_text'], $textdomain );
			}

			return $field;
		}

		/**
		 * Translate field placeholder
		 *
		 * @param $field
		 *
		 * @return mixed
		 */
		function translate_field_placeholder( $field ) {
			$textdomain = apply_filters( 'acf/settings/l10n_textdomain', 'kaliroots' );
			if ( ! empty( $field['placeholder'] ) ) {
				$field['placeholder'] = __( $field['placeholder'], $textdomain );
			}

			return $field;
		}

		/**
		 * Determine if currently in ACF admin
		 *
		 * @return bool
		 */
		static function is_acf_admin() {
			if ( isset( $_REQUEST['post_type'] ) && $_REQUEST['post_type'] === 'acf-field-group' ) {
				return true;
			}
			if ( isset( $_REQUEST['post'] ) && get_post_type( $_REQUEST['post'] ) === 'acf-field-group' ) {
				return true;
			}
			if ( isset( $_REQUEST['action'] ) && $_REQUEST['action'] === 'rename_field_groups' ) {
				return true;
			}

			return false;
		}

		/**
		 * Defines a constant if doesnt already exist.
		 *
		 * @param $name
		 * @param bool $value
		 */
		function define( $name, $value = true ) {
			if ( ! defined( $name ) ) {
				define( $name, $value );
			}
		}

	}

	/*
	 * acf_polylang
	 *
	 * The main function responsible for returning the one true ACF_Polylang Instance to functions everywhere.
	 * Use this function like you would a global variable, except without needing to declare the global.
	 *
	 * @return	ACF_Polylang
	 */
	function acf_polylang() {
		global $acf_polylang;

		// Instantiate only once.
		if ( ! isset( $acf_polylang ) ) {
			$acf_polylang = new ACF_Polylang();
			$acf_polylang->initialize();
		}

		return $acf_polylang;
	}

	// Instantiate.
	acf_polylang();

endif;
