<?php
/*
Plugin Name: ACF Polylang
Description: Polylang integration for ACF.
Version:     1.3.0
Author:      Le Web simple <pascal@lewebsimple.ca>
Author URI:  https://lewebsimple.ca
Text Domain: acf-polylang
Domain Path: /languages
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Initialize ACF Polylang
add_action( 'plugins_loaded', 'acf_polylang_initialize' );
function acf_polylang_initialize() {
	// Check if ACF is active
	if ( ! function_exists( 'acf' ) && ! class_exists( 'ACF' ) ) {
		return;
	}

	// Check if Polylang is active
	if ( ! function_exists( 'pll_get_post_types' ) && ! function_exists( 'PLL' ) ) {
		return;
	}

	add_filter( 'acf/settings/l10n_textdomain', 'acf_polylang_l10n_textdomain' );
	add_filter( 'pll_get_post_types', 'acf_polylang_pll_get_post_types', 100, 2 );

	add_action( 'publish_acf-field-group', 'acf_polylang_sanitize_field_group_key', 10, 2 );
	add_filter( 'pll_the_language_link', 'acf_polylang_pll_the_language_link', 10, 2 );

	if ( acf_polylang_is_acf_admin() ) {
		add_action( 'pre_get_posts', 'acf_polylang_field_groups_orderby_title' );
	}

	add_action( 'save_post', 'acf_polylang_export_translations' );

	add_filter( 'acf/translate_field/type=date_picker', 'acf_polylang_translate_datetime_field' );
	add_filter( 'acf/translate_field/type=date_time_picker', 'acf_polylang_translate_datetime_field' );
	add_filter( 'acf/translate_field/type=time_picker', 'acf_polylang_translate_datetime_field' );
	add_filter( 'acf/translate_field/type=number', 'acf_polylang_translate_number_field' );
	add_filter( 'acf/translate_field/type=page', 'acf_polylang_translate_page_field' );
	add_filter( 'acf/translate_field', 'acf_polylang_translate_field_placeholder' );

	add_action( 'acf/save_post', 'acf_polylang_sync_translations', 5 );
}

// Helper: Check if currently in ACF admin
function acf_polylang_is_acf_admin() {
	if ( isset( $_REQUEST['post_type'] ) && $_REQUEST['post_type'] === 'acf-field-group' ) { return true;
	}
	if ( isset( $_REQUEST['post'] ) && get_post_type( $_REQUEST['post'] ) === 'acf-field-group' ) { return true;
	}
	if ( isset( $_REQUEST['action'] ) && $_REQUEST['action'] === 'rename_field_groups' ) { return true;
	}
	return false;
}

// Set textdomain for ACF fields
function acf_polylang_l10n_textdomain( $textdomain ) {
	if ( acf_polylang_is_acf_admin() ) { return $textdomain;
	}
	return apply_filters( 'acf_polylang_textdomain', wp_get_theme()->get( 'TextDomain' ) ?: $textdomain );
}

// Prevent ACF field groups from being translated
function acf_polylang_pll_get_post_types( $post_types, $is_settings ) {
	unset( $post_types['acf-field-group'] );
	return $post_types;
}

// Sanitize ACF field group key
function acf_polylang_sanitize_field_group_key( $post_id, $post ) {
	if ( strpos( $post->post_name, 'group_' ) === 0 ) { return;
	}
	$key                                = 'group_' . str_replace( '-', '_', sanitize_title( $post->post_title ) );
	$_POST['acf_field_group']['key']    = $key;
	$_REQUEST['acf_field_group']['key'] = $key;
}

// Maintain query string in Polylang language links
function acf_polylang_pll_the_language_link( $url, $slug ) {
	return add_query_arg( $_GET, $url );
}

// Order ACF field groups by title in admin
function acf_polylang_field_groups_orderby_title( $query ) {
	if ( $query->is_main_query() && $query->get( 'post_type' ) === 'acf-field-group' ) {
		$query->set( 'orderby', 'title' );
	}
}

// Export field translations when saving ACF field groups
function acf_polylang_export_translations( $post_id ) {
	if ( get_post_type( $post_id ) !== 'acf-field-group' || get_post_status( $post_id ) !== 'publish' || isset( $_GET['acfsync'] ) ) {
		return;
	}
	$export_path = (string) acf_get_setting( 'save_json' );
	if ( ! file_exists( $export_path ) ) {
		return;
	}
	$textdomain = apply_filters( 'acf_polylang_textdomain', wp_get_theme()->get( 'TextDomain' ) ?: 'acf-polylang' );
	$output     = '';
	foreach ( acf_polylang_get_all_translations() as $translation ) {
		$output .= '__( "' . addcslashes( $translation, '"' ) . '", \'' . $textdomain . '\' );' . PHP_EOL;
	}
	file_put_contents( $export_path . '/acf-translations.php', '<?php' . PHP_EOL . $output );
}

// Helper: Get translations for all ACF field groups and their fields
function acf_polylang_get_all_translations() {
	$translations = array();
	foreach ( acf_get_field_groups() as $field_group ) {
		$translations = array_merge( $translations, acf_polylang_get_field_group_translations( $field_group ) );
	}
	$translations = array_unique( $translations );
	sort( $translations );
	return $translations;
}

// Helper: Get translations for a specific ACF field group and its fields
function acf_polylang_get_field_group_translations( $field_group ) {
	if ( empty( $field_group['ID'] ) || get_post_type( $field_group['ID'] ) !== 'acf-field-group' ) { return array();
	}
	$translations = array( get_the_title( $field_group['ID'] ) );
	foreach ( acf_get_fields( $field_group['key'] ) as $field ) {
		$translations = array_merge( $translations, acf_polylang_get_field_translations( $field ) );
	}
	return $translations;
}

// Helper: Get translations for a specific ACF field and its subfields
function acf_polylang_get_field_translations( $field ) {
	$translations = array();
	if ( empty( $field['type'] ) ) { return $translations;
	}
	$properties = array(
		'label',
		'instructions',
		'placeholder',
		'message',
		'button_label',
		'next_text',
		'previous_text',
		'ui_on_text',
		'ui_off_text',
	);
	foreach ( $properties as $prop ) {
		if ( ! empty( $field[ $prop ] ) ) {
			$translations[] = $field[ $prop ];
		}
	}

	if ( $field['type'] === 'number' ) {
		if ( ! empty( $field['prepend'] ) ) { $translations[] = $field['prepend'];
		}
		if ( ! empty( $field['append'] ) ) { $translations[] = $field['append'];
		}
	}

	if ( in_array( $field['type'], array( 'date_picker', 'date_time_picker', 'time_picker' ) ) ) {
		$translations[] = $field['return_format'];
	}

	if ( ! empty( $field['choices'] ) ) {
		foreach ( $field['choices'] as $label ) {
			$translations[] = $label;
		}
	}

	if ( ! empty( $field['sub_fields'] ) ) {
		foreach ( $field['sub_fields'] as $sub_field ) {
			$translations = array_merge( $translations, acf_polylang_get_field_translations( $sub_field ) );
		}
	}

	if ( ! empty( $field['layouts'] ) ) {
		foreach ( $field['layouts'] as $layout ) {
			$translations[] = $layout['label'];
			if ( ! empty( $layout['sub_fields'] ) ) {
				foreach ( $layout['sub_fields'] as $sub_field ) {
					$translations = array_merge( $translations, acf_polylang_get_field_translations( $sub_field ) );
				}
			}
		}
	}

	return $translations;
}

// Translate ACF datetime field
function acf_polylang_translate_datetime_field( $field ) {
	$textdomain             = apply_filters( 'acf/settings/l10n_textdomain', 'acf-polylang' );
	$field['return_format'] = __( $field['return_format'], $textdomain );
	return $field;
}

// Translate ACF number field
function acf_polylang_translate_number_field( $field ) {
	$textdomain = apply_filters( 'acf/settings/l10n_textdomain', 'kaliroots' );
	if ( ! empty( $field['prepend'] ) ) {
		$field['prepend'] = __( $field['prepend'], $textdomain );
	}
	if ( ! empty( $field['append'] ) ) {
		$field['append'] = __( $field['append'], $textdomain );
	}
	return $field;
}

// Translate ACF page field
function acf_polylang_translate_page_field( $field ) {
	$textdomain = apply_filters( 'acf/settings/l10n_textdomain', 'kaliroots' );
	if ( ! empty( $field['next_text'] ) ) {
		$field['next_text'] = __( $field['next_text'], $textdomain );
	}
	if ( ! empty( $field['previous_text'] ) ) {
		$field['previous_text'] = __( $field['previous_text'], $textdomain );
	}
	return $field;
}

// Translate ACF field placeholder
function acf_polylang_translate_field_placeholder( $field ) {
	$textdomain = apply_filters( 'acf/settings/l10n_textdomain', 'kaliroots' );
	if ( ! empty( $field['placeholder'] ) ) {
		$field['placeholder'] = __( $field['placeholder'], $textdomain );
	}
	return $field;
}

// Synchronize ACF translations
function acf_polylang_sync_translations( $post_id ) {
	if ( empty( $_POST['acf'] ) || ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) ) {
		return;
	}
	$translations = pll_get_post_translations( $post_id );
	if ( empty( $translations ) || count( $translations ) < 2 ) {
		return;
	}
	foreach ( $translations as $lang => $tr_id ) {
		if ( $tr_id === $post_id ) {
			continue;
		}
		foreach ( $_POST['acf'] as $field_key => $field_value ) {
			if ( ! is_array( $field_value ) ) {
				continue;
			}
			$field = get_field_object( $field_key, $post_id, false );
			if ( ! $field || ! in_array( $field['type'], array( 'repeater', 'flexible_content' ) ) ) {
				continue;
			}
			$translated_value = get_field( $field['name'], $tr_id, false );
			if ( ! is_array( $translated_value ) ) {
				continue;
			}
			$sync_value = acf_polylang_sync_nested_field( $field, $field_value, $translated_value );
			update_field( $field_key, $sync_value, $tr_id );
		}
	}
}

// Helper: Synchronize translated field values recursively
function acf_polylang_sync_nested_field( $field, $submitted_rows, $translated_rows ) {
	$new_rows = array();
	foreach ( $submitted_rows as $row_key => $row_data ) {
		// Determine if this is a new row or an existing one
		$index = null;
		if ( preg_match( '/^row-(\d+)$/', $row_key, $matches ) ) {
			$index = (int) $matches[1];
		}

		// Check if the row already exists in the translated rows
		$has_existing_translation = $index !== null && isset( $translated_rows[ $index ] );
		$source_row               = $has_existing_translation ? $translated_rows[ $index ] : $row_data;
		$new_row                  = $source_row;

		switch ( $field['type'] ) {
			case 'repeater':
				if ( ! is_array( $field['sub_fields'] ?? false ) ) {
					break;
				}
				foreach ( $field['sub_fields'] as $sub_field ) {
					$sub_key = $sub_field['key'];
					if ( is_array( $row_data[ $sub_key ] ?? false ) && in_array( $sub_field['type'], array( 'repeater', 'flexible_content' ) ) ) {
						$nested_submitted    = $row_data[ $sub_key ];
						$nested_translated   = $has_existing_translation
							? ( $translated_rows[ $index ][ $sub_key ] ?? array() )
							: array(); // Empty for new rows
						$new_row[ $sub_key ] = acf_polylang_sync_nested_field( $sub_field, $nested_submitted, $nested_translated );
					} elseif ( ! $has_existing_translation && isset( $row_data[ $sub_key ] ) ) {
						// Copy non-nested fields on new row
						$new_row[ $sub_key ] = $row_data[ $sub_key ];
					}
				}
				break;

			case 'flexible_content':
				if (
					isset( $field['layouts'] ) &&
					is_array( $field['layouts'] ) &&
					isset( $row_data['acf_fc_layout'] )
				) {
					$layout_def = reset(
						array_filter(
							$field['layouts'],
							fn( $l ) => isset( $l['name'] ) && $l['name'] === $row_data['acf_fc_layout']
						)
					);

					if ( $layout_def && isset( $layout_def['sub_fields'] ) ) {
						foreach ( $layout_def['sub_fields'] as $sub_field ) {
							$sub_key = $sub_field['key'];
							if (
								isset( $row_data[ $sub_key ] ) &&
								is_array( $row_data[ $sub_key ] ) &&
								in_array( $sub_field['type'], array( 'repeater', 'flexible_content' ) )
							) {
								$nested_submitted    = $row_data[ $sub_key ];
								$nested_translated   = $has_existing_translation
									? ( $translated_rows[ $index ][ $sub_key ] ?? array() )
									: array();
								$new_row[ $sub_key ] = acf_polylang_sync_nested_field(
									$sub_field,
									$nested_submitted,
									$nested_translated
								);
							} elseif ( ! $has_existing_translation && isset( $row_data[ $sub_key ] ) ) {
								// Copy simple fields for new row
								$new_row[ $sub_key ] = $row_data[ $sub_key ];
							}
						}
					}
				}
				break;
		}
		$new_rows[] = $new_row;
	}
	return $new_rows;
}
