<?php

//------------------------------------------

// don't load directly
if ( ! defined( 'ABSPATH' ) ) {
	die();
}

GFForms::include_addon_framework();

class GFSurvey extends GFAddOn {

	protected $_version = GF_SURVEY_VERSION;
	protected $_min_gravityforms_version = '2.0';
	protected $_slug = 'gravityformssurvey';
	protected $_path = 'gravityformssurvey/survey.php';
	protected $_full_path = __FILE__;
	protected $_url = 'http://www.gravityforms.com';
	protected $_title = 'Gravity Forms Survey Add-On';
	protected $_short_title = 'Survey';
	protected $_enable_rg_autoupgrade = true;
	protected $_enable_theme_layer = true;

	/**
	 * Whether this add-on has access to the Gravity Forms settings renderer.
	 *
	 * @since 3.5
	 *
	 * @var bool
	 */
	protected $_has_settings_renderer;

	/**
	 * Members plugin integration
	 */
	protected $_capabilities = array( 'gravityforms_survey', 'gravityforms_survey_uninstall', 'gravityforms_survey_results' );

	/**
	 * Permissions
	 */
	protected $_capabilities_settings_page = 'gravityforms_survey';
	protected $_capabilities_form_settings = 'gravityforms_survey';
	protected $_capabilities_uninstall = 'gravityforms_survey_uninstall';

	private static $_instance = null;

	/**
	 * Get an instance of this class.
	 *
	 * @return GFSurvey
	 */
	public static function get_instance() {
		if ( self::$_instance == null ) {
			self::$_instance = new GFSurvey();
		}

		return self::$_instance;
	}

	private function __clone() {
	} /* do nothing */

	/**
	 * Handles anything which requires early initialization.
	 */
	public function pre_init() {
		parent::pre_init();

		if ( $this->is_gravityforms_supported() && class_exists( 'GF_Field' ) ) {
			require_once( 'includes/class-gf-field-survey.php' );
			require_once( 'includes/class-gf-field-likert.php' );
			require_once( 'includes/class-gf-field-rank.php' );
			require_once( 'includes/class-gf-field-rating.php' );

			add_filter( 'gform_export_field_value', array( $this, 'export_field_value' ), 10, 4 );
		}
	}

	/**
	 * Handles hooks and loading of language files.
	 */
	public function init() {
		$this->_has_settings_renderer = $this->is_gravityforms_supported( '2.5-beta' );

		// Integration with the feed add-ons as of GF 1.9.15.12; for add-ons which don't override get_field_value().
		add_filter( 'gform_addon_field_value', array( $this, 'addon_field_value' ), 10, 5 );

		// AWeber 2.3 and newer use the gform_addon_field_value hook, only use the gform_aweber_field_value hook with older versions.
		if ( defined( 'GF_AWEBER_VERSION' ) && version_compare( 'GF_AWEBER_VERSION', '2.3', '<' ) ) {
			add_filter( 'gform_aweber_field_value', array( $this, 'legacy_addon_field_value' ), 10, 4 );
		}

		// Campaign Monitor Add-On integration
		add_filter( 'gform_campaignmonitor_field_value', array( $this, 'legacy_addon_field_value' ), 10, 4 );

		// Mailchimp Add-On integration
		add_filter( 'gform_mailchimp_field_value', array( $this, 'legacy_addon_field_value' ), 10, 4 );

		// Zapier Add-On integration
		add_filter( 'gform_zapier_field_value', array( $this, 'zapier_field_value' ), 10, 4 );

		// merge tags
		add_filter( 'gform_replace_merge_tags', array( $this, 'replace_merge_tags' ), 10, 7 );

		// add a special class to likert fields so we can identify them later
		add_action( 'gform_field_css_class', array( $this, 'add_custom_class' ), 10, 3 );

		// display survey results on entry detail
		add_filter( 'gform_entry_field_value', array( $this, 'entry_field_value' ), 10, 4 );

		// conditional logic filters
		add_filter( 'gform_entry_meta_conditional_logic_confirmations', array( $this, 'conditional_logic_filters' ), 10, 3 );
		add_filter( 'gform_entry_meta_conditional_logic_notifications', array( $this, 'conditional_logic_filters' ), 10, 3 );
		add_filter( 'gform_is_value_match', array( $this, 'is_value_match_rank' ), 10, 6 );

		parent::init();

	}

	/**
	 * Initialize the admin specific hooks.
	 */
	public function init_admin() {

		// form editor
		add_action( 'gform_field_standard_settings', array( $this, 'survey_field_settings' ), 10, 2 );
		add_filter( 'gform_tooltips', array( $this, 'add_survey_tooltips' ) );

		// display results on entry list
		add_filter( 'gform_entries_field_value', array( $this, 'export_field_value' ), 10, 4 );

		// declare arrays on form import
		add_filter( 'gform_import_form_xml_options', array( $this, 'import_file_options' ) );

		// contacts
		add_filter( 'gform_contacts_tabs_contact_detail', array( $this, 'add_tab_to_contact_detail' ), 10, 2 );
		add_action( 'gform_contacts_tab_survey', array( $this, 'contacts_tab' ) );

		parent::init_admin();

	}

	/**
	 * The Survey add-on does not support logging.
	 *
	 * @param array $plugins The plugins which support logging.
	 *
	 * @return array
	 */
	public function set_logging_supported( $plugins ) {

		return $plugins;

	}


	// # SCRIPTS & STYLES -----------------------------------------------------------------------------------------------

	/**
	 * Return the scripts which should be enqueued.
	 *
	 * @return array
	 */
	public function scripts() {

		$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG || isset( $_GET['gform_debug'] ) ? '' : '.min';

		$scripts = array(
			array(
				'handle'   => 'gsurvey_form_editor_js',
				'src'      => $this->get_base_url() . "/js/gsurvey_form_editor{$min}.js",
				'version'  => $this->_version,
				'deps'     => array( 'jquery' ),
				'callback' => array( $this, 'localize_scripts' ),
				'enqueue'  => array(
					array( 'admin_page' => array( 'form_editor', 'form_settings'  ) ),
				),
			),
			array(
				'handle'  => 'gsurvey_js',
				'src'     => $this->get_base_url() . "/js/gsurvey{$min}.js",
				'version' => $this->_version,
				'deps'    => array( 'jquery', 'jquery-ui-sortable', 'jquery-touch-punch' ),
				'enqueue' => array(
					array( $this, 'should_enqueue_gravity_theme' ),
					array(
						'field_types' => array( 'survey' )
					),
				),
			),
		);

		$merge_tags = $this->get_merge_tags();

		if ( ! empty( $merge_tags ) ) {
			$scripts[] = array(
				'handle'  => 'gform_survey_merge_tags',
				'src'     => $this->get_base_url() . "/js/gsurvey_merge_tags{$min}.js",
				'version' => $this->_version,
				'deps'    => array( 'jquery' ),
				'enqueue' => array(
					array( 'admin_page' => array( 'form_settings' ) ),
				),
				'strings' => array(
					'merge_tags' => $merge_tags,
				),
			);
		}

		return array_merge( parent::scripts(), $scripts );
	}

	/**
	 * Return the stylesheets which should be enqueued.
	 *
	 * @return array
	 */
	public function styles() {

		$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG || isset( $_GET['gform_debug'] ) ? '' : '.min';

		$styles = array(
			array(
				'handle'  => 'gsurvey_admin',
				'src'     => $this->get_base_url() . "/assets/css/dist/admin{$min}.css",
				'version' => $this->_version,
				'enqueue' => array(
					array( 'admin_page' => array( 'form_editor' ) ),
					array(
						'admin_page' => array( 'results', 'entry_view', 'entry_detail', 'entry_edit' ),
						'field_types' => array( 'survey' ),
					),
				),
			),
			array(
				'handle'  => 'gsurvey_css',
				'src'     => $this->get_base_url() . "/assets/css/dist/theme{$min}.css",
				'version' => $this->_version,
				'media'   => 'screen',
				'enqueue' => array(
					array( $this, 'should_enqueue_gravity_theme' ),
					array(
						'field_types' => array( 'survey' )
					),
				),
			),
			array(
				'handle'  => 'gsurvey-theme-foundation',
				'src'     => $this->get_base_url() . "/assets/css/dist/theme-foundation{$min}.css",
				'version' => $this->_version,
				'media'   => 'screen',
				'enqueue' => array(
					array( $this, 'should_enqueue_theme_framework' ),
				),
			),
			array(
				'handle'  => 'gsurvey-theme-framework',
				'src'     => $this->get_base_url() . "/assets/css/dist/theme-framework{$min}.css",
				'version' => $this->_version,
				'media'   => 'screen',
				'enqueue' => array(
					array( $this, 'should_enqueue_theme_framework' ),
				),
			),
		);

		return array_merge( parent::styles(), $styles );
	}


	/**
	 * Returns true if the Gravity Theme css file should be enqueued on this page. Returns false otherwise.
	 *
	 * @since 3.8
	 *
	 * @param array $form Current form object.
	 * @param bool  $is_ajax Whether form is being embedded with ajax enabled.
	 *
	 * @return bool Returns true if the Gravity Theme css file should be enqueued on this page. Returns false otherwise.
	 */
	public function should_enqueue_gravity_theme( $form, $is_ajax ) {
		// Always enqueue in the form editor.
		if ( GFCommon::is_form_editor() ) {
			return true;
		}

		// In 2.7 and above, enqueue in the block editor
		if ( $this->is_gravityforms_supported( '2.7-RC-1' ) ) {
			return GFCommon::is_block_editor_page();
		}

		return false;
	}

	/**
	 * Returns true if the theme framework css files should be enqueued on this page. Returns false otherwise.
	 *
	 * @since 3.8
	 *
	 * @param array $form Current form object.
	 * @param bool  $is_ajax Whether or not form is being embedded with ajax enabled.
	 *
	 * @return bool Returns true if the theme framework css files should be enqueued on this page. Returns false otherwise.
	 */
	public function should_enqueue_theme_framework( $form, $is_ajax ) {
		if ( ! $this->is_gravityforms_supported( '2.7-RC-1' ) ) {
			return false;
		}

		// Always enqueue on block editor page.
		if ( GFCommon::is_block_editor_page() ) {
			return true;
		}

		// Enqueues theme framework when there is a survey field on a non-admin or preview page.
		$has_survey_field = (bool) GFCommon::get_fields_by_type( $form, 'survey' );
		$is_gf_admin_page = GFForms::get_page();

		return $has_survey_field && ! $is_gf_admin_page && ! $this->is_preview();
	}

	/**
	 * Localize the strings used by the scripts.
	 */
	public function localize_scripts() {

		// Get current page protocol
		$protocol = isset( $_SERVER['HTTPS'] ) ? 'https://' : 'http://';
		// Output admin-ajax.php URL with same protocol as current page
		$params = array(
			'ajaxurl'   => admin_url( 'admin-ajax.php', $protocol ),
			'imagesUrl' => $this->get_base_url() . '/images',
			'strings'   => array(
				'untitledSurveyField' => wp_strip_all_tags( __( 'Untitled Survey Field', 'gravityformssurvey' ) ),
			),
		);
		wp_localize_script( 'gsurvey_form_editor_js', 'gsurveyVars', $params );

		//localize strings for the js file
		$strings = array(
			'isLegacy'         => ! $this->_has_settings_renderer ? 'true' : 'false',
			'firstChoice'      => wp_strip_all_tags( __( 'First row', 'gravityformssurvey' ) ),
			'secondChoice'     => wp_strip_all_tags( __( 'Second row', 'gravityformssurvey' ) ),
			'thirdChoice'      => wp_strip_all_tags( __( 'Third row', 'gravityformssurvey' ) ),
			'fourthChoice'     => wp_strip_all_tags( __( 'Fourth row', 'gravityformssurvey' ) ),
			'fifthChoice'      => wp_strip_all_tags( __( 'Fifth row', 'gravityformssurvey' ) ),
			'dragToReOrder'    => wp_strip_all_tags( __( 'Drag to re-order', 'gravityformssurvey' ) ),
			'addAnotherRow'    => wp_strip_all_tags( __( 'Add another row', 'gravityformssurvey' ) ),
			'removeThisRow'    => wp_strip_all_tags( __( 'Remove this row', 'gravityformssurvey' ) ),
			'addAnotherColumn' => wp_strip_all_tags( __( 'Add another column', 'gravityformssurvey' ) ),
			'removeThisColumn' => wp_strip_all_tags( __( 'Remove this column', 'gravityformssurvey' ) ),
			'columnLabel1'     => wp_strip_all_tags( __( 'Strongly disagree', 'gravityformssurvey' ) ),
			'columnLabel2'     => wp_strip_all_tags( __( 'Disagree', 'gravityformssurvey' ) ),
			'columnLabel3'     => wp_strip_all_tags( __( 'Neutral', 'gravityformssurvey' ) ),
			'columnLabel4'     => wp_strip_all_tags( __( 'Agree', 'gravityformssurvey' ) ),
			'columnLabel5'     => wp_strip_all_tags( __( 'Strongly agree', 'gravityformssurvey' ) ),
		);
		wp_localize_script( 'gsurvey_form_editor_js', 'gsurveyLikertStrings', $strings );

		//localize strings for the rank field
		$rank_strings = array(
			'firstChoice'  => wp_strip_all_tags( __( 'First Choice', 'gravityformssurvey' ) ),
			'secondChoice' => wp_strip_all_tags( __( 'Second Choice', 'gravityformssurvey' ) ),
			'thirdChoice'  => wp_strip_all_tags( __( 'Third Choice', 'gravityformssurvey' ) ),
			'fourthChoice' => wp_strip_all_tags( __( 'Fourth Choice', 'gravityformssurvey' ) ),
			'fifthChoice'  => wp_strip_all_tags( __( 'Fifth Choice', 'gravityformssurvey' ) ),
		);
		wp_localize_script( 'gsurvey_form_editor_js', 'gsurveyRankStrings', $rank_strings );

		//localize strings for the rank field conditional logic settings
		$rank_condition_strings = array(
			'position'  => wp_strip_all_tags( __( 'Position', 'gravityformssurvey' ) ),
			'is' => wp_strip_all_tags( __( 'is in position', 'gravityformssurvey' ) ),
			'isNot'  => wp_strip_all_tags( __( 'is not in position', 'gravityformssurvey' ) ),
			'greaterThan' => wp_strip_all_tags( __( 'is in a position greater than', 'gravityformssurvey' ) ),
			'lessThan'  => wp_strip_all_tags( __( 'is in a position less than', 'gravityformssurvey' ) ),
		);
		wp_localize_script( 'gsurvey_form_editor_js', 'gsurveyRankConditionStrings', $rank_condition_strings );

		// localize strings for the likert field conditional logic settings
		$likert_condition_strings = array(
			'is' => wp_strip_all_tags( __( 'is', 'gravityformssurvey' ) ),
			'isNot'  => wp_strip_all_tags( __( 'is not', 'gravityformssurvey' ) ),
		);
		wp_localize_script( 'gsurvey_form_editor_js', 'gsurveyLikertConditionStrings', $likert_condition_strings );

		//localize strings for the ratings field
		$rating_strings = array(
			'firstChoice'  => wp_strip_all_tags( __( 'Terrible', 'gravityformssurvey' ) ),
			'secondChoice' => wp_strip_all_tags( __( 'Not so great', 'gravityformssurvey' ) ),
			'thirdChoice'  => wp_strip_all_tags( __( 'Neutral', 'gravityformssurvey' ) ),
			'fourthChoice' => wp_strip_all_tags( __( 'Pretty good', 'gravityformssurvey' ) ),
			'fifthChoice'  => wp_strip_all_tags( __( 'Excellent', 'gravityformssurvey' ) ),
		);
		wp_localize_script( 'gsurvey_form_editor_js', 'gsurveyRatingStrings', $rating_strings );

	}

	public function localize_results_scripts() {

		$filter_fields    = rgget( 'f' );
		$filter_types     = rgget( 't' );
		$filter_operators = rgget( 'o' );
		$filter_values    = rgget( 'v' );

		// Get current page protocol
		$protocol = isset( $_SERVER['HTTPS'] ) ? 'https://' : 'http://';
		// Output admin-ajax.php URL with same protocol as current page

		$vars = array(
			'ajaxurl'         => admin_url( 'admin-ajax.php', $protocol ),
			'imagesUrl'       => $this->get_base_url() . '/images',
			'filterFields'    => $filter_fields,
			'filterTypes'     => $filter_types,
			'filterOperators' => $filter_operators,
			'filterValues'    => $filter_values,
		);

		wp_localize_script( 'gsurvey_results_js', 'gresultsVars', $vars );

		$strings = array(
			'noFilters'         => wp_strip_all_tags( __( 'No filters', 'gravityformspolls' ) ),
			'addFieldFilter'    => wp_strip_all_tags( __( 'Add a field filter', 'gravityformspolls' ) ),
			'removeFieldFilter' => wp_strip_all_tags( __( 'Remove a field filter', 'gravityformspolls' ) ),
			'ajaxError'         => wp_strip_all_tags( __( 'Error retrieving results. Please contact support.', 'gravityformspolls' ) ),
		);

		wp_localize_script( 'gsurvey_results_js', 'gresultsStrings', $strings );

	}

	/**
	 * Outputs custom css properties to the page so that the survey fields can have custom properties based on block settings selections.
	 *
	 * @since 3.8
	 *
	 * @param int   $form_id        Current form id.
	 * @param array $settings       Current settings.
	 * @param array $block_settings Current block settings.
	 *
	 * @return array Returns an array containing the custom css properties to be output to the page.
	 */
	public function theme_layer_form_css_properties( $form_id, $settings, $block_settings ) {

		$label_color = rgar( $block_settings, 'labelColor' );
		if ( ! $label_color ) {
			return array();
		}

		$color_util          = new \Gravity_Forms\Gravity_Forms\Util\Colors\Color_Modifier();
		$rgb                 = $color_util->convert_hex_to_rgb( $label_color );
		$encoded_color       = urlencode( $label_color );
		return array(
			'gform-survey-theme-field-likert-row-odd-background-color' => "rgb({$rgb['r']},{$rgb['g']},{$rgb['b']},0.08)",
			'gform-survey-theme-icon-control-rank' => "url(\"data:image/svg+xml,%3Csvg width='8' height='14' viewBox='0 0 8 14' fill='none' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath fill-rule='evenodd' clip-rule='evenodd' d='M4 0C4.26522 5.96046e-08 4.51957 0.105357 4.70711 0.292893L7.70711 3.29289C8.09763 3.68342 8.09763 4.31658 7.70711 4.70711C7.31658 5.09763 6.68342 5.09763 6.29289 4.70711L4 2.41421L1.70711 4.70711C1.31658 5.09763 0.683417 5.09763 0.292893 4.70711C-0.0976311 4.31658 -0.097631 3.68342 0.292893 3.29289L3.29289 0.292893C3.48043 0.105357 3.73478 0 4 0ZM0.292893 9.29289C0.683417 8.90237 1.31658 8.90237 1.70711 9.29289L4 11.5858L6.29289 9.29289C6.68342 8.90237 7.31658 8.90237 7.70711 9.29289C8.09763 9.68342 8.09763 10.3166 7.70711 10.7071L4.70711 13.7071C4.31658 14.0976 3.68342 14.0976 3.29289 13.7071L0.292893 10.7071C-0.0976311 10.3166 -0.0976311 9.68342 0.292893 9.29289Z' fill='{$encoded_color}'/%3E%3C/svg%3E\")",
		);
	}

	// # RESULTS & SCORING ----------------------------------------------------------------------------------------------

	/**
	 * Configure the survey results page.
	 *
	 * @return array
	 */
	public function get_results_page_config() {
		return array(
			'title'        => esc_html__( 'Survey Results', 'gravityformssurvey' ),
			'capabilities' => array( 'gravityforms_survey_results' ),
			'callbacks'    => array(
				'fields'  => array( $this, 'results_fields' ),
				'filters' => array( $this, 'results_filters' ),
			),
		);
	}

	/**
	 * Remove the score from the results page filters if scoring is not enabled on a Likert field.
	 *
	 * @param array $filters
	 * @param array $form
	 *
	 * @return array
	 */
	public function results_filters( $filters, $form ) {
		$fields = $form['fields'];
		foreach ( $fields as $field ) {
			if ( $field->get_input_type() == 'likert' && $field->gsurveyLikertEnableScoring ) {
				return $filters;
			}
		}

		foreach ( $filters as $key => $filter ) {
			if ( $filter['key'] == 'gsurvey_score' ) {
				unset( $filters[ $key ] );
			}
		}

		return $filters;
	}

	/**
	 * Get all the survey fields for the current form.
	 *
	 * @param array $form The current form object.
	 *
	 * @return GF_Field[]
	 */
	public function results_fields( $form ) {
		return GFAPI::get_fields_by_type( $form, array( 'survey' ) );
	}

	/**
	 * Helper to check if scoring is enabled on at least one of the forms Likert fields.
	 *
	 * @param array $form The current form object.
	 *
	 * @return bool
	 */
	private function scoring_enabled( $form ) {
		$survey_fields = $this->results_fields( $form );

		foreach ( $survey_fields as $field ) {
			if ( $field->get_input_type() == 'likert' && $field->gsurveyLikertEnableScoring ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Returns the total score of all likert fields with scoring enabled.
	 *
	 * @param array $form The current form object.
	 * @param array $entry The current entry object.
	 *
	 * @return float|int
	 */
	public function get_survey_score( $form, $entry ) {
		$survey_fields = $this->results_fields( $form );
		$score         = 0;
		foreach ( $survey_fields as $field ) {
			if ( $field->get_input_type() == 'likert' && $field->gsurveyLikertEnableScoring ) {
				$score += $this->get_field_score( $field, $entry );
			}
		}

		return $score;
	}

	/**
	 * Returns the total score of all likert fields with scoring enabled.
	 *
	 * @param array $form The current form object.
	 * @param array $entry The current entry object.
	 *
	 * @return float|int
	 */
	public function get_total_score( $form, $entry ) {

		return $this->get_survey_score( $form, $entry );
	}

	/**
	 * Returns the score for the specified field.
	 *
	 * Called statically by GFResults.
	 *
	 * @param GF_Field_Likert $field The current field.
	 * @param array $entry The current entry object.
	 *
	 * @return float|int
	 */
	public static function get_field_score( $field, $entry ) {
		$score = 0;
		if ( $field->gsurveyLikertEnableMultipleRows ) {
			// cycle through the entry values in case the the number of choices has changed since the entry was submitted
			foreach ( $entry as $key => $value ) {
				if ( intval( $key ) != $field->id ) {
					continue;
				}

				if ( false === strpos( $value, ':' ) ) {
					continue;
				}

				list( $row_val, $col_val ) = explode( ':', $value, 2 );

				foreach ( $field->gsurveyLikertRows as $row ) {
					if ( $row['value'] == $row_val ) {
						foreach ( $field->choices as $choice ) {
							if ( $choice['value'] == $col_val ) {
								$score += floatval( rgar( $choice, 'score' ) );
							}
						}
					}
				}
			}
		} else {
			$value = rgar( $entry, $field->id );
			if ( ! empty( $value ) ) {
				foreach ( $field->choices as $choice ) {
					if ( $choice['value'] == $value ) {
						$score += floatval( rgar( $choice, 'score' ) );
					}
				}
			}
		}

		return $score;
	}

	/**
	 * Returns the score for the specified row.
	 *
	 * @param string $target_row_val The unique id (value) for the likert row.
	 * @param GF_Field_Likert $field The current field.
	 * @param array $entry The current entry.
	 *
	 * @return float|int
	 */
	public static function get_likert_row_score( $target_row_val, $field, $entry ) {
		$score = 0;

		if ( $field->gsurveyLikertEnableMultipleRows ) {

			foreach ( $entry as $key => $value ) {
				if ( intval( $key ) != $field->id ) {
					continue;
				}

				if ( false === strpos( $value, ':' ) ) {
					continue;
				}

				list( $row_val, $col_val ) = explode( ':', $value, 2 );

				foreach ( $field->gsurveyLikertRows as $row ) {
					if ( $row['value'] == $row_val && $target_row_val == $row_val ) {
						foreach ( $field->choices as $choice ) {
							if ( $choice['value'] == $col_val ) {
								$score = floatval( rgar( $choice, 'score' ) );

								return $score;
							}
						}
					}
				}
			}
		} else {
			$score = self::get_field_score( $field, $entry );
		}

		return $score;
	}

	// # MERGE TAGS -----------------------------------------------------------------------------------------------------

	/**
	 * Add the score merge tags to the merge tag drop downs in the admin.
	 *
	 * @deprecated 3.7 Use GFSurvey::get_merge_tags().
	 *
	 * @param array $form The current form.
	 *
	 * @return array
	 */
	public function add_merge_tags( $form ) {
		_deprecated_function( __METHOD__, '3.7', 'GFSurvey::get_merge_tags()' );
		return $form;
	}

	/**
	 * Gets the merge tags to add to the merge tag drop downs in the admin.
	 *
	 * @return array
	 */
	private function get_merge_tags() {
		$merge_tags = array();
		$form       = $this->get_current_form();

		if ( ! $form ) {
			return $merge_tags;
		}

		$survey_fields = GFAPI::get_fields_by_type( $form, array( 'survey' ) );

		if ( empty( $survey_fields ) ) {
			return $merge_tags;
		}

		$scoring_enabled = false;

		foreach ( $form['fields'] as $field ) {
			if ( $field->get_input_type() == 'likert' && $field->gsurveyLikertEnableScoring ) {
				$scoring_enabled = true;
				$field_id        = $field->id;
				$field_label     = $field->label;
				$group           = $field->isRequired ? 'required' : 'optional';
				$merge_tags[]    = array(
					'group' => $group,
					'label' => esc_html__( 'Survey Field Score: ', 'gravityformssurvey' ) . $field_label,
					'tag'   => "{score:id={$field_id}}",
				);
			}
		}

		if ( $scoring_enabled ) {
			$merge_tags[] = array(
				'group' => 'other',
				'label' => esc_html__( 'Survey Total Score', 'gravityformssurvey' ),
				'tag'   => '{survey_total_score}',
			);
		}

		return $merge_tags;
	}

	/**
	 * Replace the score merge tags.
	 *
	 * @param string $text The current text in which merge tags are being replaced.
	 * @param array $form The current form object.
	 * @param array $entry The current entry object.
	 * @param bool $url_encode Whether or not to encode any URLs found in the replaced value.
	 * @param bool $esc_html Whether or not to encode HTML found in the replaced value.
	 * @param bool $nl2br Whether or not to convert newlines to break tags.
	 * @param string $format The format requested for the location the merge is being used. Possible values: html, text or url.
	 *
	 * @return string
	 */
	public function replace_merge_tags( $text, $form, $entry, $url_encode, $esc_html, $nl2br, $format ) {

		if ( empty( $entry ) || empty( $form ) ) {
			return $text;
		}

		$survey_fields = GFAPI::get_fields_by_type( $form, array( 'survey' ) );
		if ( empty( $survey_fields ) ) {
			return $text;
		}

		$total_merge_tag = '{survey_total_score}';

		if ( false !== strpos( $text, $total_merge_tag ) ) {
			$score_total = $this->get_total_score( $form, $entry );
			$text        = str_replace( $total_merge_tag, $score_total, $text );
		}

		preg_match_all( "/\{score:(.*?)\}/", $text, $matches, PREG_SET_ORDER );
		foreach ( $matches as $match ) {

			$full_tag       = $match[0];
			$options_string = isset( $match[1] ) ? $match[1] : '';
			$options        = shortcode_parse_atts( $options_string );
			extract(
				shortcode_atts(
					array(
						'id' => 0,
					), $options
				)
			);
			if ( 0 == $id ) {
				continue;
			}

			$field = GFFormsModel::get_field( $form, $id );
			if ( ! is_object( $field ) || $field->get_input_type() != 'likert' ) {
				continue;
			}

			if ( $id == intval( $id ) ) {
				$score = $this->get_field_score( $field, $entry );
			} else {
				$score = $this->get_likert_row_score( $field->get_row_id( $id ), $field, $entry );
			}

			$text = str_replace( $full_tag, $url_encode ? urlencode( $score ) : $score, $text );
		}

		return $text;
	}


	// # ENTRY RELATED --------------------------------------------------------------------------------------------------

	/**
	 * Add the Survey Total Score entry meta property.
	 *
	 * @param array $entry_meta An array of entry meta already registered with the gform_entry_meta filter.
	 * @param int $form_id The form id
	 *
	 * @return array The filtered entry meta array.
	 */
	public function get_entry_meta( $entry_meta, $form_id ) {
		if ( empty( $form_id ) ) {
			return $entry_meta;
		}
		$form          = RGFormsModel::get_form_meta( $form_id );
		$survey_fields = GFAPI::get_fields_by_type( $form, array( 'survey' ) );
		if ( false === empty( $survey_fields ) && $this->scoring_enabled( $form ) ) {

			$entry_meta['gsurvey_score'] = array(
				'label'                      => esc_html__( 'Survey Total Score', 'gravityformssurvey' ),
				'is_numeric'                 => true,
				'is_default_column'          => false,
				'update_entry_meta_callback' => array( $this, 'update_entry_meta' ),
				'filter'                     => array(
					'operators' => array( 'is', 'isnot', '>', '<' ),
				),
			);

		}

		return $entry_meta;
	}

	/**
	 * Used to update the Survey Total Score entry meta property
	 *
	 * @param string $key The key of the property to be updated.
	 * @param array $entry The current entry object.
	 * @param array $form The current form object.
	 *
	 * @return mixed
	 */
	public function update_entry_meta( $key, $entry, $form ) {
		$value = '';

		if ( $key == 'gsurvey_score' ) {
			$value = $this->get_survey_score( $form, $entry );
		}


		return $value;
	}

	/**
	 * Remove the survey score from the entry meta conditional logic filters on the notifications/confirmations pages.
	 *
	 * @param array $filters The array of filters.
	 * @param array $form The current form object.
	 * @param string $id The ID of the notification/confirmation being edited.
	 *
	 * @return mixed
	 */
	public function conditional_logic_filters( $filters, $form, $id ) {
		$survey_fields = GFAPI::get_fields_by_type( $form, array( 'survey' ) );
		if ( empty( $survey_fields ) ) {
			return $filters;
		}

		if ( false === $this->scoring_enabled( $form ) ) {
			unset( $filters['gsurvey_score'] );
		}

		return $filters;
	}

	/**
	 * Evaluate conditional logic on the rank field.
	 *
	 * @since 3.9
	 *
	 * @param $is_match
	 * @param $field_value
	 * @param $target_value
	 * @param $operation
	 * @param $source_field
	 * @param $rule
	 *
	 * @return bool
	 */
	public function is_value_match_rank( $is_match, $field_value, $target_value, $operation, $source_field, $rule ) {

		if ( $source_field->type == 'survey' && $source_field->inputType == 'rank' ) {
			if ( '' === $target_value ) {
				// If a target value doesn't get saved, it's because the target value is 1.
				$target_value = 1;
			}

			$field_id_parts = explode('.', $rule['fieldId']);
			if ( 1 === count( $field_id_parts ) ) {
				return $is_match;
			}

			$field_index       = $field_id_parts[1];
			$choice_value      = $source_field->choices[$field_index]['value'];
			$choice_value_index = array_search( $choice_value, $source_field->choices[$field_index] );
			$field_value_array = explode(',', $field_value);
			$target_index	   = $target_value - 1;

			switch ( $rule['operator'] ) {
				case 'is':
					return $choice_value === $field_value_array[ $target_index ];
				case 'isnot':
					return $choice_value !== $field_value_array[ $target_index ];
				case '>':
					return $choice_value_index > $target_index;
				case '<':
					return $choice_value_index < $target_index;
				default:
					return false;
			}
		}

		return $is_match;
	}

	/**
	 * Format the Survey field values for entry exports and the entry list page so they use the choice text instead of values.
	 *
	 * @param string|array $value The field value.
	 * @param int $form_id The ID of the form currently being processed.
	 * @param string $field_id The ID of the field currently being processed.
	 * @param array $entry The entry object currently being processed.
	 *
	 * @return string|array
	 */
	public function export_field_value( $value, $form_id, $field_id, $entry ) {
		if ( ! rgblank( $value ) ) {
			$form_meta = RGFormsModel::get_form_meta( $form_id );
			$field     = RGFormsModel::get_field( $form_meta, $field_id );

			return $this->maybe_format_field_values( $value, $field );
		}

		return $value;
	}

	/**
	 * Format the Survey field values on the entry detail page so they use the choice text instead of values.
	 *
	 * @param string|array $value The field value.
	 * @param GF_Field $field The field currently being processed.
	 * @param array $entry The entry object currently being processed.
	 * @param array $form The form object currently being processed.
	 *
	 * @return string|array
	 */
	public function entry_field_value( $value, $field, $entry, $form ) {

		return ! rgblank( $value ) ? $this->maybe_format_field_values( $value, $field ) : $value;
	}

	/**
	 * Format the Survey field values so they use the choice text instead of values before being passed to the third-party.
	 *
	 * @param string $value The field value.
	 * @param array $form The form currently being processed.
	 * @param array $entry The entry object currently being processed.
	 * @param string $field_id The ID of the field currently being processed.
	 *
	 * @return string
	 */
	public function addon_field_value( $value, $form, $entry, $field_id, $slug ) {
		if ( ! rgblank( $value ) ) {
			$field = RGFormsModel::get_field( $form, $field_id );

			return $this->maybe_format_field_values( $value, $field );
		}

		return $value;
	}

	/**
	 * If the field is a Survey type radio, select or checkbox then replace the choice value with the choice text.
	 *
	 * @param string $value The field value.
	 * @param GF_Field|null $field The field object being processed or null.
	 *
	 * @return string
	 */
	public function maybe_format_field_values( $value, $field ) {

		if ( is_object( $field ) && $field->type == 'survey' ) {
			switch ( $field->inputType ) {
				case 'radio' :
				case 'select' :
					return RGFormsModel::get_choice_text( $field, $value );

				case 'checkbox' :
					if ( is_array( $value ) ) {
						foreach ( $value as &$choice ) {
							if ( ! empty( $choice ) ) {
								$choice = RGFormsModel::get_choice_text( $field, $choice );
							}
						}
					} else {
						foreach ( $field->choices as $choice ) {
							$val   = rgar( $choice, 'value' );
							$text  = rgar( $choice, 'text' );
							$value = str_replace( $val, $text, $value );
						}
					}
			}
		}

		return $value;
	}

	/**
	 * Format the Survey field values so they use the choice text instead of values before being passed to the third-party.
	 *
	 * @param string $value The field value.
	 * @param int $form_id The ID of the form currently being processed.
	 * @param string $field_id The ID of the field currently being processed.
	 * @param array $entry The entry object currently being processed.
	 *
	 * @return string
	 */
	public function legacy_addon_field_value( $value, $form_id, $field_id, $entry ) {
		if ( ! rgblank( $value ) ) {
			$form_meta = RGFormsModel::get_form_meta( $form_id );
			$field     = RGFormsModel::get_field( $form_meta, $field_id );

			if ( is_object( $field ) && $field->type == 'survey' ) {
				return $field->get_value_export( $entry, $field_id, true );
			}
		}

		return $value;
	}

	/**
	 * Format the Survey field values so they use the choice text instead of values before they are sent to Zapier.
	 *
	 * @param string|array $value The field value.
	 * @param int $form_id The ID of the form currently being processed.
	 * @param string $field_id The ID of the field currently being processed.
	 * @param array $entry The entry object currently being processed.
	 *
	 * @return string|array
	 */
	public function zapier_field_value( $value, $form_id, $field_id, $entry ) {
		if ( ! rgblank( $value ) ) {
			$form_meta = RGFormsModel::get_form_meta( $form_id );
			$field     = RGFormsModel::get_field( $form_meta, $field_id );

			if ( is_object( $field ) && $field->type == 'survey' ) {
				switch ( $field->inputType ) {
					case 'likert' :
						if ( is_array( $value ) ) {
							foreach ( $value as $key => &$row ) {
								if ( ! empty( $row ) ) {
									$row = $field->get_column_text( $value, false, $key );
								}
							}
						} else {

							return $field->get_column_text( $value );
						}

						break;

					case 'rank' :
					case 'rating' :
					case 'radio' :
					case 'select' :
						return $field->get_value_export( $entry, $field_id, true );

					case 'checkbox' :
						foreach ( $value as &$choice ) {
							if ( ! empty( $choice ) ) {
								$choice = RGFormsModel::get_choice_text( $field, $choice );
							}
						}
				}
			}
		}

		return $value;
	}


	// # FIELD SETTINGS -------------------------------------------------------------------------------------------------

	/**
	 * Add the gsurvey-field class to the Survey field.
	 *
	 * @param string $classes The CSS classes to be filtered, separated by empty spaces.
	 * @param GF_Field $field The field currently being processed.
	 * @param array $form The form currently being processed.
	 *
	 * @return string
	 */
	public function add_custom_class( $classes, $field, $form ) {
		if ( $field->type == 'survey' ) {
			$classes .= ' gsurvey-survey-field ';
		}

		return $classes;
	}

	/**
	 * Add the tooltips for the Survey field.
	 *
	 * @param array $tooltips An associative array of tooltips where the key is the tooltip name and the value is the tooltip.
	 *
	 * @return array
	 */
	public function add_survey_tooltips( $tooltips ) {
		$tooltips['gsurvey_question']                    = '<h6>' . esc_html__( 'Survey Question', 'gravityformssurvey' ) . '</h6>' . esc_html__( 'Enter the question you would like to ask the user.', 'gravityformssurvey' );
		$tooltips['gsurvey_field_type']                  = '<h6>' . esc_html__( 'Survey Field Type', 'gravityformssurvey' ) . '</h6>' . esc_html__( 'Select the type of field that will be used for the survey.', 'gravityformssurvey' );
		$tooltips['gsurvey_likert_columns']              = '<h6>' . esc_html__( 'Likert Columns', 'gravityformssurvey' ) . '</h6>' . esc_html__( 'Edit the choices for this likert field.', 'gravityformssurvey' );
		$tooltips['gsurvey_likert_enable_multiple_rows'] = '<h6>' . esc_html__( 'Enable Multiple Rows', 'gravityformssurvey' ) . '</h6>' . esc_html__( 'Select to add multiple rows to the likert field.', 'gravityformssurvey' );
		$tooltips['gsurvey_likert_rows']                 = '<h6>' . esc_html__( 'Likert Rows', 'gravityformssurvey' ) . '</h6>' . esc_html__( 'Edit the texts that will appear to the left of each row of choices.', 'gravityformssurvey' );
		$tooltips['gsurvey_likert_enable_scoring']       = '<h6>' . esc_html__( 'Enable Scoring', 'gravityformssurvey' ) . '</h6>' . esc_html__( 'Scoring allows different scores for each column. Aggregate scores are displayed in the results page and can be used in merge tags.', 'gravityformssurvey' );

		return $tooltips;
	}

	/**
	 * Add the custom settings for the Survey fields to the fields general tab.
	 *
	 * @param int $position The position the settings should be located at.
	 * @param int $form_id  The ID of the form currently being edited.
	 */
	public function survey_field_settings( $position, $form_id ) {
		if ( $position == 25 ) {
			?>
			<li class="gsurvey-setting-question field_setting">
				<label for="gsurvey-question" class="section_label">
					<?php esc_html_e( 'Survey Question', 'gravityformssurvey' ); ?>
					<?php gform_tooltip( 'gsurvey_question' ); ?>
				</label>
				<textarea
					id="gsurvey-question"
					class="fieldwidth-3 fieldheight-2"
					onkeyup="SetFieldLabel(this.value)"
					size="35">
				</textarea>
			</li>
			<li class="gsurvey-setting-field-type field_setting">
				<label for="gsurvey-field-type" class="section_label">
					<?php esc_html_e( 'Survey Field Type', 'gravityformssurvey' ); ?>
					<?php gform_tooltip( 'gsurvey_field_type' ); ?>
				</label>
				<select
					id="gsurvey-field-type"
					onchange="if(jQuery(this).val() == '') return; jQuery('#field_settings').slideUp(function(){StartChangeSurveyType(jQuery('#gsurvey-field-type').val());});"
				>
					<option value="likert"><?php esc_html_e( 'Likert', 'gravityformssurvey' ); ?></option>
					<option value="rank"><?php esc_html_e( 'Rank', 'gravityformssurvey' ); ?></option>
					<option value="rating"><?php esc_html_e( 'Rating', 'gravityformssurvey' ); ?></option>
					<option value="radio"><?php esc_html_e( 'Radio Buttons', 'gravityformssurvey' ); ?></option>
					<option value="checkbox"><?php esc_html_e( 'Checkboxes', 'gravityformssurvey' ); ?></option>
					<option value="text"><?php esc_html_e( 'Single Line Text', 'gravityformssurvey' ); ?></option>
					<option value="textarea"><?php esc_html_e( 'Paragraph Text', 'gravityformssurvey' ); ?></option>
					<option value="select"><?php esc_html_e( 'Drop Down', 'gravityformssurvey' ); ?></option>
				</select>
			</li>
			<?php
		} elseif ( $position == 1362 ) {
			?>
			<li class="gsurvey-likert-setting-columns field_setting">
				<div class="gsurvey-likert-setting-columns-header">
					<label for="gsurvey-likert-columns" class="section_label">
						<?php esc_html_e( 'Columns', 'gravityformssurvey' ); ?>
						<?php gform_tooltip( 'gsurvey_likert_columns' ); ?>
					</label>

					<div class="gsurvey-likert-enable-scoring-container">
						<input
							id="gsurvey-likert-enable-scoring"
							type="checkbox"
							onclick="SetFieldProperty('gsurveyLikertEnableScoring', this.checked); jQuery('#gsurvey-likert-columns-container').toggleClass('gsurvey-likert-scoring-enabled');"
						/>
						<label class="inline gfield_value_label" for="gsurvey-likert-enable-scoring">
							<?php esc_html_e( 'Enable Scoring', 'gravityformssurvey' ); ?>
						</label> <?php gform_tooltip( 'gsurvey_likert_enable_scoring' ); ?>
					</div>
				</div>

				<div id="gsurvey-likert-columns-container">
					<ul id="gsurvey-likert-columns"></ul>
				</div>
			</li>
			<li class="gsurvey-likert-setting-enable-multiple-rows field_setting">
				<input
					type="checkbox"
					id="gsurvey-likert-enable-multiple-rows"
					onclick="field = GetSelectedField(); var value = jQuery(this).is(':checked'); SetFieldProperty('gsurveyLikertEnableMultipleRows', value); gsurveyLikertUpdateInputs(field); gsurveyLikertUpdatePreview(); jQuery('.gsurvey-likert-setting-rows').toggle();"
				/>
				<label for="gsurvey-likert-enable-multiple-rows" class="inline">
					<?php esc_html_e( 'Enable Multiple Rows', 'gravityformssurvey' ); ?>
					<?php gform_tooltip( 'gsurvey_likert_enable_multiple_rows' ); ?>
				</label>

			</li>
			<li class="gsurvey-likert-setting-rows field_setting">
				<label for="gsurvey-likert-rows" class="section_label">
					<?php esc_html_e( 'Rows', 'gravityformssurvey' ); ?>
					<?php gform_tooltip( 'gsurvey_likert_rows' ); ?>
				</label>
				<div id="gsurvey-likert-rows-container">
					<ul id="gsurvey-likert-rows"></ul>
				</div>
			</li>
			<?php
		}
	}


	// # CONTACTS INTEGRATION -------------------------------------------------------------------------------------------

	public function add_tab_to_contact_detail( $tabs, $contact_id ) {
		if ( $contact_id > 0 ) {
			$tabs[] = array( 'name' => 'survey', 'label' => esc_html__( 'Survey Entries', 'gravityformssurvey' ) );
		}

		return $tabs;
	}

	public function contacts_tab( $contact_id ) {

		if ( false === empty( $contact_id ) ) :
			$search_criteria['status'] = 'active';
			$search_criteria['field_filters'][] = array(
				'type'  => 'meta',
				'key'   => 'gcontacts_contact_id',
				'value' => $contact_id,
			);
			$form_ids                           = array();
			$forms                              = GFFormsModel::get_forms( true );
			foreach ( $forms as $form ) {
				$form_meta     = GFFormsModel::get_form_meta( $form->id );
				$survey_fields = GFCommon::get_fields_by_type( $form_meta, array( 'survey' ) );
				if ( ! empty( $survey_fields ) ) {
					$form_ids[] = $form->id;
				}
			}

			if ( empty( $form_ids ) ) {
				return;
			}
			$entries                   = GFAPI::get_entries( $form_ids, $search_criteria );

			if ( empty( $entries ) ) :
				esc_html_e( 'This contact has not submitted any survey entries yet.', 'gravityformssurvey' );
			else :
				?>
				<h3><span><?php esc_html_e( 'Survey Entries', 'gravityformssurvey' ) ?></span></h3>
				<div>
					<table id="gcontacts-entry-list" class="widefat">
						<tr class="gcontacts-entries-header">
							<td>
								<?php esc_html_e( 'Entry ID', 'gravityformssurvey' ) ?>
							</td>
							<td>
								<?php esc_html_e( 'Date', 'gravityformssurvey' ) ?>
							</td>
							<td>
								<?php esc_html_e( 'Form', 'gravityformssurvey' ) ?>
							</td>
						</tr>
						<?php


						foreach ( $entries as $entry ) {
							$form_id    = $entry['form_id'];
							$form       = GFFormsModel::get_form_meta( $form_id );
							$form_title = rgar( $form, 'title' );
							$entry_id   = $entry['id'];
							$entry_date = GFCommon::format_date( rgar( $entry, 'date_created' ), false );
							$entry_url  = admin_url( "admin.php?page=gf_entries&view=entry&id={$form_id}&lid={$entry_id}" );

							?>
							<tr>
								<td>
									<a href="<?php echo $entry_url; ?>"><?php echo $entry_id; ?></a>
								</td>
								<td>
									<?php echo $entry_date; ?>
								</td>
								<td>
									<?php echo $form_title; ?>
								</td>


							</tr>
							<?php
						}
						?>
					</table>
				</div>
				<?php
			endif;
		endif;

	}


	// # DEPRECATED -----------------------------------------------------------------------------------------------------

	/**
	 * Returns the Likert column text.
	 *
	 * @deprecated
	 * @param GF_Field_Likert $field The field being processed.
	 * @param string $value The field value.
	 *
	 * @return string
	 */
	public function get_likert_column_text( $field, $value ) {
		_deprecated_function( __FUNCTION__, '3.0', '$field->get_column_text( $value, $entry = false, $field_id = false, $include_row_text = false )' );

		if ( $field->gsurveyLikertEnableMultipleRows ) {
			if ( false === strpos( $value, ':' ) ) {
				return '';
			}
			list( $row_val, $col_val ) = explode( ':', $value, 2 );

			foreach ( $field->gsurveyLikertRows as $row ) {
				if ( $row['value'] == $row_val ) {
					foreach ( $field->choices as $choice ) {
						if ( $choice['value'] == $col_val ) {
							return $choice['text'];
						}
					}
				}
			}
		} else {
			foreach ( $field->choices as $choice ) {
				if ( $choice['value'] == $value ) {
					return $choice['text'];
				}
			}
		}

	}

	/**
	 * Returns the label for the row at the specified index in the gsurveyLikertRows array.
	 *
	 * @param GF_Field_Likert $field The field being processed.
	 * @param integer $index The row index.
	 *
	 * @return string
	 */
	public function get_likert_row_text( $field, $index ) {

		return $field->gsurveyLikertEnableMultipleRows ? $field->gsurveyLikertRows[ $index ]['text'] : '';
	}

	/**
	 * Support importing forms using the old XML format.
	 *
	 * @param array $options Array of options for the XML import.
	 *
	 * @return array
	 */
	public function import_file_options( $options ) {
		$options['gsurveyLikertRow'] = array( 'unserialize_as_array' => true );

		return $options;
	}

} // end class
