<?php // -*- coding: utf-8 -*-

namespace QuickAssortmentsSP\COG\Assets;

/**
 * Class AssetsEnqueue.
 *
 * @package  QuickAssortmentsSP\COG\AssetsSP
 *
 * @since    1.0.0
 */
final class AssetsSP {

	/**
	 * The assets of WooCommerce.
	 *
	 * @since    1.0.0
		 *
	 * @var string $version The current version of this plugin.
	 */
	private $wc_assets;

	/**
	 * AssetsEnqueue constructor.
	 *
	 * @return void
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
	}

	/**
	 * Enqueueing scripts and styles.
	 *
	 * @return object
	 *
	 * @since 1.0.0
	 */
	public function init() {
		add_action('admin_enqueue_scripts', [$this, 'styles']);
		add_action('admin_enqueue_scripts', [$this, 'scripts']);

		global $pagenow, $wp_filter;
		if ('admin.php' === $pagenow && QA_COG_RI_SLUGSP === sanitize_key(empty($_GET['page']) ? '' : $_GET['page'])) {
			// Preventing 'woocommerce-admin' scripts from enqueue. It's important!
			unset($wp_filter['admin_enqueue_scripts']->callbacks[10]['Automattic\WooCommerce\Admin\Loader::register_scripts'] , $wp_filter['admin_enqueue_scripts']->callbacks[15]['Automattic\WooCommerce\Admin\Loader::load_scripts']);

			add_action('admin_print_scripts', [$this, 'print_script_wc_settings'], 1);
			add_action('admin_enqueue_scripts', [$this, 'ri_styles_scripts']);
		}

		return $this;
	}

	/**
	 * Enqueueing styles.
	 *
	 * @return void
	 *
	 * @since 1.0.0
	 */
	public function styles() {
		$remove_pages = ['shelf_planner_retail_insights', 'shelf_planner_product_management'];

		if (isset($_GET['page']) && in_array($_GET['page'], $remove_pages)) {
			$this->wc_assets = new \WC_Admin_Assets();
			$this->wc_assets->admin_styles();

			wp_enqueue_style('qa-cog-sp', QA_COG_BASE_URLSP . 'assets/dist/quickassortments/quickassortments.css?45', null, '1.0.3', 'all');
			wp_enqueue_style('woocommerce_admin_styles');
			wp_enqueue_style('woocommerce_admin_dashboard_styles');
		}
	}

	/**
	 * Enqueueing scripts.
	 *
	 * @return void
	 *
	 * @since 1.0.0
	 */
	public function scripts() {
		$this->wc_assets = new \WC_Admin_Assets();
		$this->wc_assets->admin_scripts();

		$remove_pages = ['shelf_planner_retail_insights', 'shelf_planner_product_management'];

		$params = [
			'strings' => [
				'import_products' => esc_html__('Import', 'woocommerce'),
				'export_products' => esc_html__('Export', 'woocommerce'),
			],
			'urls'    => [
				'import_products' => current_user_can('import')
					? esc_url_raw(admin_url('edit.php?post_type=product&page=product_importer'))
					: null,
				'export_products' => current_user_can('export')
					? esc_url_raw(admin_url('edit.php?post_type=product&page=product_exporter'))
					: null,
			],
		];
		wp_localize_script('woocommerce_admin', 'woocommerce_admin', $params);

		if (isset($_GET['page']) && in_array($_GET['page'], $remove_pages)) {
			// Registering the script.
			wp_register_script('qa-cog-sp', QA_COG_BASE_URLSP . 'assets/dist/quickassortments/quickassortments.js?32', [ 'jquery' ], '1.0.0', true);
			// Local JS data
			$local_js_data = [ 'ajax_url' => admin_url('admin-ajax.php'), 'currency' => get_woocommerce_currency(), ];
			// Pass data to myscript.js on page load
			wp_localize_script('qa-cog', 'QACOGAjaxObj', $local_js_data);
			// Enqueueing JS file.
			wp_enqueue_script('qa-cog');

			// Style ans Scripts Including
			wp_enqueue_script('woocommerce_admin');
			wp_enqueue_script('jquery-blockui');
			wp_enqueue_script('jquery-tiptip');
			wp_enqueue_script('flot');
			wp_enqueue_script('flot-resize');
			wp_enqueue_script('flot-time');
			wp_enqueue_script('flot-pie');
			wp_enqueue_script('flot-stack');
			wp_enqueue_script('select2');
			wp_enqueue_script('wc-enhanced-select');
		}
	}

	/**
	 * These are used by @woocommerce/components & the block library to set up defaults
	 * based on user-controlled settings from WordPress. Only use this in wp-admin.
	 */
	public function print_script_wc_settings() {
		global $wp_locale;
		$code = get_woocommerce_currency();

		// NOTE: wcSettings is not used directly, it's only for @woocommerce/components
		//
		// SettingsSPSP and variables can be passed here for access in the app.
		// Will need `wcAdminAssetUrl` if the ImageAsset component is used.
		// Will need `dataEndpoints.countries` if Search component is used with 'country' type.
		// Will need `orderStatuses` if the OrderStatus component is used.
		// Deliberately excluding: `embedBreadcrumbs`, `trackingEnabled`.

		$woocommerce_currency = get_woocommerce_currency();
		$default              = get_option('sp.default_currency', true);

		$settings = [
			'adminUrl'      => admin_url(),
			'wcAssetUrl'    => plugins_url('assets/', WC_PLUGIN_FILE),
			'siteLocale'    => esc_attr(get_bloginfo('language')),
			'currency'      => [
				'code'              => $code,
				'precision'         => wc_get_price_decimals(),
				'symbol'            => $woocommerce_currency != $default ? html_entity_decode(get_woocommerce_currency_symbol($default)) : html_entity_decode(get_woocommerce_currency_symbol($code)),
				'symbolPosition'    => get_option('woocommerce_currency_pos'),
				'decimalSeparator'  => wc_get_price_decimal_separator(),
				'thousandSeparator' => wc_get_price_thousand_separator(),
				'priceFormat'       => html_entity_decode(get_woocommerce_price_format()),
			],
			'stockStatuses' => wc_get_product_stock_status_options(),
			'siteTitle'     => get_bloginfo('name'),
			'dataEndpoints' => [
				'leaderboards' => [
					[
						'id'      => 'categories_net_profit',
						'label'   => esc_html__('Top Categories - Net Profit', 'qa-cost-of-goods-margins'),
						'headers' => [
							[
								'label' => esc_html__('Category', 'qa-cost-of-goods-margins'),
							],
							[
								'label' => esc_html__('Items Sold', 'qa-cost-of-goods-margins'),
							],
							[
								'label' => esc_html__('Net Profit', 'qa-cost-of-goods-margins'),
							],
						],
						'rows'    => [],
					],
					[
						'id'      => 'products_net_profit',
						'label'   => esc_html__('Top Products - Net Profit', 'qa-cost-of-goods-margins'),
						'headers' => [
							[
								'label' => esc_html__('Product', 'qa-cost-of-goods-margins'),
							],
							[
								'label' => esc_html__('Items Sold', 'qa-cost-of-goods-margins'),
							],
							[
								'label' => esc_html__('Net Profit', 'qa-cost-of-goods-margins'),
							],
						],
						'rows'    => [],
					],
					[
						'id'      => 'categories_net_margin',
						'label'   => esc_html__('Top Categories - Net Margin', 'qa-cost-of-goods-margins'),
						'headers' => [
							[
								'label' => esc_html__('Category', 'qa-cost-of-goods-margins'),
							],
							[
								'label' => esc_html__('Items Sold', 'qa-cost-of-goods-margins'),
							],
							[
								'label' => esc_html__('Net Margin', 'qa-cost-of-goods-margins'),
							],
						],
						'rows'    => [],
					],
					[
						'id'      => 'products_net_margin',
						'label'   => esc_html__('Top Products - Net Margin', 'qa-cost-of-goods-margins'),
						'headers' => [
							[
								'label' => esc_html__('Product', 'qa-cost-of-goods-margins'),
							],
							[
								'label' => esc_html__('Items Sold', 'qa-cost-of-goods-margins'),
							],
							[
								'label' => esc_html__('Net Margin', 'qa-cost-of-goods-margins'),
							],
						],
						'rows'    => [],
					],
				],
			],
			'wcAdminSettings' => [
				'id'          => 'woocommerce_default_date_range',
				'option_key'  => 'woocommerce_default_date_range',
				'label'       => esc_html__('Default Date Range', 'qa-cost-of-goods-margins'),
				'description' => esc_html__('Default Date Range', 'qa-cost-of-goods-margins'),
				'default'     => 'period=month&compare=previous_year',
				'type'        => 'text',
			],
			'l10n'            => [
				'userLocale'    => get_user_locale(),
				'weekdaysShort' => array_values($wp_locale->weekday_abbrev),
			],
		];
		// NOTE: wcSettings is not used directly, it's only for @woocommerce/components.
		// $settings = apply_filters('woocommerce_components_settings', $settings);?>
		<script type="text/javascript">
			var wcSettings = wcSettings || JSON.parse( decodeURIComponent( '<?php echo rawurlencode(wp_json_encode($settings)); ?>' ) );
		</script>
		<?php
	}

	/**
	 * Enqueueing RISP styles and scripts.
	 *
	 * @return void
	 *
	 * @since 1.0.0
	 */
	public function ri_styles_scripts() {
		if (isset($_GET['page']) && sanitize_text_field('_qa_cog_retail_insights' != $_GET['page'])) {
			wp_register_style(
				'wc-components',
				QA_COG_BASE_URLSP . 'assets/dist/components/style.css',
				['wp-components'],
				true
			);
		}
		wp_style_add_data('wc-components', 'rtl', 'replace');

		wp_register_style(
			'wc-components-ie',
			QA_COG_BASE_URLSP . 'assets/dist/components/ie.css',
			['wp-components'],
			true
		);
		wp_style_add_data('wc-components-ie', 'rtl', 'replace');

		wp_register_script(
			'qa-admin-app',
			QA_COG_BASE_URLSP . 'assets/dist/app/index.js?4',
			['wc-components', 'wc-navigation', 'wp-date', 'wp-html-entities', 'wp-keycodes', 'wp-i18n'],
			true,
			true
		);

		wp_set_script_translations('qa-admin-app', 'qa-cost-of-goods-margins');

		if (isset($_GET['page']) && sanitize_text_field('_qa_cog_retail_insights' != $_GET['page'])) {
			wp_register_style(
				'qa-admin-app',
				QA_COG_BASE_URLSP . 'assets/dist/app/style.css?39',
				['wc-components'],
				true
			);
		}

		wp_style_add_data('qa-admin-app', 'rtl', 'replace');

		wp_register_style(
			'wc-material-icons',
			'https://fonts.googleapis.com/icon?family=Material+Icons+Outlined',
			[],
			true
		);

		wp_enqueue_style('qa-admin-app');
		wp_enqueue_style('wc-material-icons');

		wp_register_script(
			'wc-csv',
			QA_COG_BASE_URLSP . 'assets/dist/csv-export/index.js?4',
			[],
			true,
			true
		);

		wp_register_script(
			'wc-currency',
			QA_COG_BASE_URLSP . 'assets/dist/currency/index.js?4',
			['wc-number'],
			true,
			true
		);

		wp_set_script_translations('wc-currency', 'qa-cost-of-goods-margins');

		wp_register_script(
			'wc-navigation',
			QA_COG_BASE_URLSP . 'assets/dist/navigation/index.js?4',
			[],
			true,
			true
		);

		wp_register_script(
			'wc-number',
			QA_COG_BASE_URLSP . 'assets/dist/number/index.js?4',
			[],
			true,
			true
		);

		wp_register_script(
			'wc-date',
			QA_COG_BASE_URLSP . 'assets/dist/date/index.js?4',
			['wp-date', 'wp-i18n'],
			true,
			true
		);

		wp_set_script_translations('wc-date', 'qa-cost-of-goods-margins');

		wp_register_script(
			'wc-components',
			QA_COG_BASE_URLSP . 'assets/dist/components/index.js?4',
			[
				'wp-api-fetch',
				'wp-components',
				'wp-data',
				'wp-element',
				'wp-hooks',
				'wp-i18n',
				'wp-keycodes',
				'wc-csv',
				'wc-currency',
				'wc-date',
				'wc-navigation',
				'wc-number',
			],
			true,
			true
		);

		wp_set_script_translations('wc-components', 'qa-cost-of-goods-margins');

		wp_enqueue_script('qa-admin-app');
	}
}
