/**
 * External dependencies
 */
import classnames from 'classnames';
import memize from 'memize';

/**
 * WordPress dependencies
 */
import { __, sprintf } from '@wordpress/i18n';
import { dateI18n } from '@wordpress/date';
import { Button, Modal } from '@wordpress/components';
import { compose, withState, withInstanceId } from '@wordpress/compose';

/**
 * Internal dependencies
 */
import { getConfigValue } from '../../utils';
import ScanOnly from './scan-only';
import Header, { Title, Date } from '../../components/card/header';
import { FooterSchemaActions } from '../../components/card/footer';
import { SiteScanResults, PrintR } from '@ithemes/security-components';
import './style.scss';

const isSameUrl = memize( ( a, b ) => String( a ).replace( /https?:\/\//, '' ) === String( b ).replace( /https?:\/\//, '' ) );

function MalwareScan( { card, config, viewEntry, scanResults, showRawDetails, setState, instanceId } ) {
	return (
		<div className="itsec-card--type-malware-scan">
			<Header>
				<Title card={ card } config={ config } />
				<Date card={ card } config={ config } />
			</Header>
			<section className="itsec-card-malware-scan__scans-section">
				<table className="itsec-card-malware-scan__scans">
					<thead>
						<tr>
							<th>{ __( 'Time', 'it-l10n-ithemes-security-pro' ) }</th>
							<th>{ __( 'Status', 'it-l10n-ithemes-security-pro' ) }</th>
							<th><span className="screen-reader-text">{ __( 'Actions', 'it-l10n-ithemes-security-pro' ) }</span></th>
						</tr>
					</thead>
					<tbody>
						{ card.data.map( ( scan ) => {
							const id = scan.id;
							const status = scan.status;
							const label = scan.description;

							return (
								<tr key={ id }>
									<th scope="row">{ dateI18n( 'M d, Y g:i A', scan.time ) }</th>
									<td>
										<span className={ classnames( 'itsec-card-malware-scan__scan-status', `itsec-card-malware-scan__scan-status--${ status }` ) }>
											{ label }
										</span>
									</td>
									<td>
										<Button isLink aria-pressed={ viewEntry === id } onClick={ () => setState( { viewEntry: id } ) }>
											{ __( 'View', 'it-l10n-ithemes-security-pro' ) }
										</Button>
										{ viewEntry === id && (
											<Modal
												title={ sprintf( __( 'View Scan Details for %s', 'it-l10n-ithemes-security-pro' ), dateI18n( 'M d, Y g:i A', scan.time ) ) }
												onRequestClose={ () => setState( { viewEntry: 0, showRawDetails: false } ) }
											>
												<SiteScanResults results={ scan } showSiteUrl={ ! isSameUrl( scan.url, getConfigValue( 'site_url' ) ) } />
												<Button className="itsec-card-malware-scan__raw-details-toggle" isLink onClick={ () => setState( { showRawDetails: ! showRawDetails } ) }
													aria-expanded={ showRawDetails } aria-controls={ `itsec-card-malware-scan__raw-details--${ instanceId }` }>
													{ showRawDetails ? __( 'Hide Raw Details', 'it-l10n-ithemes-security-pro' ) : __( 'Show Raw Details', 'it-l10n-ithemes-security-pro' ) }
												</Button>
												<div id={ `itsec-card-malware-scan__raw-details--${ instanceId }` } style={ { visibility: showRawDetails ? 'visible' : 'hidden' } }>
													{ showRawDetails && <PrintR json={ scan } /> }
												</div>
											</Modal>
										) }
									</td>
								</tr>
							);
						} ) }
					</tbody>
				</table>
			</section>
			<FooterSchemaActions card={ card } onComplete={ ( href, response ) => href.endsWith( '/scan' ) && setState( { scanResults: response } ) } />
			{ scanResults && (
				<Modal
					title={ __( 'Scan Results', 'it-l10n-ithemes-security-pro' ) }
					onRequestClose={ () => setState( { scanResults: undefined } ) }
				>
					<SiteScanResults results={ scanResults } showSiteUrl={ false } />
				</Modal>
			) }
		</div>
	);
}

export const slug = 'malware-scan';
export const settings = {
	render: getConfigValue( 'db_logs' ) ? compose( [
		withState( { viewEntry: 0, scanResults: undefined, showRawDetails: false } ),
		withInstanceId,
	] )( MalwareScan ) : ScanOnly,
};
