<?php

// Allow ':', '[' and ']' characters in HTML class attribute
add_filter( 'sanitize_html_class', 'kaliroots_sanitize_html_class', 10, 2 );
function kaliroots_sanitize_html_class( $sanitized, $classname ) {
	if ( $sanitized !== $classname ) {
		$sanitized = preg_replace( '/[^A-Za-z0-9:\[\]_-]/', '', $classname );
	}
	return $sanitized;
}

// Helper: Determine PurgeCSS whitelist file path
function kaliroots_purgecss_whitelist_path() {
	return get_stylesheet_directory() . '/includes/core/purgecss-whitelist.php';
}

// Helper: Load PurgeCSS whitelist
function kaliroots_load_purgecss_whitelist() {
	$whitelist = array();
	if ( ! file_exists( kaliroots_purgecss_whitelist_path() ) ) {
		return $whitelist;
	}
	if ( preg_match( '/\/\*(.*)\*\//', file_get_contents( kaliroots_purgecss_whitelist_path() ), $matches ) > 0 ) {
		$whitelist = explode( ' ', $matches[1] );
	}
	return array_filter( $whitelist );
}

// Helper: Save PurgeCSS whitelist (sorted / unique)
function kaliroots_save_purgecss_whitelist( $whitelist ) {
	if ( ! file_exists( kaliroots_purgecss_whitelist_path() ) ) {
		touch( kaliroots_purgecss_whitelist_path() );
	}
	$whitelist = array_unique( array_filter( $whitelist ) );
	sort( $whitelist );
	$content = "<?php\n/* " . implode( ' ', $whitelist ) . " */\n";
	file_put_contents( kaliroots_purgecss_whitelist_path(), $content );
}

// Helper: Extract classes from WP_Post
function kaliroots_purgecss_whitelist_from_post( WP_Post $post ) {
	$whitelist = array();
	switch ( $post->post_type ) {
		case 'nav_menu_item':
			$menu_item_classes = get_post_meta( $post->ID, '_menu_item_classes', true );
			$whitelist         = array_merge( $whitelist, is_array( $menu_item_classes ) ? $menu_item_classes : array( $menu_item_classes ) );
			break;
		default:
			if ( preg_match_all( '/class=\"(.*?)\"/', do_shortcode( $post->post_content ), $matches ) > 0 ) {
				foreach ( $matches[1] as $match ) {
					$whitelist = array_merge( $whitelist, explode( ' ', $match ) );
				}
			}
			break;
	}
	return array_filter( array_unique( $whitelist ) );
}

// AJAX: Regenerate PurgeCSS whitelist
add_action( 'wp_ajax_regenerate_purgecss_whitelist', 'cg_ajax_regenerate_purgecss_whitelist' );
function cg_ajax_regenerate_purgecss_whitelist() {
	$whitelist = array();
	$posts     = get_posts(
		array(
			'post_type' => apply_filters( 'kaliroots_purgecss_post_types', array( 'post', 'page', 'nav_menu_item' ) ),
			'nopaging'  => true,
		)
	);
	foreach ( $posts as $post ) {
		$whitelist = array_merge( $whitelist, kaliroots_purgecss_whitelist_from_post( $post ) );
	}
	kaliroots_save_purgecss_whitelist( $whitelist );
	wp_die( 'OK' );
}

// Update PurgeCSS whitelist when saving post
add_action( 'save_post', 'kaliroots_purgecss_whitelist_save_post', 10, 2 );
function kaliroots_purgecss_whitelist_save_post( $post_id, $post ) {
	if ( ! in_array( $post->post_type, apply_filters( 'kaliroots_purgecss_post_types', array( 'post', 'page', 'nav_menu_item' ) ) ) ) {
		return;
	}
	$whitelist = array_merge(
		kaliroots_load_purgecss_whitelist(),
		kaliroots_purgecss_whitelist_from_post( $post ),
	);
	kaliroots_save_purgecss_whitelist( $whitelist );
}
