<?php

/**
 * Initialize the Plugin.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Install' ) ) {

	/**
	 * Main MMP_Install Class.
	 * */
	class MMP_Install {

		public static function init() {
			add_action( 'woocommerce_init', array( __CLASS__, 'check_version' ) );
			add_filter( 'plugin_action_links_' . MMP_PLUGIN_SLUG, array( __CLASS__, 'settings_link' ) );
		}

		/**
		 * Check current version of the plugin is updated when activating plugin, if not run updater.
		 */
		public static function check_version() {
			if ( version_compare( get_option( 'mmp_version' ), MMP_VERSION, '>=' ) ) {
				return;
			}

			self::install();
		}

		/**
		 * Install
		 */
		public static function install() {
			self::set_default_values(); // default values
			self::set_form_fields();
			self::update_version();
		}

		/**
		 *  Settings link.
		 */
		public static function settings_link( $links ) {
			$setting_page_link = '<a href="' . mmp_get_settings_page_url() . '">' . esc_html__( 'Settings', 'lower-price-for-woocommerce' ) . '</a>';

			array_unshift( $links, $setting_page_link );

			return $links;
		}

		/**
		 *  Set settings default values
		 */
		public static function set_default_values() {

			if ( ! class_exists( 'MMP_Settings' ) ) {
				include_once MMP_PLUGIN_PATH . '/inc/admin/menu/class-mmp-settings.php';
			}

			// default for settings
			$settings = MMP_Settings::get_settings_pages();

			foreach ( $settings as $setting_key => $value ) {

				$settings_array = $value->get_settings( $setting_key );

				foreach ( $settings_array as $value ) {
					if ( isset( $value['default'] ) && isset( $value['id'] ) ) {
						if ( get_option( $value['id'] ) === false ) {
							add_option( $value['id'], $value['default'] );
						}
					}
				}
			}
		}

		/**
		 * Update current version.
		 */
		private static function update_version() {
			update_option( 'mmp_version', MMP_VERSION );
		}

		/**
		 * Register Default Fields
		 */
		public static function set_form_fields() {
			if ( self::get_fields() ) {
				return true; // default fields is exists.
			}

			$fields = self::default_fields_args();

			if ( ! mmp_check_is_array( $fields ) ) {
				return false;
			}

			foreach ( $fields as $field ) {
				self::create( $field ); // register a field as post
			}
		}

		/**
		 * Create a new field
		 */
		public static function create( $fields_data ) {
			$fields_data = wp_parse_args(
				$fields_data,
				array(
					'label'       => '',
					'name'        => '',
					'mandatory'   => 'no',
					'width'       => 'full',
					'type'        => 'text',
					'message'     => '',
					'placeholder' => '',
					'sort_order'  => '1',
				)
			);

			// Meta args.
			$meta_args = array(
				'mmp_field_name'        => $fields_data['name'],
				'mmp_field_type'        => $fields_data['type'],
				'mmp_field_placeholder' => $fields_data['placeholder'],
				'mmp_field_mandatory'   => $fields_data['mandatory'],
			);

			// Post args.
			$post_data_arg = array(
				'post_title'  => $fields_data['label'],
				'post_type'   => MMP_Register_Post_Types::MMP_FORM_FIELDS_POSTTYPE,
				'post_status' => $fields_data['status'],
				'ping_status' => 'closed',
				'menu_order'  => $fields_data['sort_order'],
				'meta_input'  => $meta_args,
			);

			// Insert a Post.
			$id = wp_insert_post( $post_data_arg );

			return $id;
		}

		/**
		 * Default Fields Argument
		 */
		public static function default_fields_args() {
			/**
			 * Default args.
			 *
			 * @since 1.0
			 */
			return apply_filters(
				'mmp_default_fields_args',
				array(
					array(
						'label'       => 'Customer Name',
						'name'        => 'mmp_name',
						'type'        => 'text',
						'mandatory'   => 'yes',
						'placeholder' => 'Enter your name',
						'status'      => 'mmp_enabled',
						'sort_order'  => '1',
					),
					array(
						'label'       => 'Email ID',
						'name'        => 'mmp_mail',
						'type'        => 'email',
						'mandatory'   => 'yes',
						'message'     => '',
						'placeholder' => 'Enter your email id',
						'status'      => 'mmp_enabled',
						'sort_order'  => '2',
					),
					array(
						'label'       => 'Description',
						'name'        => 'mmp_description',
						'type'        => 'textarea',
						'mandatory'   => 'yes',
						'placeholder' => 'Enter the details where you have seen this product. Online - Enter the product URL. Offline - Enter the site details',
						'status'      => 'mmp_enabled',
						'sort_order'  => '3',
					),
					array(
						'label'       => 'Lowest Price',
						'name'        => 'mmp_pref_price',
						'type'        => 'number',
						'mandatory'   => 'yes',
						'placeholder' => 'Enter the price',
						'status'      => 'mmp_enabled',
						'sort_order'  => '4',
					),
					array(
						'label'       => 'Mobile Number',
						'name'        => 'mmp_number',
						'type'        => 'text',
						'mandatory'   => 'no',
						'placeholder' => 'Enter your mobile number',
						'status'      => 'mmp_disabled',
						'sort_order'  => '6',
					),
				)
			);
		}

		/**
		 * Get Fields
		 */
		public static function get_fields( $status = false, $order = 'ASC' ) {
			if ( ! $status ) {
				$status = array( 'mmp_disabled', 'mmp_enabled' );
			}

			$post_args = array(
				'post_type'   => MMP_Register_Post_Types::MMP_FORM_FIELDS_POSTTYPE,
				'post_status' => $status,
				'fields'      => 'ids',
				'order'       => $order,
				'numberposts' => -1,
			);

			$posts = get_posts( $post_args );

			if ( ! mmp_check_is_array( $posts ) ) { // default fields not exists
				return false;
			}

			return $posts;
		}
	}

	MMP_Install::init();
}
