<?php
/**
 * Handles the Order Item.
 *
 * @package Lower Price/Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_My_Account_Handler' ) ) {

	/**
	 * Class MMP_My_Account_Handler
	 */
	class MMP_My_Account_Handler {

		/**
		 * Buy Again endpoint.
		 *
		 * @var String
		 */
		public static $mmp_price_list;

		/**
		 * Class Initialization.
		 *
		 * @since 1.0.0
		 */
		public static function init() {

			if ( 'yes' === get_option( 'mmp_general_mmp_enable' ) ) {
				self::$mmp_price_list = get_option( 'mmp_localization_price_list_endpoint', 'mmp_price_list' );
				// Add custom rewrite endpoint.
				add_action( 'init', array( __CLASS__, 'custom_rewrite_endpoint' ) );
				// flush rewrite rules.
				add_action( 'wp_loaded', array( __CLASS__, 'flush_rewrite_rules' ) );
				// Add custom query vars.
				add_filter( 'query_vars', array( __CLASS__, 'custom_query_vars' ), 0 );
				// Customize the myaccount menu title.
				add_filter( 'the_title', array( __CLASS__, 'customize_menu_title' ) );
				// add custom menu to my account.
				add_filter( 'woocommerce_account_menu_items', array( __CLASS__, 'add_custom_myaccount_menu' ), 10, 1 );
				// add custom menu to my account content.
				add_action( 'woocommerce_account_' . self::$mmp_price_list . '_endpoint', array( __CLASS__, 'display_price_list_menu_content' ), 10 );
			}
		}


		/**
		 * Custom rewrite endpoint
		 *
		 * @since 1.0.0
		 */
		public static function custom_rewrite_endpoint() {
			add_rewrite_endpoint( self::$mmp_price_list, EP_ROOT | EP_PAGES );
		}

		/**
		 * Flush Rewrite Rules
		 *
		 * @since 1.0.0
		 */
		public static function flush_rewrite_rules() {
			flush_rewrite_rules();
		}

		/**
		 * Add custom Query variable
		 *
		 * @since 1.0.0
		 * @param String $vars Query Variable.
		 * @return Array
		 */
		public static function custom_query_vars( $vars ) {
			$vars[] = self::$mmp_price_list;

			return $vars;
		}

		/**
		 * Customize the My account menu title
		 */
		public static function customize_menu_title( $title ) {
			global $wp_query;
			if ( is_main_query() && in_the_loop() && is_account_page() && isset( $wp_query->query_vars['mmp_price_list'] ) ) {
				$title = get_option( 'mmp_localization_my_account_label', esc_html__( 'Price List', 'lower-price-for-woocommerce' ) );

				remove_filter( 'the_title', array( __CLASS__, 'customize_menu_title' ) );
			}

			return $title;
		}

		/**
		 * Customize the My account menu title
		 */
		public static function add_custom_myaccount_menu( $menus ) {
			if ( ! is_user_logged_in() ) {
				return $menus;
			}

			$price_list_menu = array( self::$mmp_price_list => get_option( 'mmp_localization_my_account_label', esc_html__( 'Price List', 'lower-price-for-woocommerce' ) ) );
			$menus           = mmp_customize_array_position( $menus, 'orders', $price_list_menu );

			return $menus;
		}

		/**
		 * Display the buy again menu content
		 */
		public static function display_price_list_menu_content() {

			$user       = get_userdata( get_current_user_id() );
			$user_email = $user->user_email;

			$product_ids = mmp_is_get_request_by_value_display( $user_email );

			if ( ! mmp_check_is_array( $product_ids ) ) {
				echo wp_kses_post( '<div>' . esc_html__( 'No List Found', 'lower-price-for-woocommerce' ) . '</div>' );
				return;
			}

			$per_page_count = 5;
			$current_page   = 1;
			/* Calculate Page Count */
			$default_args['posts_per_page'] = $per_page_count;
			$default_args['offset']         = ( $current_page - 1 ) * $per_page_count;
			$page_count                     = ceil( count( $product_ids ) / $per_page_count );
			$args                           = array(
				'user_id'     => get_current_user_id(),
				'product_ids' => array_slice( $product_ids, $default_args['offset'], $per_page_count ),
				'pagination'  => array(
					'page_count'      => $page_count,
					'current_page'    => $current_page,
					'prev_page_count' => ( ( $current_page - 1 ) == 0 ) ? ( $current_page ) : ( $current_page - 1 ),
					'next_page_count' => ( ( $current_page + 1 ) > ( $page_count - 1 ) ) ? ( $current_page ) : ( $current_page + 1 ),
				),
			);

			mmp_get_template( 'mmp-price-list.php', $args );
		}
	}

	MMP_My_Account_Handler::init();
}
