<?php
/**
 * Admin Ajax.
 *
 * @package Lower Price|AJAX
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
if ( ! class_exists( 'MMP_Admin_Ajax' ) ) {

	/**
	 * Main Class.
	 */
	class MMP_Admin_Ajax {

		/**
		 * Class initialization.
		 */
		public static function init() {

			$actions = array(
				'product_search'                 => false,
				'products_and_variations_search' => false,
				'customers_search'               => false,
				'form_fields_sorting'            => false,
				'form_fields_save'               => true,
				'list_pagination'                => true,
				'get_form_fields'                => true,
				'update_viewed_once'             => false,
				'get_custom_field_details'       => false,
				'get_custom_field_data'          => false,
				'add_custom_field'               => false,
				'update_custom_field'            => false,
			);

			foreach ( $actions as $action => $nopriv ) {
				add_action( 'wp_ajax_mmp_' . $action, array( __CLASS__, $action ) );

				if ( $nopriv ) {
					add_action( 'wp_ajax_nopriv_mmp_' . $action, array( __CLASS__, $action ) );
				}
			}
		}

		/**
		 * Product search.
		 */
		public static function product_search() {
			check_ajax_referer( 'mmp-search-nonce', 'mmp_security' );

			try {
				$term = isset( $_GET['term'] ) ? (string) wc_clean( wp_unslash( $_GET['term'] ) ) : '';

				if ( empty( $term ) ) {
					throw new exception( esc_html__( 'No Product(s) found', 'lower-price-for-woocommerce' ) );
				}

				$data_store              = WC_Data_Store::load( 'product' );
				$ids                     = $data_store->search_products( $term, '', false );
				$product_objects         = array_filter( array_map( 'wc_get_product', $ids ), 'wc_products_array_filter_readable' );
				$products                = array();
				$exclude_global_variable = isset($_GET['exclude_global_variable']) ? wc_clean(wp_unslash($_GET['exclude_global_variable'])) : 'no'; // @codingStandardsIgnoreLine.

				foreach ( $product_objects as $product_object ) {
					if ( 'yes' === $exclude_global_variable && $product_object->is_type( 'variable' ) ) {
						continue;
					}

					$products[ $product_object->get_id() ] = rawurldecode( $product_object->get_formatted_name() );
				}
				wp_send_json( $products );
			} catch ( Exception $ex ) {
				wp_die();
			}
		}

		/**
		 * Search for product variations
		 *
		 * @since 1.0
		 * @param String  $term term to search.
		 * @param Boolean $include_variations Include Variations.
		 */
		public static function products_and_variations_search( $term = '', $include_variations = false ) {
			self::product_search( '', true );
		}

		/**
		 * Customer search.
		 */
		public static function customers_search() {
			check_ajax_referer( 'mmp-search-nonce', 'mmp_security' );

			try {
				$term = isset( $_GET['term'] ) ? (string) wc_clean( wp_unslash( $_GET['term'] ) ) : '';

				if ( empty( $term ) ) {
					throw new exception( esc_html__( 'No Customer(s) found', 'lower-price-for-woocommerce' ) );
				}

				$exclude = isset( $_GET['exclude'] ) ? (string) wc_clean( wp_unslash( $_GET['exclude'] ) ) : '';
				$exclude = ! empty( $exclude ) ? array_map( 'intval', explode( ',', $exclude ) ) : array();

				$found_customers = array();

				$customers_query = new WP_User_Query(
					array(
						'fields'         => 'all',
						'orderby'        => 'display_name',
						'search'         => '*' . $term . '*',
						'search_columns' => array( 'ID', 'user_login', 'user_email', 'user_nicename' ),
					)
				);

				$customers = $customers_query->get_results();

				if ( mmp_check_is_array( $customers ) ) {
					foreach ( $customers as $customer ) {
						if ( ! in_array( $customer->ID, $exclude ) ) {
							$found_customers[ $customer->ID ] = $customer->display_name . ' (#' . $customer->ID . ' &ndash; ' . sanitize_email( $customer->user_email ) . ')';
						}
					}
				}

				wp_send_json( $found_customers );
			} catch ( Exception $ex ) {
				wp_die();
			}
		}

		/**
		 * Form Fields Sorting
		 */
		public static function form_fields_sorting() {
			check_ajax_referer( 'mmp-field-sort-nonce', 'mmp_security' );

			try {
				if ( ! isset( $_POST ) || ! isset( $_POST['sort_order'] ) ) {
					throw new exception( __( 'Invalid Request', 'lower-price-for-woocommerce' ) );
				}

				$sort_ids   = array();
				$sort_order = wc_clean( wp_unslash( $_POST['sort_order'] ) );

				foreach ( $sort_order as $menu_order => $post_id ) {
					$custom_field_obj = mmp_get_custom_field( $post_id );

					$custom_field_obj->set_sort_order( $menu_order );
					$custom_field_obj->save();
				}

				wp_send_json_success();
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => $e->getMessage() ) );
			}
		}

		/**
		 * Form Fields Save
		 *
		 * @since 1.0.0
		 */
		public static function form_fields_save() {
			check_ajax_referer( 'mmp-form-nonce', 'mmp_security' );

			try {
				if ( ! isset( $_REQUEST ) ) {
					throw new exception( __( 'Invalid Request', 'lower-price-for-woocommerce' ) );
				}

				$form_data      = array();
				$request        = $_REQUEST;
				$submitted_data = isset( $request['form_data'] ) ? urldecode( $request['form_data'] ) : '';
				parse_str( wc_clean( $submitted_data ), $form_data );

				$product_id  = isset( $form_data['mmp_post_id'] ) ? $form_data['mmp_post_id'] : '';
				$product_obj = wc_get_product( $product_id );

				if ( empty( $product_id ) || ! is_a( $product_obj, 'WC_Product' ) ) {
					throw new exception( __( 'Invalid Product', 'lower-price-for-woocommerce' ) );
				}

				$custom_fields_obj = mmp_get_custom_fields(
					array(
						'status'  => 'enabled',
						'orderby' => 'sort_order',
						'order'   => 'ASC',
					)
				);

				if ( ! $custom_fields_obj->has_custom_field ) {
					throw new exception( __( 'Invalid Data', 'lower-price-for-woocommerce' ) );
				}

				$meta_args = array( 'mmp_viewed_once' => 'no' );

				foreach ( $custom_fields_obj->custom_fields as $custom_field_obj ) {
					$field_set   = ( 'custom' === $custom_field_obj->get_source_from() ) ? isset( $form_data['_custom_fields'][ $custom_field_obj->get_name() ] ) : isset( $form_data[ $custom_field_obj->get_name() ] );
					$field_value = '';

					if ( $field_set ) {
						$field_value = ( 'custom' === $custom_field_obj->get_source_from() ) ? $form_data['_custom_fields'][ $custom_field_obj->get_name() ] : $form_data[ $custom_field_obj->get_name() ];
					}

					if ( 'yes' === $custom_field_obj->get_required() ) {
						if ( ! $field_set || empty( $field_value ) ) {
							/* translators: %s: Field name. */
							throw new exception( sprintf( __( '%s is a required field.', 'lower-price-for-woocommerce' ), esc_attr( $custom_field_obj->get_label() ) ) );
						}
					}

					// Request Count Restriction.
					if ( 'mmp_mail' === $custom_field_obj->get_name() ) {
						if ( ! filter_var( $field_value, FILTER_VALIDATE_EMAIL ) ) {
							throw new exception( esc_html__( 'Invalid Email ID', 'lower-price-for-woocommerce' ) );
						}

						if ( 'yes' === get_option( 'mmp_advanced_overall_product_restriction' ) ) {
							$request_count = mmp_is_get_request_by_value( $field_value );

							// Overall Product Count Restriction.
							if ( '' !== get_option( 'mmp_advanced_pro_restriction_val' ) && ( $request_count ) && count( $request_count ) >= get_option( 'mmp_advanced_pro_restriction_val' ) ) {
								throw new exception( get_option( 'mmp_messages_overall_pro_restriction_msg' ) );
							}
						}

						// Per product check.
						if ( 'yes' === get_option( 'mmp_advanced_per_product_restriction' ) && mmp_is_per_count_check( $field_value, $form_data['mmp_post_id'] ) ) {
							throw new exception( get_option( 'mmp_messages_per_pro_restriction_msg' ) );
						}

						// If accept message.
						if ( mmp_is_per_count_check( $field_value, $form_data['mmp_post_id'], 'accept' ) ) {
							throw new exception( get_option( 'mmp_messages_req_approved_msg' ) );
						}

						// Default already submit message.
						if ( mmp_is_per_count_check( $field_value, $form_data['mmp_post_id'] ) ) {
							throw new exception( get_option( 'mmp_messages_req_submitted_msg' ) );
						}
					}

					// Mobile Number Validation.
					if ( 'mmp_number' === $custom_field_obj->get_name() ) {
						if ( $field_set && ! empty( $field_value ) && ! is_numeric( $field_value ) ) {
							/* translators: %s: Field name. */
							throw new exception( sprintf( '%s - Please Enter Only Numbers', $custom_field_obj->get_label() ) );
						}
					}

					// Price Validation.
					if ( ( 'mmp_pref_price' === $custom_field_obj->get_name() ) ) {
						$field_value = wc_format_decimal( $field_value );

						if ( $field_set && ! empty( $field_value ) ) {
							$product_price = ( 'variable' === $product_obj->get_type() ) ? $product_obj->get_variation_price( 'max' ) : $product_obj->get_price();

							/**
							 * Lower Price Product Original Price
							 *
							 * @since 3.0.0
							 */
							$product_price = apply_filters( 'mmp_product_original_price', $product_price, $product_obj );

							if ( $field_value >= $product_price ) {
								/* translators: %s: Product Price */
								throw new exception( sprintf( esc_html__( 'Enter the price lesser than the product price %s', 'lower-price-for-woocommerce' ), wp_kses_post( wc_price( $product_price ) ) ) );
							}
						}
					}

					if ( 'custom' === $custom_field_obj->get_source_from() ) {
						$meta_args['mmp_custom_fields'][ $custom_field_obj->get_name() ]['label'] = $custom_field_obj->get_label();
						switch ( $custom_field_obj->get_field_type() ) {
							case 'dropdown':
							case 'radio':
							case 'checkbox':
								if ( $field_value ) {
									if ( mmp_check_is_array( $field_value ) ) {
										$meta_args['mmp_custom_fields'][ $custom_field_obj->get_name() ]['value'] = implode( ', ', $field_value );
									} else {
										$meta_args['mmp_custom_fields'][ $custom_field_obj->get_name() ]['value'] = $field_value;
									}
								}
								break;
							default:
								$meta_args['mmp_custom_fields'][ $custom_field_obj->get_name() ]['value'] = $field_value;
								break;

						}
					} else {
						$meta_args[ $custom_field_obj->get_name() ] = $field_value;
					}
				}

				$meta_args['mmp_org_price']      = mmp_get_display_product_price( $product_obj );
				$meta_args['mmp_pro_id']         = $product_id;
				$meta_args['mmp_submitted_date'] = current_time( 'mysql', true );
				$meta_args['mmp_lst_activity']   = current_time( 'mysql', true );
				$meta_args['mmp_request_type']   = get_option( 'mmp_general_mode_of_operaton' );
				$meta_args['mmp_user_type']      = ( is_user_logged_in() ) ? 'user' : 'guest';

				$entry_id = mmp_create_entry(
					$meta_args,
					array(
						'post_status' => 'mmp_new_req',
						'post_author' => ( is_user_logged_in() ) ? get_current_user_id() : 1,
					)
				);

				/**
				 * Admin ack.
				 *
				 * @since 1.0
				 */
				do_action( 'wc_mmp_admin_ack', $entry_id );

				if ( is_user_logged_in() ) {
					/**
					 * Customer ack.
					 *
					 * @since 1.0
					 */
					do_action( 'wc_mmp_customer_ack', $entry_id );
				} else {
					/**
					 * Customer ack guest.
					 *
					 * @since 1.0
					 */
					do_action( 'wc_mmp_customer_ack_guest', $entry_id );
				}

				wp_send_json_success(
					array(
						'success' => true,
						'layout'  => get_thanks_layout(),
					)
				);
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => $e->getMessage() ) );
			}
		}

		/**
		 * Display MMP list based on pagination
		 *
		 * @since 1.0
		 */
		public static function list_pagination() {
			check_ajax_referer( 'mmp-list-pagination', 'mmp_security' );

			try {
				if ( ! isset( $_POST ) || ! isset( $_POST[ 'page_number' ] ) ) { // @codingStandardsIgnoreLine.
					throw new exception( esc_html__( 'Invalid Request', 'lower-price-for-woocommerce' ) );
				}

				// Sanitize post values
				$html         = '';
				$current_page    = ! empty( $_POST[ 'page_number' ] ) ? absint( $_POST[ 'page_number' ] ) : 0 ; // @codingStandardsIgnoreLine.
				$page_url        = ! empty( $_POST[ 'page_url' ] ) ? wc_clean( wp_unslash( $_POST[ 'page_url' ] ) ) : '' ; // @codingStandardsIgnoreLine.
				$per_page     = 5;
				$offset       = ( $current_page - 1 ) * $per_page;

				// Get gift products based on per page count
				if ( isset( $_POST['is_logged_in'] ) && isset( $_POST['current_user'] ) ) {
					$user        = get_userdata( wc_clean( wp_unslash( $_POST['current_user'] ) ) );
					$user_email  = $user->user_email;
					$product_ids = mmp_is_get_request_by_value_display( $user_email );

					$product_ids = array_slice( $product_ids, $offset, $per_page );
					// Get gift products table body content
					$html = mmp_get_template_html(
						'mmp-price-list-data.php',
						array(
							'product_ids' => $product_ids,
							'permalink'   => esc_url( $page_url ),
						)
					);
				}
				wp_send_json_success( array( 'html' => $html ) );
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) );
			}
		}

		/**
		 * Get Form Fields
		 */
		public static function get_form_fields() {
			check_ajax_referer( 'mmp-form-nonce', 'mmp_security' );

			$variation_id = isset( $_POST['mmp_post_id'] ) ? wc_clean( wp_unslash( $_POST['mmp_post_id'] ) ) : '';

			if ( empty( $variation_id ) ) {
				return;
			}

			ob_start();
			MMP_Handler::mmp_form_fields( $variation_id, 'variable' );
			$contents = ob_get_contents();
			ob_end_clean();

			wp_send_json(
				array(
					'success' => true,
					'content' => $contents,
				)
			);
		}

		/**
		 * Update Viewed Once
		 *
		 * @since 2.3
		 */
		public static function update_viewed_once() {
			check_ajax_referer( 'mmp-viewed-once-nonce', 'mmp_security' );

			try {
				if ( ! isset( $_POST ) || ! isset( $_POST[ 'post_id' ] ) ) { // @codingStandardsIgnoreLine.
					throw new exception( esc_html__( 'Invalid Request', 'lower-price-for-woocommerce' ) );
				}

				$post_id = wc_clean( wp_unslash( $_POST['post_id'] ) );
				update_post_meta( $post_id, 'mmp_viewed_once', 'yes' );

				$unseen_count = mmp_get_product_list_count();

				wp_send_json_success( array( 'unseen_count' => $unseen_count ) );
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) );
			}
		}

		/**
		 * Get Custom Field details.
		 *
		 * @since 3.3.0
		 */
		public static function get_custom_field_details() {
			check_admin_referer( 'mmp-preview-custom-field', 'security' );

			if ( ! isset( $_GET['custom_field_id'] ) ) {
				wp_die( -1 );
			}

			$custom_field_obj = mmp_get_custom_field( absint( $_GET['custom_field_id'] ) );

			if ( mmp_is_custom_field( $custom_field_obj ) ) {
				include_once __DIR__ . '/list-tables/class-mmp-admin-list-table-custom-field.php';

				wp_send_json_success( MMP_Admin_List_Table_Custom_Field::custom_field_preview_get_custom_field_details( $custom_field_obj ) );
			}

			wp_die();
		}

		/**
		 * Get Custom Field details.
		 *
		 * @since 3.3.0
		 */
		public static function get_custom_field_data() {
			check_admin_referer( 'mmp-custom-field-data', 'security' );

			if ( ! isset( $_GET['custom_field_id'] ) ) {
				wp_die( -1 );
			}

			$custom_field_obj = mmp_get_custom_field( absint( $_GET['custom_field_id'] ) );

			if ( mmp_is_custom_field( $custom_field_obj ) ) {
				include_once __DIR__ . '/list-tables/class-mmp-admin-list-table-custom-field.php';

				wp_send_json_success( MMP_Admin_List_Table_Custom_Field::custom_field_get_custom_field_data( $custom_field_obj ) );
			}

			wp_die();
		}

		/**
		 * Add Custom Field.
		 *
		 * @since 1.1.0
		 * @throws Exception Invalid Exception.
		 */
		public static function add_custom_field() {
			check_ajax_referer( 'mmp-add-custom-field', 'security' );

			try {
				$request           = $_POST;
				$form_data         = isset( $request['form_data'] ) ? $request['form_data'] : array();
				$status            = isset( $request['status'] ) ? $request['status'] : 'mmp_disabled';
				$options           = array();
				$data_attr         = array();
				$custom_fields_obj = mmp_get_custom_fields( array( 'fields' => 'ids' ) );

				if ( ! mmp_check_is_array( $form_data ) ) {
					throw new Exception( esc_html__( 'Invalid Data', 'lower-price-for-woocommerce' ) );
				}

				if ( ! isset( $form_data['name'] ) ) {
					throw new Exception( esc_html__( 'Field Name Required', 'lower-price-for-woocommerce' ) );
				}

				if ( isset( $form_data['option_attr'] ) && ! empty( $form_data['option_attr'] ) ) {
					$options_data = array_filter( array_map( 'wc_clean', explode( "\n", $form_data['option_attr'] ) ) );

					if ( mmp_check_is_array( $options_data ) ) {
						foreach ( $options_data as $option_data ) {
							$option_data = explode( '|', $option_data );

							if ( mmp_check_is_array( $option_data ) && 2 === count( $option_data ) ) {
								$options[ strtolower( str_replace( ' ', '_', $option_data[0] ) ) ] = $option_data[1];
							}
						}
					}
				}

				if ( isset( $form_data['data_attr'] ) && ! empty( $form_data['data_attr'] ) ) {
					$data_attrs = array_filter( array_map( 'wc_clean', explode( "\n", $form_data['data_attr'] ) ) );

					if ( mmp_check_is_array( $data_attrs ) ) {
						foreach ( $data_attrs as $_data_attr ) {
							$_data_attr = explode( '|', $_data_attr );

							if ( mmp_check_is_array( $_data_attr ) && 2 === count( $_data_attr ) ) {
								$data_attr[ strtolower( str_replace( ' ', '_', $_data_attr[0] ) ) ] = $_data_attr[1];
							}
						}
					}
				}
				$name = strtolower( str_replace( ' ', '_', $form_data['name'] ) );

				if ( in_array( $name, array( 'mmp_name', 'mmp_mail', 'mmp_description', 'mmp_pref_price', 'mmp_number', 'mmp_post_id' ), true ) ) {
					throw new Exception( esc_html__( 'Field Name Already Exist. Please Choose Different Name', 'lower-price-for-woocommerce' ) );
				}

				// Custom Field Object Creation.
				$custom_field_obj = new MMP_Custom_Field();
				$custom_field_obj->set_props(
					array(
						'label'        => $form_data['label'],
						'class_attr'   => $form_data['class_attr'],
						'cols'         => $form_data['cols_attr'],
						'data_attr'    => wp_json_encode( $data_attr ),
						'default'      => $form_data['default_attr'],
						'field_type'   => $form_data['custom_field_type'],
						'id_attr'      => ( isset( $form_data['id_attr'] ) && ! empty( $form_data['id_attr'] ) ) ? strtolower( str_replace( ' ', '_', $form_data['id_attr'] ) ) : '',
						'max'          => $form_data['max_attr'],
						'min'          => $form_data['min_attr'],
						'multiple'     => $form_data['multiple_attr'],
						'name'         => $name,
						'options_attr' => wp_json_encode( $options ),
						'placeholder'  => $form_data['placeholder_attr'],
						'step'         => $form_data['step_attr'],
						'required'     => $form_data['required'],
						'rows'         => $form_data['rows_attr'],
						'tool_tip'     => $form_data['tool_tip_attr'],
						'sort_order'   => $custom_fields_obj->total_custom_fields + 1,
						'source_from'  => 'custom',
						'created_via'  => 'admin',
					)
				);
				$custom_field_obj->save();
				$custom_field_obj->update_status( $status );

				wp_send_json_success( array( 'redirect' => mmp_get_custom_field_page_url() ) );
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => wp_kses_post( $e->getMessage() ) ) );
			}
		}

		/**
		 * Update Custom Field.
		 *
		 * @since 1.1.0
		 * @throws Exception Invalid Exception.
		 */
		public static function update_custom_field() {
			check_ajax_referer( 'mmp-update-custom-field', 'security' );

			try {
				$request   = $_POST;
				$form_data = isset( $request['form_data'] ) ? $request['form_data'] : array();
				$status    = isset( $request['status'] ) ? $request['status'] : 'mmp-disable';
				$options   = array();
				$data_attr = array();

				if ( ! mmp_check_is_array( $form_data ) ) {
					throw new Exception( esc_html__( 'Invalid Data', 'lower-price-for-woocommerce' ) );
				}

				if ( ! isset( $form_data['name'] ) ) {
					throw new Exception( esc_html__( 'Field Name Required', 'lower-price-for-woocommerce' ) );
				}

				$custom_field_id = isset( $form_data['custom_field_id'] ) ? $form_data['custom_field_id'] : '';

				if ( empty( $custom_field_id ) ) {
					throw new Exception( esc_html__( 'Invalid Custom Field', 'lower-price-for-woocommerce' ) );
				}

				$custom_field_obj = mmp_get_custom_field( $custom_field_id );

				if ( ! mmp_is_custom_field( $custom_field_obj ) ) {
					throw new Exception( esc_html__( 'Invalid Custom Field', 'lower-price-for-woocommerce' ) );
				}

				if ( isset( $form_data['option_attr'] ) && ! empty( $form_data['option_attr'] ) ) {
					$options_data = array_filter( array_map( 'strtoupper', array_map( 'wc_clean', explode( "\n", $form_data['option_attr'] ) ) ) );

					if ( mmp_check_is_array( $options_data ) ) {
						foreach ( $options_data as $option_data ) {
							$option_data = explode( '|', $option_data );

							if ( mmp_check_is_array( $option_data ) && 2 === count( $option_data ) ) {
								$options[ strtolower( str_replace( ' ', '_', $option_data[0] ) ) ] = $option_data[1];
							}
						}
					}
				}

				if ( isset( $form_data['data_attr'] ) && ! empty( $form_data['data_attr'] ) ) {
					$data_attrs = array_filter( array_map( 'strtoupper', array_map( 'wc_clean', explode( "\n", $form_data['data_attr'] ) ) ) );

					if ( mmp_check_is_array( $data_attrs ) ) {
						foreach ( $data_attrs as $_data_attr ) {
							$_data_attr = explode( '|', $_data_attr );

							if ( mmp_check_is_array( $_data_attr ) && 2 === count( $_data_attr ) ) {
								$data_attr[ strtolower( str_replace( ' ', '_', $_data_attr[0] ) ) ] = $_data_attr[1];
							}
						}
					}
				}

				$name = strtolower( str_replace( ' ', '_', $form_data['name'] ) );

				if ( ! in_array( $custom_field_obj->get_name(), array( 'mmp_name', 'mmp_mail', 'mmp_description', 'mmp_pref_price', 'mmp_number', 'mmp_post_id' ), true ) ) {
					if ( in_array( $name, array( 'mmp_name', 'mmp_mail', 'mmp_description', 'mmp_pref_price', 'mmp_number' ), true ) ) {
						throw new Exception( esc_html__( 'Field Name Already Exist. Please Choose Different Name', 'lower-price-for-woocommerce' ) );
					}
				}

				$custom_field_obj->set_props(
					array(
						'label'        => $form_data['label'],
						'class_attr'   => $form_data['class_attr'],
						'cols'         => $form_data['cols_attr'],
						'data_attr'    => wp_json_encode( $data_attr ),
						'default'      => $form_data['default_attr'],
						'field_type'   => $form_data['custom_field_type'],
						'id_attr'      => ( isset( $form_data['id_attr'] ) && ! empty( $form_data['id_attr'] ) ) ? strtolower( str_replace( ' ', '_', $form_data['id_attr'] ) ) : '',
						'max'          => $form_data['max_attr'],
						'min'          => $form_data['min_attr'],
						'multiple'     => $form_data['multiple_attr'],
						'name'         => $name,
						'options_attr' => wp_json_encode( $options ),
						'placeholder'  => $form_data['placeholder_attr'],
						'step'         => $form_data['step_attr'],
						'required'     => $form_data['required'],
						'rows'         => $form_data['rows_attr'],
						'tool_tip'     => $form_data['tool_tip_attr'],
						'source_from'  => $form_data['source_from'],
						'created_via'  => 'admin',
					)
				);

				$custom_field_obj->save();
				$custom_field_obj->update_status( $status );

				wp_send_json_success( array( 'redirect' => mmp_get_custom_field_page_url() ) );
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => wp_kses_post( $e->getMessage() ) ) );
			}
		}
	}

	MMP_Admin_Ajax::init();
}
