<?php
/**
 * Custom Field List Table.
 *
 * @package Multi-Vendor/List Table
 * */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

if ( ! class_exists( 'MMP_Admin_List_Table_Custom_Field' ) ) {

	/**
	 * MMP_Admin_List_Table_Custom_Field Class.
	 * */
	class MMP_Admin_List_Table_Custom_Field extends WP_List_Table {

		/**
		 * Per page count
		 *
		 * @var Integer
		 * */
		private $limit = 10;

		/**
		 * Offset
		 *
		 * @var Integer
		 * */
		private $offset;

		/**
		 * Offset
		 *
		 * @var String
		 * */
		private $database;

		/**
		 * Offset
		 *
		 * @var Integer
		 * */
		private $total_items;

		/**
		 * List Slug
		 *
		 * @var String
		 * */
		private $list_slug = 'mmp';

		/**
		 * Order BY
		 *
		 * @var String
		 * */
		private $orderby = 'sort_order';

		/**
		 * Order.
		 *
		 * @var String
		 * */
		private $order = 'ASC';

		/**
		 * Custom Field IDs.
		 *
		 * @var Array
		 * */
		private $custom_field_ids;

		/**
		 * Base URL.
		 *
		 * @var String
		 * */
		private $base_url;

		/**
		 * Current URL.
		 *
		 * @var String
		 * */
		private $current_url;

		/**
		 * Prepares the list of items for displaying.
		 *
		 * @since 3.3.0
		 * */
		public function prepare_items() {
			global $wpdb;
			$this->database = $wpdb;
			$this->base_url = mmp_get_custom_field_page_url();

			$this->prepare_current_url();
			$this->process_bulk_action();
			$this->get_current_pagenum();
			$this->get_current_custom_fields();
			$this->prepare_pagination_args();
			$this->prepare_column_headers();
		}

		/**
		 * Prepare pagination
		 *
		 * @since 3.3.0
		 * */
		private function prepare_pagination_args() {
			$args = array(
				'per_page' => $this->limit,
			);

			if ( $this->total_items ) {
				$args['total_items'] = $this->total_items;
			}

			$this->set_pagination_args( $args );
		}

		/**
		 * Get current page number
		 *
		 * @since 3.3.0
		 * */
		private function get_current_pagenum() {
			$this->offset = $this->limit * ( $this->get_pagenum() - 1 );
		}

		/**
		 * Prepare header columns
		 *
		 * @since 3.3.0
		 * */
		private function prepare_column_headers() {
			$columns  = $this->get_columns();
			$hidden   = $this->get_hidden_columns();
			$sortable = $this->get_sortable_columns();

			$this->_column_headers = array( $columns, $hidden, $sortable );
		}

		/**
		 * Get a list of columns.
		 *
		 * @since 3.3.0
		 * @return Array
		 * */
		public function get_columns() {
			$columns      = array(
				'cb' => '<input type="checkbox" />',
			);
			$keys         = array( 'name', 'type', 'status', 'required', 'date', 'sort' );
			$labels       = array(
				'name'     => esc_html__( 'Label', 'lower-price-for-woocommerce' ),
				'type'     => esc_html__( 'Type', 'lower-price-for-woocommerce' ),
				'status'   => esc_html__( 'Status', 'lower-price-for-woocommerce' ),
				'required' => esc_html__( 'Field Type', 'lower-price-for-woocommerce' ),
				'date'     => esc_html__( 'Date & Time', 'lower-price-for-woocommerce' ),
				'sort'     => esc_html__( 'Sort', 'lower-price-for-woocommerce' ),
			);
			$show_columns = array(
				'cb' => $columns['cb'],
			);

			foreach ( $keys as $key ) {
				$show_columns[ $key ] = ( isset( $labels[ $key ] ) ) ? $labels[ $key ] : '';
			}

			return $show_columns;
		}

		/**
		 * Get a list of hidden columns.
		 *
		 * @since 3.3.0
		 * @return Array
		 * */
		protected function get_hidden_columns() {
			return array();
		}

		/**
		 * Get a list of sortable columns.
		 *
		 * @since 3.3.0
		 * @return array
		 * */
		protected function get_sortable_columns() {
			return array();
		}

		/**
		 * Get current url
		 *
		 * @since 3.3.0
		 * */
		private function prepare_current_url() {
			$pagenum       = $this->get_pagenum();
			$args['paged'] = $pagenum;
			$url           = add_query_arg( $args, $this->base_url );

			$this->current_url = $url;
		}

		/**
		 * Get a list of bulk actions.
		 *
		 * @since 3.3.0
		 * @return Array
		 * */
		protected function get_bulk_actions() {
			$action = array(
				'delete'  => esc_html__( 'Delete', 'lower-price-for-woocommerce' ),
				'enable'  => esc_html__( 'Enable', 'lower-price-for-woocommerce' ),
				'disable' => esc_html__( 'Disable', 'lower-price-for-woocommerce' ),
			);

			/**
			 * Get bulk actions.
			 *
			 * @since 3.3.0
			 */
			$action = apply_filters( $this->list_slug . '_bulk_actions', $action );

			return $action;
		}

		/**
		 * Processes the bulk action.
		 *
		 * @since 3.3.0
		 * */
		public function process_bulk_action() {
			if ( ! isset( $_REQUEST['_mmp_nonce'] ) ) {
				return;
			}

			$nonce = sanitize_key( wp_unslash( $_REQUEST['_mmp_nonce'] ) );

			if ( ! wp_verify_nonce( $nonce, 'mmp-search_custom_field' ) ) {
				return;
			}

			$action = $this->current_action();
			$ids    = isset( $_REQUEST['id'] ) ? wc_clean( wp_unslash( $_REQUEST['id'] ) ) : array();
			$ids    = ! is_array( $ids ) ? explode( ',', $ids ) : $ids;

			if ( ! mmp_check_is_array( $ids ) ) {
				return;
			}

			foreach ( $ids as $id ) {
				$custom_field_obj = mmp_get_custom_field( $id );

				switch ( $action ) {
					case 'delete':
						if ( mmp_is_custom_field( $custom_field_obj ) && 'custom' === $custom_field_obj->get_source_from() ) {
							$custom_field_obj->delete();
						}
						break;
					case 'enable':
						if ( 'enabled' !== $custom_field_obj->get_status() ) {
							$custom_field_obj->update_status( 'enabled' );
						}
						break;
					case 'disable':
						if ( 'disabled' !== $custom_field_obj->get_status() ) {
							$custom_field_obj->update_status( 'disabled' );
						}
						break;
				}
			}

			wp_safe_redirect( $this->current_url );
			exit();
		}

		/**
		 * Display the list of views available on this table.
		 *
		 * @since 3.3.0
		 * @return Array
		 * */
		protected function get_views() {
			$args         = array();
			$views        = array();
			$status       = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : '';
			$status_array = mmp_get_custom_field_statuses();
			$status_array = array( 'all' => esc_html__( 'All', 'lower-price-for-woocommerce' ) ) + $status_array;

			foreach ( $status_array as $status_name => $status_label ) {
				$status_count = $this->get_item_count_for_status( $status_name );

				if ( ! $status_count ) {
					continue;
				}

				$args['status'] = $status_name;
				$label          = $status_label . ' (' . $status_count . ')';
				$class          = array( strtolower( $status_name ) );

				if ( $status === $status_name || ( 'all' === $status_name && empty( $status ) ) ) {
					$class[] = 'current';
				}

				$search_term = isset( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : '';

				if ( $search_term ) {
					$args['s'] = $search_term;
				}

				$views[ $status_name ] = $this->get_edit_link( $args, $label, implode( ' ', $class ) );
			}

			return $views;
		}

		/**
		 * Get the edit link for status.
		 *
		 * @since 3.3.0
		 * @param Array  $args Arguments.
		 * @param String $label Label.
		 * @param String $class Class.
		 * @return String
		 * */
		private function get_edit_link( $args, $label, $class = '' ) {
			$url        = add_query_arg( $args, $this->base_url );
			$class_html = '';

			if ( ! empty( $class ) ) {
				/* translators: %s: Class */
				$class_html = sprintf( 'class="%s"', esc_attr( $class ) );
			}
			/* translators: %1$s: URL  %2$s: Class %3$s: Link Name */
			return sprintf( '<a href="%1$s" %2$s>%3$s</a>', esc_url( $url ), $class_html, $label );
		}

		/**
		 * Prepare the CB column data.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field_obj Custom Field object.
		 * @return HTML
		 * */
		protected function column_cb( $custom_field_obj ) {
			return sprintf( '<input class="mmp-custom-field-cb" type="checkbox" name="id[]" value="%s" />', $custom_field_obj->get_id() );
		}

		/**
		 * Prepare the each column data.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field_obj Custom Field object.
		 * @param String           $column_name Name of the column.
		 * @return mixed
		 * */
		protected function column_default( $custom_field_obj, $column_name ) {
			switch ( $column_name ) {
				case 'name':
					$actions = array();

					/* translators: %s: Custom Field ID */
					$actions ['id']   = sprintf( esc_html__( 'ID: %s', 'lower-price-for-woocommerce' ), $custom_field_obj->get_id() );
					$actions ['edit'] = '<a class="mmp-custom-field-edit-btn" data-custom_field_id="' . absint( $custom_field_obj->get_id() ) . '" href="' . esc_url( $custom_field_obj->get_admin_edit_url() ) . '">' . esc_html__( 'Edit', 'lower-price-for-woocommerce' ) . '</a>';

					if ( mmp_is_custom_field( $custom_field_obj ) ) {
						echo '<a href="#" class="mmp-custom-field-preview" data-custom_field_id="' . absint( $custom_field_obj->get_id() ) . '" title="' . esc_html__( 'Preview', 'lower-price-for-woocommerce' ) . '">' . esc_html__( 'Preview', 'lower-price-for-woocommerce' ) . '</a>';
						$views = '<a class="mmp-custom-field-edit-btn row-title" data-custom_field_id="' . absint( $custom_field_obj->get_id() ) . '" href="' . esc_url( $custom_field_obj->get_admin_edit_url() ) . '" class="custom-field-view"><strong>' . esc_html( $custom_field_obj->get_label() ) . '</strong></a> <br/> <div class="row-actions">';
					} else {
						$views = esc_html__( 'Not found', 'lower-price-for-woocommerce' ) . ' <br/> <div class="row-actions">';
					}

					if ( 'custom' === $custom_field_obj->get_source_from() ) {
						$actions ['delete'] = '<a class="mmp-delete-custom-field mmp-delete" href="' . esc_url(
							add_query_arg(
								array(
									'action'     => 'delete',
									'id'         => $custom_field_obj->get_id(),
									'_mmp_nonce' => wp_create_nonce( 'mmp-search_custom_field' ),
								),
								$this->current_url
							)
						) . '">' . esc_html__( 'Delete', 'lower-price-for-woocommerce' ) . '</a>';
					}

					end( $actions );

					$last_key = key( $actions );

					foreach ( $actions as $key => $action ) {
						$views .= '<span class="' . $key . '">' . $action;

						if ( $last_key === $key ) {
							$views .= '</span>';
							break;
						}

						$views .= ' | </span>';
					}

					return $views . '</div>';
				case 'type':
					echo wp_kses_post( $custom_field_obj->get_field_type() );
					break;
				case 'status':
					printf( '<mark class="mmp-post-status %s"><span>%s</span></mark>', esc_attr( sanitize_html_class( 'status-' . $custom_field_obj->get_status() ) ), esc_html( mmp_get_custom_field_status_name( $custom_field_obj->get_status() ) ) );
					break;
				case 'date':
					echo esc_html( $custom_field_obj->get_date_created()->date_i18n( wc_date_format() . ' ' . wc_time_format() ) );
					break;
				case 'required':
					echo 'yes' === $custom_field_obj->get_required() ? esc_html__( 'Mandatory', 'lower-price-for-woocommerce' ) : esc_html__( 'Optional', 'lower-price-for-woocommerce' );
					break;
				case 'sort':
					echo '<input type="hidden" class="mmp-sortable-order" value="' . esc_attr( $custom_field_obj->get_id() ) . '">';
					echo wp_kses_post( "<div class='mmp_post_sort_handle'><img src=" . esc_attr( MMP_PLUGIN_URL . '\assets\images\drag-icon.png' ) . '></div>' );
					break;
			}
		}

		/**
		 * Get the current page items.
		 *
		 * @since 3.3.0
		 * */
		private function get_current_custom_fields() {
			$request           = $_REQUEST;
			$status            = isset( $request['status'] ) ? sanitize_text_field( wp_unslash( $request['status'] ) ) : '';
			$search_term       = isset( $request['s'] ) ? sanitize_text_field( wp_unslash( $request['s'] ) ) : '';
			$args              = array(
				'status'  => ( $status && 'all' !== $status ) ? $status : array_keys( mmp_get_custom_field_statuses() ),
				'fields'  => 'objects',
				's'       => $search_term,
				'orderby' => isset( $request['orderby'] ) ? sanitize_text_field( wp_unslash( $request['orderby'] ) ) : $this->orderby,
				'order'   => isset( $request['order'] ) ? sanitize_text_field( wp_unslash( $request['order'] ) ) : $this->order,
				'limit'   => $this->limit,
				'page'    => $this->get_pagenum(),
			);
			$custom_fields_obj = mmp_get_custom_fields( $args );
			$this->items       = $custom_fields_obj->custom_fields;
			$this->total_items = $custom_fields_obj->total_custom_fields;
		}

		/**
		 * Get the custom field count for the status.
		 *
		 * @since 3.3.0
		 * @param String $status Status.
		 * @return Integer
		 * */
		private function get_item_count_for_status( $status = '' ) {
			if ( empty( $status ) ) {
				$status = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : '';
			}

			if ( 'all' === $status || '' === $status ) {
				$status = array_keys( mmp_get_custom_field_statuses() );
			}

			$args = array(
				'status' => $status,
				'fields' => 'ids',
			);

			$search_term = isset( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : '';

			if ( $search_term ) {
				$args['s'] = $search_term;
			}

			$custom_fields_obj = mmp_get_custom_fields( $args );

			return $custom_fields_obj->total_custom_fields;
		}

		/**
		 * Prepare the Custom Field IDs.
		 *
		 * @since 3.3.0
		 * */
		private function prepare_custom_field_ids() {
			$all_custom_field_ids = mmp_get_custom_fields(
				array(
					'fields' => 'ids',
				)
			);

			/**
			 * Custom Field ids in list table.
			 *
			 * @since 3.3.0
			 */
			$this->custom_field_ids = apply_filters( 'mmp_admin_list_table_custom_field_ids', $all_custom_field_ids );
		}

		/**
		 * Get custom field details to send to the ajax endpoint for previews.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field_obj Custom Field object.
		 * @return Array
		 */
		public static function custom_field_get_custom_field_data( $custom_field_obj ) {
			if ( ! mmp_is_custom_field( $custom_field_obj ) ) {
				return array();
			}

			$option_attr_str = '';
			$option_attr     = json_decode( $custom_field_obj->get_options_attr() );

			if ( ! empty( $option_attr ) ) {
				end( $option_attr );

				$last_key = key( $option_attr );

				foreach ( $option_attr as $key => $value ) {
					$option_attr_str .= $key . '|' . $value;

					if ( $last_key !== $key ) {
						$option_attr_str .= '
';
					}
				}
			}

			$data_attr_str = '';
			$data_attr     = json_decode( $custom_field_obj->get_data_attr() );

			if ( ! empty( $data_attr ) ) {
				end( $data_attr );

				$last_key = key( $data_attr );

				foreach ( $data_attr as $key => $value ) {
					$data_attr_str .= $key . '|' . $value;

					if ( $last_key !== $key ) {
						$data_attr_str .= '
';
					}
				}
			}

			/**
			 * Custom Field Preview details
			 *
			 * @since 3.3.0
			 */
			return apply_filters(
				'mmp_admin_custom_field_preview_get_custom_field_details',
				array(
					'custom_field_id' => $custom_field_obj->get_id(),
					'field_data'      => $custom_field_obj->get_data(),
					'options_attr'    => $option_attr_str,
					'data_attr'       => $data_attr_str,
					'status'          => $custom_field_obj->get_status(),
					'status_name'     => mmp_get_custom_field_status_name( $custom_field_obj->get_status() ),
				),
				$custom_field_obj
			);
		}

		/**
		 * Get custom field details to send to the ajax endpoint for previews.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field_obj Custom Field object.
		 * @return Array
		 */
		public static function custom_field_preview_get_custom_field_details( $custom_field_obj ) {
			if ( ! mmp_is_custom_field( $custom_field_obj ) ) {
				return array();
			}

			/**
			 * Custom Field Preview details
			 *
			 * @since 3.3.0
			 */
			return apply_filters(
				'mmp_admin_custom_field_preview_get_custom_field_details',
				array(
					'custom_field_id' => $custom_field_obj->get_id(),
					'field_data'      => $custom_field_obj->get_data(),
					'item_html'       => self::get_custom_field_preview_item_html( $custom_field_obj ),
					'actions_html'    => self::get_custom_field_preview_actions_html( $custom_field_obj ),
					'status'          => $custom_field_obj->get_status(),
					'status_name'     => mmp_get_custom_field_status_name( $custom_field_obj->get_status() ),
				),
				$custom_field_obj
			);
		}

		/**
		 * Get items to display in the preview as HTML.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field_obj Custom Field object.
		 * @return String
		 */
		public static function get_custom_field_preview_item_html( $custom_field_obj ) {
			ob_start();

			mmp_output_custom_fields( $custom_field_obj, $custom_field_obj->get_default() );

			return ob_get_clean();
		}

		/**
		 * Get items to display in the preview as HTML.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field_obj Custom Field object.
		 * @return String
		 */
		public static function get_custom_field_preview_actions_html( $custom_field_obj ) {
			$actions = array();

			/**
			 * Custom Field Actions
			 *
			 * @since 3.3.0
			 */
			$actions = apply_filters( 'mmp_admin_custom_field_preview_actions', $actions, $custom_field_obj );

			return wc_render_action_buttons( $actions );
		}
	}

}
