<?php
/**
 * Autoload
 *
 * @since 3.3.0
 * @package Multi-Vendor for WooCommerce\Admin\Meta Boxes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MVR_Autoload' ) ) {
	/**
	 * Multi-Vendor for WooCommerce Autoloader.
	 */
	class MMP_Autoload {

		/**
		 * Path to the includes directory.
		 *
		 * @var String
		 */
		private $include_path = '';

		/**
		 * Path to the includes directory.
		 *
		 * @var String
		 */
		private $file = '';

		/**
		 * Construct MMP_Autoload
		 */
		public function __construct() {
			if ( function_exists( '__autoload' ) ) {
				spl_autoload_register( '__autoload' );
			}

			$this->include_path = MMP_ABSPATH . 'inc/';

			spl_autoload_register( array( $this, 'autoload' ) );
		}

		/**
		 * Take a class name and turn it into a file name.
		 *
		 * @since 3.3.0
		 * @param  String $class_name Class name.
		 * @return String
		 */
		private function get_file_name_from_class( $class_name ) {
			return 'class-' . str_replace( '_', '-', $class_name ) . '.php';
		}

		/**
		 * Include a class file.
		 *
		 * @since 3.3.0
		 * @return Boolean Successful or not.
		 */
		private function load_file() {
			if ( ! empty( $this->file ) && is_readable( $this->file ) ) {
				include_once $this->file;
				return true;
			}

			return false;
		}

		/**
		 * Auto-load our classes on demand to reduce memory consumption.
		 *
		 * @since 3.3.0
		 * @param string $class_name Class name.
		 */
		public function autoload( $class_name ) {
			$class_name = strtolower( $class_name );

			// Make sure our classes are going to load.
			if ( 0 !== strpos( $class_name, 'mmp_' ) ) {
				return;
			}

			$file = $this->get_file_name_from_class( $class_name ); // Retrieve file name from class name.
			$path = $this->include_path;

			if ( false !== strpos( $class_name, '_data_store' ) ) {
				$path = $this->include_path . 'data-stores/';
			} elseif ( false !== strpos( $class_name, 'meta_box_' ) ) {
				$path = $this->include_path . 'admin/meta-boxes/';
			} elseif ( false !== strpos( $class_name, 'compatible_' ) ) {
				$path = $this->include_path . 'compatibles/';
			} elseif ( false !== strpos( $class_name, '_shortcode_' ) ) {
				$path = $this->include_path . 'shortcodes/';
			} elseif ( in_array( $class_name, array_keys( mmp()->entity ), true ) ) {
				$path = $this->include_path . 'entity/';
			}

			// Include a class file.
			if ( is_readable( $path . $file ) ) {
				$this->file = $path . $file;
				$this->load_file();
			}
		}
	}

	new MMP_Autoload();
}
