<?php

/**
 * Initialize the Plugin.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Install' ) ) {

	/**
	 * Main MMP_Install Class.
	 * */
	class MMP_Install {
		/**
		 * Data Base Table.
		 *
		 * @var Array
		 * */
		public static $db_tables = array(
			'custom_field',
		);

		/**
		 * Hook in methods.
		 *
		 * @since 1.0.0
		 * */
		public static function init() {
			add_action( 'admin_init', array( __CLASS__, 'check_db_tables' ) );
			add_action( 'woocommerce_init', array( __CLASS__, 'check_version' ) );
			add_filter( 'plugin_action_links_' . MMP_PLUGIN_SLUG, array( __CLASS__, 'settings_link' ) );
		}

		/**
		 * Check Database Tables are there.
		 *
		 * @since 3.3.0
		 * */
		public static function check_db_tables() {
			global $wpdb;
			$wpdb_ref       = &$wpdb;
			$missing_tables = array();

			foreach ( self::$db_tables as $db_table ) {
				$db_table = "{$wpdb_ref->prefix}mmp_" . $db_table;
				$result   = $wpdb_ref->get_var( $wpdb_ref->prepare( 'SHOW TABLES LIKE %s', $db_table ) );

				if ( $result !== $db_table ) {
					$missing_tables[] = $db_table;
				}
			}

			if ( mmp_check_is_array( $missing_tables ) ) {
				/* translators: %s: Table Name */
				$message = sprintf( esc_html__( 'Lower Price or Best Price Guarantee for WooCommerce plugin will not work properly some of Tables are missing in the database %s. Please verify the database.', 'lower-price-for-woocommerce' ), esc_html( implode( ', ', $missing_tables ) ) );

				MMP_Settings::add_error( $message );
			}
		}

		/**
		 * Check current version of the plugin is updated when activating plugin, if not run updater.
		 */
		public static function check_version() {
			if ( version_compare( get_option( 'mmp_version', '1.0.0' ), MMP_VERSION, '!=' ) ) {
				self::install();

				/**
				 * Plugin updated.
				 *
				 * @since 1.0.0
				 */
				do_action( 'mmp_updated' );
			}
		}

		/**
		 * Install
		 */
		public static function install() {
			self::create_tables(); // Create Tables.
			self::verify_base_tables(); // Verify base tables.
			self::set_default_values(); // default values
			self::set_form_fields(); // Set Default Form Field.
			self::update_form_fields(); // Update Default Form Field.
			self::update_version(); // Update Version.
		}

		/**
		 * Creating Multi Vendor Tables
		 *
		 * @since 3.3.0
		 */
		private static function create_tables() {
			global $wpdb;

			$wpdb->hide_errors();

			require_once ABSPATH . 'wp-admin/includes/upgrade.php';

			$db_delta_result = dbDelta( self::get_schema() );

			return $db_delta_result;
		}

		/**
		 * Table Schema
		 *
		 * @since 3.3.0
		 */
		private static function get_schema() {
			global $wpdb;

			$collate = '';

			if ( $wpdb->has_cap( 'collation' ) ) {
				$collate = $wpdb->get_charset_collate();
			}

			$tables = " CREATE TABLE `{$wpdb->prefix}mmp_custom_field` (
						`ID` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
						`alt` varchar(20),
						`class_attr` longtext,
						`cols` varchar(20),
						`data_attr` longtext,
						`default` varchar(20),
						`field_type` varchar(20) NOT NULL,
						`for_attr` varchar(20),
						`height` varchar(20),
						`href` varchar(20),
						`id_attr` longtext,
						`label` longtext NOT NULL,
						`max` varchar(20),
						`min` varchar(20),
						`multiple` varchar(20),
						`name` varchar(20) NOT NULL,
						`options_attr` longtext NOT NULL,
						`placeholder` longtext,
						`src` varchar(20),
						`step` varchar(20),
						`readonly` varchar(20),
						`rel` varchar(20),
						`required` varchar(20),
						`rows` varchar(20),
						`target` varchar(20),
						`tool_tip` longtext,
						`date_created` datetime,
						`date_created_gmt` datetime,
						`status` varchar(20) NOT NULL,
						`created_via` varchar(20) NOT NULL,
						`source_from` varchar(20),
						`date_modified` datetime,
						`date_modified_gmt` datetime,
						`sort_order` bigint(20),
						`parent_id` bigint(20),
						`version` varchar(20) NOT NULL,
						PRIMARY KEY  (`ID`)
						) $collate;
						";

			return $tables;
		}

		/**
		 * Check if all the base tables are present.
		 *
		 * @since 3.3.0
		 * @param Boolean $execute Whether to execute get_schema queries as well.
		 * @return Array List of queries.
		 */
		public static function verify_base_tables( $execute = false ) {
			if ( $execute ) {
				self::create_tables();
			}

			$missing_tables = self::get_missing_tables( self::get_schema() );

			if ( 0 < count( $missing_tables ) ) {
				$message  = esc_html__( 'Verifying database... One or more tables are still missing: ', 'lower-price-for-woocommerce' );
				$message .= implode( ', ', $missing_tables );

				MMP_Settings::add_error( $message );
			}

			return $missing_tables;
		}

		/**
		 * Get Missing Table
		 *
		 * @since 3.3.0
		 * @param String $creation_queries The output from the execution of dbDelta.
		 * @return Array.
		 */
		public static function get_missing_tables( $creation_queries ) {
			global $wpdb;

			$suppress_errors = $wpdb->suppress_errors( true );

			require_once ABSPATH . 'wp-admin/includes/upgrade.php';

			$output = dbDelta( $creation_queries, false );

			$wpdb->suppress_errors( $suppress_errors );

			$parsed_output = self::parse_dbdelta_output( $output );

			return $parsed_output['missing_tables'];
		}

		/**
		 * Parses the output given by dbdelta and returns information about it.
		 *
		 * @since 3.3.0
		 * @param Array $dbdelta_output The output from the execution of dbdelta.
		 * @return Array.
		 */
		public static function parse_dbdelta_output( array $dbdelta_output ) {
			$created_tables = array();

			foreach ( $dbdelta_output as $table_name => $result ) {
				if ( "Created table $table_name" !== $result ) {
					$created_tables[] = $table_name;
				}
			}

			return array( 'missing_tables' => array_filter( $created_tables ) );
		}

		/**
		 *  Settings link.
		 */
		public static function settings_link( $links ) {
			$setting_page_link = '<a href="' . mmp_get_settings_page_url() . '">' . esc_html__( 'Settings', 'lower-price-for-woocommerce' ) . '</a>';

			array_unshift( $links, $setting_page_link );

			return $links;
		}

		/**
		 *  Set settings default values
		 */
		public static function set_default_values() {
			if ( ! class_exists( 'MMP_Settings' ) ) {
				include_once MMP_PLUGIN_PATH . '/inc/admin/menu/class-mmp-settings.php';
			}

			// default for settings
			$settings = MMP_Settings::get_settings_pages();

			foreach ( $settings as $setting_key => $value ) {

				$settings_array = $value->get_settings( $setting_key );

				foreach ( $settings_array as $value ) {
					if ( isset( $value['default'] ) && isset( $value['id'] ) ) {
						if ( get_option( $value['id'] ) === false ) {
							add_option( $value['id'], $value['default'] );
						}
					}
				}
			}
		}

		/**
		 * Update current version.
		 */
		private static function update_version() {
			update_option( 'mmp_version', MMP_VERSION );
		}

		/**
		 * Register Default Fields
		 *
		 * @since 3.3.0
		 */
		public static function set_form_fields() {
			if ( self::get_fields() ) {
				return true; // default fields is exists.
			}

			$fields = self::default_fields_args();

			if ( ! mmp_check_is_array( $fields ) ) {
				return false;
			}

			foreach ( $fields as $field ) {
				self::create( $field ); // register a field as post
			}
		}

		/**
		 * Create a new field
		 *
		 * @since 3.3.0
		 */
		public static function create( $form_data ) {
			$options   = array();
			$form_data = wp_parse_args(
				$form_data,
				array(
					'alt'           => '',
					'class_attr'    => '',
					'cols'          => '',
					'data_attr'     => array(),
					'default'       => '',
					'field_type'    => '',
					'for_attr'      => '',
					'height'        => '',
					'href'          => '',
					'id_attr'       => '',
					'label'         => '',
					'max'           => '',
					'min'           => '',
					'multiple'      => '',
					'name'          => '',
					'options_attr'  => array(),
					'placeholder'   => '',
					'src'           => '',
					'step'          => '',
					'readonly'      => '',
					'rel'           => '',
					'required'      => '',
					'rows'          => '',
					'target'        => '',
					'tool_tip'      => '',
					'date_created'  => null,
					'date_modified' => null,
					'status'        => '',
					'source_from'   => '',
					'created_via'   => '',
					'version'       => '',
					'sort_order'    => '1',
				)
			);

			// Custom Field Object Creation.
			$custom_field_obj = new MMP_Custom_Field();
			$custom_field_obj->set_props(
				array(
					'label'        => $form_data['label'],
					'class_attr'   => $form_data['class_attr'],
					'cols'         => $form_data['cols'],
					'default'      => $form_data['default'],
					'field_type'   => $form_data['field_type'],
					'id_attr'      => ( isset( $form_data['id_attr'] ) && ! empty( $form_data['id_attr'] ) ) ? strtolower( str_replace( ' ', '_', $form_data['id_attr'] ) ) : '',
					'max'          => $form_data['max'],
					'min'          => $form_data['min'],
					'multiple'     => $form_data['multiple'],
					'name'         => strtolower( str_replace( ' ', '_', $form_data['name'] ) ),
					'options_attr' => wp_json_encode( $options ),
					'placeholder'  => $form_data['placeholder'],
					'step'         => $form_data['step'],
					'required'     => $form_data['required'],
					'rows'         => $form_data['rows'],
					'tool_tip'     => $form_data['tool_tip'],
					'source_from'  => 'default',
					'created_via'  => 'admin',
				)
			);
			$custom_field_obj->save();
			$custom_field_obj->update_status( $form_data['status'] );

			return $custom_field_obj->get_id();
		}

		/**
		 * Default Fields Argument
		 */
		public static function default_fields_args() {
			/**
			 * Default Form Fields.
			 *
			 * @since 3.3.0
			 */
			return apply_filters(
				'mmp_default_fields_args',
				array(
					array(
						'label'       => esc_html__( 'Customer Name', 'lower-price-for-woocommerce' ),
						'name'        => 'mmp_name',
						'class_attr'  => 'mmp-form-fields mmp-name',
						'field_type'  => 'text',
						'required'    => 'yes',
						'placeholder' => esc_html__( 'Enter your name', 'lower-price-for-woocommerce' ),
						'status'      => 'enabled',
						'sort_order'  => 1,
					),
					array(
						'label'       => esc_html__( 'Email ID', 'lower-price-for-woocommerce' ),
						'name'        => 'mmp_mail',
						'class_attr'  => 'mmp-form-fields mmp-email',
						'field_type'  => 'email',
						'required'    => 'yes',
						'message'     => '',
						'placeholder' => esc_html__( 'Enter your email id', 'lower-price-for-woocommerce' ),
						'status'      => 'enabled',
						'sort_order'  => 2,
					),
					array(
						'label'       => esc_html__( 'Description', 'lower-price-for-woocommerce' ),
						'name'        => 'mmp_description',
						'class_attr'  => 'mmp-form-fields mmp-description',
						'field_type'  => 'textarea',
						'required'    => 'yes',
						'placeholder' => esc_html__( 'Enter the details where you have seen this product. Online - Enter the product URL. Offline - Enter the site details', 'lower-price-for-woocommerce' ),
						'status'      => 'enabled',
						'sort_order'  => 3,
					),
					array(
						'label'       => esc_html__( 'Lowest Price', 'lower-price-for-woocommerce' ),
						'name'        => 'mmp_pref_price',
						'class_attr'  => 'mmp-form-fields mmp-price',
						'field_type'  => 'text',
						'required'    => 'yes',
						'placeholder' => esc_html__( 'Enter the price', 'lower-price-for-woocommerce' ),
						'status'      => 'enabled',
						'sort_order'  => 4,
					),
					array(
						'label'       => esc_html__( 'Mobile Number', 'lower-price-for-woocommerce' ),
						'name'        => 'mmp_number',
						'class_attr'  => 'mmp-form-fields mmp-mobile',
						'field_type'  => 'text',
						'required'    => 'no',
						'placeholder' => esc_html__( 'Enter your mobile number', 'lower-price-for-woocommerce' ),
						'status'      => 'disabled',
						'sort_order'  => 5,
					),
				)
			);
		}

		/**
		 * Get Fields
		 *
		 * @since 3.3.0
		 * @return Object|Boolean
		 */
		public static function get_fields() {
			$custom_fields = mmp_get_custom_fields(
				array(
					'fields'      => 'ids',
					'source_from' => 'default',
				)
			);

			return $custom_fields->has_custom_field ? $custom_fields->custom_fields : false;
		}

		/**
		 * Get Fields
		 *
		 * @since 3.3.0
		 * @return Object|Boolean
		 */
		public static function update_form_fields() {
			$price_fields = mmp_get_custom_fields(
				array(
					'source_from' => 'default',
					'name'        => 'mmp_pref_price',
					'field_type'  => 'number',
				)
			);

			if ( $price_fields->has_custom_field ) {
				foreach ( $price_fields->custom_fields as $price_field ) {
					$price_field->set_props( array( 'field_type' => 'text' ) );
					$price_field->save();
				}
			}
		}
	}

	MMP_Install::init();
}
