<?php

defined( 'ABSPATH' ) || exit;

/**
 * Emails class.
 *
 * @class WC_MMP_Emails
 * @package Class
 */
class WC_MMP_Emails {

	/**
	 * Email notification classes
	 *
	 * @var WC_Email[]
	 */
	protected $emails = array();

	/**
	 * Available email notification classes to load
	 *
	 * @var WC_Email::id => WC_Email class
	 */
	protected $email_classes = array(
		'admin-ack'               => 'WC_MMP_Admin_Ack',
		'customer-ack'            => 'WC_MMP_Customer_Ack',
		'customer-ack-guest'      => 'WC_MMP_Customer_Ack_Guest',
		'customer-sending-coupon' => 'WC_MMP_Customer_Sending_Coupon',
		'customer-coupon-apply'   => 'WC_MMP_Customer_Coupon_Apply',
		'customer-rejected'       => 'WC_MMP_Customer_Rejected',
		'customer-rejected-guest' => 'WC_MMP_Customer_Rejected_Guest',
	);

	/**
	 * The single instance of the class
	 *
	 * @var WC_MMP_Emails
	 */
	protected static $_instance = null;

	/**
	 * Main WC_MMP_Emails Instance.
	 * Ensures only one instance of WC_MMP_Emails is loaded or can be loaded.
	 *
	 * @return WC_MMP_Emails Main instance
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Init the email class hooks in all emails that can be sent.
	 */
	public function init() {
		add_filter( 'woocommerce_email_classes', array( $this, 'add_email_classes' ) );
		add_filter( 'woocommerce_template_directory', array( $this, 'set_template_directory' ), 10, 2 );
		add_action( 'admin_init', array( $this, 'hide_plain_text_template' ) );

		self::init_notifications();
	}

	/**
	 * Hook in all our emails to notify.
	 */
	public static function init_notifications() {
		/**
		 * Email actions.
		 *
		 * @since 1.0
		 */
		$email_actions = apply_filters(
			'wc_mmp_email_actions',
			array(
				'wc_mmp_admin_ack',
				'wc_mmp_customer_ack',
				'wc_mmp_coupon_apply',
				'wc_mmp_customer_rejected',
				'wc_mmp_customer_sending_coupon',
				'wc_mmp_customer_ack_guest',
				'wc_mmp_customer_rejected_guest',
			)
		);

		foreach ( $email_actions as $action ) {
			add_action( $action, array( __CLASS__, 'send_notification' ), 10, 10 );
		}
	}

	/**
	 * Init the WC mailer instance and call the notifications for the current filter.
	 *
	 * @param array $args Email args (default: []).
	 */
	public static function send_notification( $args = array() ) {
		try {
			WC()->mailer();
			$args = func_get_args();
			/**
			 * Notification.
			 *
			 * @since 1.0
			 */
			do_action_ref_array( current_filter() . '_notification', $args );
		} catch ( Exception $e ) {
			return;
		}
	}

	/**
	 * Load our email classes.
	 *
	 * @param array $emails
	 */
	public function add_email_classes( $emails ) {
		if ( ! empty( $this->emails ) ) {
			return $emails + $this->emails;
		}

		// Include email classes.
		include_once 'abstracts/abstract-wc-mmp-email.php';

		foreach ( $this->email_classes as $id => $class ) {
			$file_name = 'class-' . strtolower( str_replace( '_', '-', $class ) );
			$path      = MMP_PLUGIN_PATH . "/inc/emails/{$file_name}.php";

			if ( is_readable( $path ) ) {
				$this->emails[ $class ] = include $path;
			}
		}

		return $emails + $this->emails;
	}

	/**
	 * Hide Template - Plain text
	 */
	public function hide_plain_text_template() {
		if ( ! isset( $_GET['section'] ) ) {
			return;
		}

		WC()->mailer();

		if ( in_array( $_GET['section'], array_map( 'strtolower', array_keys( $this->emails ) ) ) ) {
			echo '<style>div.template_plain{display:none;}</style>';
		}
	}

	/**
	 * Set our email templates directory.
	 *
	 * @param string $template_directory
	 * @param string $template
	 * @return string
	 */
	public function set_template_directory( $template_directory, $template ) {
		$templates = array_map( array( $this, 'get_template_name' ), array_keys( $this->email_classes ) );

		foreach ( $templates as $name ) {
			if ( in_array(
				$template,
				array(
					"emails/{$name}.php",
					"emails/plain/{$name}.php",
				)
			)
			) {
				return untrailingslashit( WC_MMP_BASENAME_DIR );
			}
		}

		return $template_directory;
	}

	/**
	 * Get the template name from email ID
	 */
	public function get_template_name( $id ) {
		return str_replace( '_', '-', $id );
	}

	/**
	 * Are emails available ?
	 *
	 * @return WC_Email class
	 */
	public function available() {
		WC()->mailer();

		return ! empty( $this->emails ) ? true : false;
	}

	/**
	 * Return the email class
	 *
	 * @param string $id
	 * @return null|WC_Email class name
	 */
	public function get_email_class( $id ) {
		$id = strtolower( $id );

		if ( false !== stripos( $id, WC_MMP_PREFIX ) ) {
			$id = ltrim( $id, WC_MMP_PREFIX );
		}

		return isset( $this->email_classes[ $id ] ) ? $this->email_classes[ $id ] : null;
	}

	/**
	 * Return the emails
	 *
	 * @return WC_Email[]
	 */
	public function get_emails() {
		WC()->mailer();

		return $this->emails;
	}

	/**
	 * Return the email
	 *
	 * @param string $id
	 * @return WC_Email
	 */
	public function get_email( $id ) {
		WC()->mailer();

		$class = $this->get_email_class( $id );

		return isset( $this->emails[ $class ] ) ? $this->emails[ $class ] : null;
	}
}
