<?php
/**
 * Custom Field Data Store
 *
 * @package Lower Price
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Custom_Field_Data_Store_CPT' ) ) {

	/**
	 * Custom Field Data Store CPT
	 *
	 * @class MMP_Custom_Field_Data_Store_CPT
	 * @package Class
	 */
	class MMP_Custom_Field_Data_Store_CPT extends WC_Data_Store_WP implements WC_Object_Data_Store_Interface {

		/**
		 * Data stored props.
		 *
		 * @var Array
		 */
		protected $internal_props = array(
			'alt',
			'class_attr',
			'cols',
			'data_attr',
			'default',
			'field_type',
			'for_attr',
			'height',
			'href',
			'id_attr',
			'label',
			'max',
			'min',
			'multiple',
			'name',
			'options_attr',
			'placeholder',
			'src',
			'step',
			'readonly',
			'rel',
			'required',
			'rows',
			'target',
			'tool_tip',
			'status',
			'date_created',
			'date_modified',
			'source_from',
			'sort_order',
			'created_via',
			'version',
		);

		/*
		|--------------------------------------------------------------------------
		| CRUD Methods
		|--------------------------------------------------------------------------
		 */

		/**
		 * Method to create a new ID in the database from the new changes.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field Custom Field object.
		 */
		public function create( &$custom_field ) {
			global $wpdb;

			$wpdb_ref = &$wpdb;

			$custom_field->set_version( MMP_VERSION );
			$custom_field->set_date_created( time() );

			$table  = "{$wpdb->prefix}mmp_custom_field";
			$format = array(
				'alt'               => '%s',
				'class_attr'        => '%s',
				'cols'              => '%s',
				'data_attr'         => '%s',
				'default'           => '%s',
				'field_type'        => '%s',
				'for_attr'          => '%s',
				'height'            => '%s',
				'href'              => '%s',
				'id_attr'           => '%s',
				'label'             => '%s',
				'max'               => '%s',
				'min'               => '%s',
				'multiple'          => '%s',
				'name'              => '%s',
				'options_attr'      => '%s',
				'placeholder'       => '%s',
				'src'               => '%s',
				'step'              => '%s',
				'readonly'          => '%s',
				'rel'               => '%s',
				'required'          => '%s',
				'rows'              => '%s',
				'target'            => '%s',
				'tool_tip'          => '%s',
				'status'            => '%s',
				'date_created'      => '%s',
				'date_created_gmt'  => '%s',
				'date_modified'     => '%s',
				'date_modified_gmt' => '%s',
				'source_from'       => '%s',
				'sort_order'        => '%d',
				'created_via'       => '%s',
				'version'           => '%s',
			);
			$data   = array(
				'alt'               => $custom_field->get_alt(),
				'class_attr'        => $custom_field->get_class_attr(),
				'cols'              => $custom_field->get_cols(),
				'data_attr'         => $custom_field->get_data_attr(),
				'default'           => $custom_field->get_default(),
				'field_type'        => $custom_field->get_field_type(),
				'for_attr'          => $custom_field->get_for_attr(),
				'height'            => $custom_field->get_height(),
				'href'              => $custom_field->get_href(),
				'id_attr'           => $custom_field->get_id_attr(),
				'label'             => $custom_field->get_label(),
				'max'               => $custom_field->get_max(),
				'min'               => $custom_field->get_min(),
				'multiple'          => $custom_field->get_multiple(),
				'name'              => $custom_field->get_name(),
				'options_attr'      => $custom_field->get_options_attr(),
				'placeholder'       => $custom_field->get_placeholder(),
				'src'               => $custom_field->get_src(),
				'step'              => $custom_field->get_step(),
				'readonly'          => $custom_field->get_readonly(),
				'rel'               => $custom_field->get_rel(),
				'required'          => $custom_field->get_required(),
				'rows'              => $custom_field->get_rows(),
				'target'            => $custom_field->get_target(),
				'tool_tip'          => $custom_field->get_tool_tip(),
				'status'            => $this->get_custom_field_status( $custom_field ),
				'date_created'      => current_time( 'mysql' ),
				'date_created_gmt'  => current_time( 'mysql', 1 ),
				'date_modified'     => current_time( 'mysql' ),
				'date_modified_gmt' => current_time( 'mysql', 1 ),
				'source_from'       => $custom_field->get_source_from(),
				'sort_order'        => $custom_field->get_sort_order(),
				'created_via'       => $custom_field->get_created_via(),
				'version'           => $custom_field->get_version(),
			);

			$id = mmp_insert_row_query( $table, $data, $format );

			if ( $id && ! is_wp_error( $id ) ) {
				$custom_field->set_id( $id );
				$custom_field->apply_changes();

				/**
				 * New Custom Field Hook
				 *
				 * @since 3.3.0
				 */
				do_action( 'mmp_new_custom_field', $custom_field->get_id(), $custom_field );
			}
		}

		/**
		 * Method to read data from the database.
		 *
		 * @since 1.1.0
		 * @param MMP_Custom_Field $custom_field Custom Field object.
		 * @throws Exception Invalid Post.
		 */
		public function read( &$custom_field ) {
			$custom_field->set_defaults();

			if ( ! $custom_field->get_id() ) {
				throw new Exception( esc_html__( 'Invalid Field.', 'lower-price-for-woocommerce' ) );
			}

			global $wpdb;

			$wpdb_ref = &$wpdb;
			$data     = $wpdb_ref->get_row(
				$wpdb_ref->prepare( "SELECT * from {$wpdb->prefix}mmp_custom_field WHERE ID=%d", $custom_field->get_id() )
			);

			foreach ( $this->internal_props as $prop ) {
				$setter = "set_$prop";

				if ( is_callable( array( $custom_field, $setter ) ) && is_object( $data ) && property_exists( $data, $prop ) ) {
					$custom_field->{$setter}( $data->$prop );
				}
			}

			$custom_field->set_object_read( true );
		}

		/**
		 * Method to update changes in the database.
		 *
		 * @since 1.1.0
		 * @param MMP_Custom_Field $custom_field Custom Field object.
		 */
		public function update( &$custom_field ) {
			global $wpdb;
			$wpdb_ref = &$wpdb;
			$custom_field->set_version( MMP_VERSION );

			if ( ! $custom_field->get_date_created( 'edit' ) ) {
				$custom_field->set_date_created( time() );
			}

			$format = array(
				'alt'               => '%s',
				'class_attr'        => '%s',
				'cols'              => '%s',
				'data_attr'         => '%s',
				'default'           => '%s',
				'field_type'        => '%s',
				'for_attr'          => '%s',
				'height'            => '%s',
				'href'              => '%s',
				'id_attr'           => '%s',
				'label'             => '%s',
				'max'               => '%s',
				'min'               => '%s',
				'multiple'          => '%s',
				'name'              => '%s',
				'options_attr'      => '%s',
				'placeholder'       => '%s',
				'src'               => '%s',
				'step'              => '%s',
				'readonly'          => '%s',
				'rel'               => '%s',
				'required'          => '%s',
				'rows'              => '%s',
				'target'            => '%s',
				'tool_tip'          => '%s',
				'status'            => '%s',
				'date_modified'     => '%s',
				'date_modified_gmt' => '%s',
				'source_from'       => '%s',
				'sort_order'        => '%d',
				'created_via'       => '%s',
				'version'           => '%s',
			);
			$data   = array(
				'alt'               => $custom_field->get_alt(),
				'class_attr'        => $custom_field->get_class_attr(),
				'cols'              => $custom_field->get_cols(),
				'data_attr'         => $custom_field->get_data_attr(),
				'default'           => $custom_field->get_default(),
				'field_type'        => $custom_field->get_field_type(),
				'for_attr'          => $custom_field->get_for_attr(),
				'height'            => $custom_field->get_height(),
				'href'              => $custom_field->get_href(),
				'id_attr'           => $custom_field->get_id_attr(),
				'label'             => $custom_field->get_label(),
				'max'               => $custom_field->get_max(),
				'min'               => $custom_field->get_min(),
				'multiple'          => $custom_field->get_multiple(),
				'name'              => $custom_field->get_name(),
				'options_attr'      => $custom_field->get_options_attr(),
				'placeholder'       => $custom_field->get_placeholder(),
				'src'               => $custom_field->get_src(),
				'step'              => $custom_field->get_step(),
				'readonly'          => $custom_field->get_readonly(),
				'rel'               => $custom_field->get_rel(),
				'required'          => $custom_field->get_required(),
				'rows'              => $custom_field->get_rows(),
				'target'            => $custom_field->get_target(),
				'tool_tip'          => $custom_field->get_tool_tip(),
				'status'            => $this->get_custom_field_status( $custom_field ),
				'date_modified'     => current_time( 'mysql' ),
				'date_modified_gmt' => current_time( 'mysql', 1 ),
				'source_from'       => $custom_field->get_source_from(),
				'sort_order'        => $custom_field->get_sort_order(),
				'created_via'       => $custom_field->get_created_via(),
				'version'           => $custom_field->get_version(),
			);
			$table  = "{$wpdb_ref->prefix}mmp_custom_field";
			$where  = '`ID` = ' . $custom_field->get_id();
			$id     = mmp_update_row_query( $table, $format, $data, $where );

			if ( $id && ! is_wp_error( $id ) ) {
				$custom_field->apply_changes();

				/**
				 * Update Custom Field Hook
				 *
				 * @since 3.3.0
				 */
				do_action( 'mmp_update_custom_field', $custom_field->get_id(), $custom_field );
			}
		}

		/**
		 * Delete an object, set the ID to 0.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field Custom Field object.
		 * @param Array            $args Array of args to pass to the delete method.
		 * @return Boolean
		 */
		public function delete( &$custom_field, $args = array() ) {
			$id = $custom_field->get_id();

			if ( ! $id ) {
				return false;
			}

			global $wpdb;

			$wpdb_ref = &$wpdb;
			$result   = $wpdb_ref->delete( "{$wpdb->prefix}mmp_custom_field", array( 'ID' => $custom_field->get_id() ) );

			if ( ! $result ) {
				return false;
			}

			$custom_field->set_id( 0 );

			/**
			 * Delete Custom Field.
			 *
			 * @since 3.3.0
			 */
			do_action( 'mmp_delete_custom_field', $id );

			return true;
		}

		/*
		|------------------------|
		|   Additional Methods   |
		|------------------------|
		 */

		/**
		 * Get the status to object.
		 *
		 * @since 3.3.0
		 * @param MMP_Custom_Field $custom_field object.
		 * @return String
		 */
		protected function get_custom_field_status( $custom_field ) {
			$custom_field_status = $custom_field->get_status( 'edit' );

			if ( ! $custom_field_status ) {
				/**
				 * Default Custom Field Status.
				 *
				 * @since 3.3.0
				 */
				$custom_field_status = apply_filters( 'mmp_default_custom_field_status', 'disabled' );
			}

			if ( in_array( 'mmp_' . $custom_field_status, $custom_field->get_valid_statuses(), true ) ) {
				$custom_field_status = 'mmp_' . $custom_field_status;
			}

			return $custom_field_status;
		}
	}
}
