<?php
/**
 * Custom Field Data.
 *
 * @package Lower Price
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Custom_Field' ) ) {
	/**
	 * Order
	 *
	 * @class MMP_Custom_Field
	 * @package Class
	 */
	class MMP_Custom_Field extends WC_Data {

		/**
		 * Stores data about status changes so relevant hooks can be fired.
		 *
		 * @var Boolean|Array
		 */
		protected $status_transition = false;

		/**
		 * Order Data array.
		 *
		 * @var Array
		 */
		protected $data = array(
			'alt'           => '',
			'class_attr'    => '',
			'cols'          => '',
			'data_attr'     => array(),
			'default'       => '',
			'field_type'    => '',
			'for_attr'      => '',
			'height'        => '',
			'href'          => '',
			'id_attr'       => '',
			'label'         => '',
			'max'           => '',
			'min'           => '',
			'multiple'      => '',
			'name'          => '',
			'options_attr'  => array(),
			'placeholder'   => '',
			'src'           => '',
			'step'          => '',
			'readonly'      => '',
			'rel'           => '',
			'required'      => '',
			'rows'          => '',
			'target'        => '',
			'tool_tip'      => '',
			'date_created'  => null,
			'date_modified' => null,
			'status'        => '',
			'source_from'   => '',
			'sort_order'    => '',
			'created_via'   => '',
			'version'       => '',
		);

		/**
		 * Which data store to load.
		 *
		 * @var String
		 */
		protected $data_store_name = 'mmp_custom_field';

		/**
		 * Get the custom field if ID is passed, otherwise the custom field is new and empty.
		 *
		 * @since 3.3.0
		 * @param  int|object|MMP_Custom_Field $custom_field Custom Field to read.
		 */
		public function __construct( $custom_field = 0 ) {
			parent::__construct( $custom_field );

			if ( is_numeric( $custom_field ) && $custom_field > 0 ) {
				$this->set_id( $custom_field );
			} elseif ( $custom_field instanceof self ) {
				$this->set_id( $custom_field->get_id() );
			} elseif ( ! empty( $custom_field->ID ) ) {
				$this->set_id( $custom_field->ID );
			} else {
				$this->set_object_read( true );
			}

			$this->data_store = WC_Data_Store::load( $this->data_store_name );

			if ( $this->get_id() > 0 ) {
				$this->data_store->read( $this );
			}
		}

		/**
		 * Get all valid statuses for this Custom Field
		 *
		 * @since 3.3.0
		 * @return Array Internal status keys e.g. 'mmp_enabled'
		 */
		public function get_valid_statuses() {
			return array_keys( mmp_get_custom_field_statuses() );
		}

		/**
		 * Handle the status transition.
		 *
		 * @since 3.3.0
		 */
		protected function status_transition() {
			$status_transition = $this->status_transition;

			// Reset status transition variable.
			$this->status_transition = false;

			if ( $status_transition ) {
				try {
					/**
					 * Custom Field status updated to.
					 *
					 * @since 3.3.0
					 */
					do_action( 'mmp_custom_field_status_' . $status_transition['to'], $this );

					if ( ! empty( $status_transition['from'] ) && $status_transition['from'] !== $status_transition['to'] ) {
						/* translators: 1: old status 2: new status */
						$transition_note = sprintf( __( 'Status changed from <b>%1$s</b> to <b>%2$s</b>.', 'lower-price-for-woocommerce' ), mmp_get_custom_field_status_name( $status_transition['from'] ), mmp_get_custom_field_status_name( $status_transition['to'] ) );

						/**
						 * Custom Field status updated from and to.
						 *
						 * @since 3.3.0
						 */
						do_action( 'mmp_custom_field_status_' . $status_transition['from'] . '_to_' . $status_transition['to'], $this );

						/**
						 * Custom Field status changed.
						 *
						 * @since 3.3.0
						 */
						do_action( 'mmp_custom_field_status_changed', $status_transition['from'], $status_transition['to'], $this );

						/**
						 * Custom Field status updated.
						 *
						 * @since 3.3.0
						 */
						do_action( 'mmp_custom_field_status_updated', $status_transition['to'], $status_transition['manual'], $this );
					}
				} catch ( Exception $e ) {
					$this->handle_exception( $e, sprintf( 'Status transition of custom field #%d errored!', $this->get_id() ) );
				}
			}
		}

		/**
		 * Updates status of custom field immediately.
		 *
		 * @since 3.3.0
		 * @uses MMP_Custom_Field::set_status()
		 * @param String $new_status    Status to change the custom field to. No internal mmp_ prefix is required.
		 * @return Boolean
		 */
		public function update_status( $new_status ) {
			if ( ! $this->get_id() ) { // Custom field must exist.
				return false;
			}

			try {
				$this->set_status( $new_status );
				$this->save();
			} catch ( Exception $e ) {
				$logger = wc_get_logger();
				$logger->error(
					sprintf(
						'Error updating status for custom field #%d',
						$this->get_id()
					),
					array(
						'custom_field' => $this,
						'error'        => $e,
					)
				);

				return false;
			}

			return true;
		}

		/**
		 * Log an error about this custom field is exception is encountered.
		 *
		 * @since 3.3.0
		 * @param Exception $e Exception object.
		 * @param String    $message Message regarding exception thrown.
		 */
		protected function handle_exception( $e, $message = 'Error' ) {
			wc_get_logger()->error(
				$message,
				array(
					'custom_field' => $this,
					'error'        => $e,
				)
			);
		}

		/*
		|--------------------------------------------------------------------------
		| Conditionals
		|--------------------------------------------------------------------------
		|
		| Checks if a condition is true or false.
		|
		 */

		/**
		 * Checks the commission status against a passed in status.
		 *
		 * @since 3.3.0
		 * @param Array|String $status Status to check.
		 * @return Boolean
		 */
		public function has_status( $status ) {
			/**
			 * Has Status.
			 *
			 * @since 3.3.0
			 */
			return apply_filters( 'mmp_custom_field_has_status', ( is_array( $status ) && in_array( $this->get_status(), $status, true ) ) || $this->get_status() === $status, $this, $status );
		}

		/*
		|--------------------------------------------------------------------------
		| URLs and Endpoints
		|--------------------------------------------------------------------------
		 */

		/**
		 * Get's the URL to edit the commission in the backend.
		 *
		 * @since 3.3.0
		 * @return String
		 */
		public function get_admin_edit_url() {
			/**
			 * Edit Commission URL.
			 *
			 * @since 3.3.0
			 */
			return apply_filters(
				'mmp_get_admin_edit_custom_field_url',
				wp_nonce_url(
					mmp_get_custom_field_page_url(
						array(
							'action' => 'view',
							'id'     => $this->get_id(),
						)
					),
					'mmp-edit-custom-field',
					'_mmp_nonce'
				),
				$this
			);
		}

		/*
		|--------------------------------------------------------------------------
		| Getters
		|--------------------------------------------------------------------------
		 */

		/**
		 * Get alt.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_alt( $context = 'view' ) {
			return $this->get_prop( 'alt', $context );
		}

		/**
		 * Get class attr.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_class_attr( $context = 'view' ) {
			return $this->get_prop( 'class_attr', $context );
		}

		/**
		 * Get cols.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_cols( $context = 'view' ) {
			return $this->get_prop( 'cols', $context );
		}

		/**
		 * Get data Attr.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_data_attr( $context = 'view' ) {
			return $this->get_prop( 'data_attr', $context );
		}

		/**
		 * Get Default.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_default( $context = 'view' ) {
			return $this->get_prop( 'default', $context );
		}

		/**
		 * Get for attribute.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_for_attr( $context = 'view' ) {
			return $this->get_prop( 'for_attr', $context );
		}

		/**
		 * Get height.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_height( $context = 'view' ) {
			return $this->get_prop( 'height', $context );
		}

		/**
		 * Get href.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_href( $context = 'view' ) {
			return $this->get_prop( 'href', $context );
		}

		/**
		 * Get id attr.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_id_attr( $context = 'view' ) {
			return $this->get_prop( 'id_attr', $context );
		}

		/**
		 * Get label.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_label( $context = 'view' ) {
			return $this->get_prop( 'label', $context );
		}

		/**
		 * Get max.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_max( $context = 'view' ) {
			return $this->get_prop( 'max', $context );
		}

		/**
		 * Get min.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_min( $context = 'view' ) {
			return $this->get_prop( 'min', $context );
		}

		/**
		 * Get multiple.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_multiple( $context = 'view' ) {
			return $this->get_prop( 'multiple', $context );
		}

		/**
		 * Get name.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_name( $context = 'view' ) {
			return $this->get_prop( 'name', $context );
		}

		/**
		 * Get options.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_options_attr( $context = 'view' ) {
			return $this->get_prop( 'options_attr', $context );
		}

		/**
		 * Get placeholder.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_placeholder( $context = 'view' ) {
			return $this->get_prop( 'placeholder', $context );
		}

		/**
		 * Get src.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_src( $context = 'view' ) {
			return $this->get_prop( 'src', $context );
		}

		/**
		 * Get step.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_step( $context = 'view' ) {
			return $this->get_prop( 'step', $context );
		}

		/**
		 * Get readonly.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_readonly( $context = 'view' ) {
			return $this->get_prop( 'readonly', $context );
		}

		/**
		 * Get rel.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_rel( $context = 'view' ) {
			return $this->get_prop( 'rel', $context );
		}

		/**
		 * Get required.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_required( $context = 'view' ) {
			return $this->get_prop( 'required', $context );
		}

		/**
		 * Get rows.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_rows( $context = 'view' ) {
			return $this->get_prop( 'rows', $context );
		}

		/**
		 * Get target.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_target( $context = 'view' ) {
			return $this->get_prop( 'target', $context );
		}

		/**
		 * Get tool tip.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_tool_tip( $context = 'view' ) {
			return $this->get_prop( 'tool_tip', $context );
		}

		/**
		 * Get field_type.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_field_type( $context = 'view' ) {
			return $this->get_prop( 'field_type', $context );
		}

		/**
		 * Get version.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return String
		 */
		public function get_version( $context = 'view' ) {
			return $this->get_prop( 'version', $context );
		}

		/**
		 * Get date created.
		 *
		 * @since 3.3.0
		 * @param  String $context View or edit context.
		 * @return WC_DateTime|NULL object if the date is set or null if there is no date.
		 */
		public function get_date_created( $context = 'view' ) {
			return $this->get_prop( 'date_created', $context );
		}

		/**
		 * Get date modified.
		 *
		 * @since 3.3.0
		 * @param String $context View or edit context.
		 * @return WC_DateTime|NULL object if the date is set or null if there is no date.
		 */
		public function get_date_modified( $context = 'view' ) {
			return $this->get_prop( 'date_modified', $context );
		}

		/**
		 * Get Source from.
		 *
		 * @since 3.3.0
		 * @param String $context View or edit context.
		 * @return String|NULL String if the source from is set or null if there is no source from.
		 */
		public function get_source_from( $context = 'view' ) {
			return $this->get_prop( 'source_from', $context );
		}

		/**
		 * Get sort order.
		 *
		 * @since 3.3.0
		 * @param String $context View or edit context.
		 * @return String|NULL String if the sort order is set or null if there is no sort order.
		 */
		public function get_sort_order( $context = 'view' ) {
			return $this->get_prop( 'sort_order', $context );
		}

		/**
		 * Get created via.
		 *
		 * @since 3.3.0
		 * @param String $context View or edit context.
		 * @return String|NULL String if the created via is set or null if there is no created via.
		 */
		public function get_created_via( $context = 'view' ) {
			return $this->get_prop( 'created_via', $context );
		}

		/**
		 * Return the statuses without mmp- internal prefix.
		 *
		 * @since 3.3.0
		 * @param String $context View or edit context.
		 * @return String
		 */
		public function get_status( $context = 'view' ) {
			$status = $this->get_prop( 'status', $context );

			if ( empty( $status ) && 'view' === $context ) {
				/**
				 * Default Status
				 *
				 * @since 3.3.0
				 */
				$status = apply_filters( 'mmp_default_custom_field_status', 'disabled' );
			}

			return $status;
		}

		/*
		|--------------------------------------------------------------------------
		| Setters
		|--------------------------------------------------------------------------
		|
		| Functions for setting vendor data. These should not update anything in the
		| database itself and should only change what is stored in the class
		| object.
		 */

		/**
		 * Set status.
		 *
		 * @since 3.3.0
		 * @param String  $new_status Status to change the vendor to. No internal mmp- prefix is required.
		 * @param Boolean $manual_update Is this a manual user payment status change?.
		 * @return Array details of change
		 */
		public function set_status( $new_status, $manual_update = false ) {
			$old_status        = $this->get_status();
			$new_status        = mmp_trim_post_status( $new_status );
			$status_exceptions = array( 'auto-draft', 'trash' );

			// If setting the status, ensure it's set to a valid status.
			if ( true === $this->object_read ) {
				// Only allow valid new status.
				if ( ! in_array( 'mmp_' . $new_status, $this->get_valid_statuses(), true ) && ! in_array( $new_status, $status_exceptions, true ) ) {
					$new_status = 'disabled';
				}

				// If the old status is set but unknown (e.g. draft) assume its pending for action usage.
				if ( $old_status && ! in_array( 'mmp_' . $old_status, $this->get_valid_statuses(), true ) && ! in_array( $old_status, $status_exceptions, true ) ) {
					$old_status = 'disabled';
				}

				if ( ! empty( $old_status ) && $old_status !== $new_status ) {
					$this->status_transition = array(
						'from'   => ! empty( $this->status_transition['from'] ) ? $this->status_transition['from'] : $old_status,
						'to'     => $new_status,
						'note'   => '',
						'manual' => (bool) $manual_update,
					);

					if ( $manual_update ) {
						/**
						 * When custom field status has been manually edited.
						 *
						 * @since 3.3.0
						 */
						do_action( 'mmp_custom_field_edit_status', $this->get_id(), $new_status );
					}
				}
			}

			$this->set_prop( 'status', $new_status );

			return array(
				'from' => $old_status,
				'to'   => $new_status,
			);
		}

		/**
		 * Set version.
		 *
		 * @since 3.3.0
		 * @param String $value Value to set.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_version( $value ) {
			$this->set_prop( 'version', $value );
		}

		/**
		 * Set date created.
		 *
		 * @since 3.3.0
		 * @param String|Integer|Null $date UTC timestamp, or ISO 8601 DateTime. If the DateTime string has no timezone or offset, WordPress site timezone will be assumed. Null if there is no date.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_date_created( $date = null ) {
			$this->set_date_prop( 'date_created', $date );
		}

		/**
		 * Set date modified.
		 *
		 * @since 3.3.0
		 * @param String|Integer|Null $date UTC timestamp, or ISO 8601 DateTime. If the DateTime string has no timezone or offset, WordPress site timezone will be assumed. Null if there is no date.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_date_modified( $date = null ) {
			$this->set_date_prop( 'date_modified', $date );
		}

		/**
		 * Set source from.
		 *
		 * @since 3.3.0
		 * @param String $value Source from.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_source_from( $value = null ) {
			$this->set_prop( 'source_from', $value );
		}

		/**
		 * Set created via.
		 *
		 * @since 3.3.0
		 * @param String $value sort order.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_sort_order( $value = null ) {
			$this->set_prop( 'sort_order', $value );
		}

		/**
		 * Set created via.
		 *
		 * @since 3.3.0
		 * @param String $value created via.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_created_via( $value = null ) {
			$this->set_prop( 'created_via', $value );
		}

		/**
		 * Set alt.
		 *
		 * @since 3.3.0
		 * @param String $value alt.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_alt( $value = null ) {
			$this->set_prop( 'alt', $value );
		}

		/**
		 * Set class attribute.
		 *
		 * @since 3.3.0
		 * @param String $value class attribute.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_class_attr( $value = null ) {
			$this->set_prop( 'class_attr', $value );
		}

		/**
		 * Set cols.
		 *
		 * @since 3.3.0
		 * @param String $value cols.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_cols( $value = null ) {
			$this->set_prop( 'cols', $value );
		}

		/**
		 * Set data attribute.
		 *
		 * @since 3.3.0
		 * @param String $value data attribute.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_data_attr( $value = null ) {
			$this->set_prop( 'data_attr', $value );
		}

		/**
		 * Set default.
		 *
		 * @since 3.3.0
		 * @param String $value default.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_default( $value = null ) {
			$this->set_prop( 'default', $value );
		}

		/**
		 * Set for attribute.
		 *
		 * @since 3.3.0
		 * @param String $value for attribute.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_for_attr( $value = null ) {
			$this->set_prop( 'for_attr', $value );
		}

		/**
		 * Set height.
		 *
		 * @since 3.3.0
		 * @param String $value height.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_height( $value = null ) {
			$this->set_prop( 'height', $value );
		}

		/**
		 * Set href.
		 *
		 * @since 3.3.0
		 * @param String $value href.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_href( $value = null ) {
			$this->set_prop( 'href', $value );
		}

		/**
		 * Set id attribute.
		 *
		 * @since 3.3.0
		 * @param String $value id attribute.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_id_attr( $value = null ) {
			$this->set_prop( 'id_attr', $value );
		}

		/**
		 * Set label.
		 *
		 * @since 3.3.0
		 * @param String $value label.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_label( $value = null ) {
			$this->set_prop( 'label', $value );
		}

		/**
		 * Set max.
		 *
		 * @since 3.3.0
		 * @param String $value max.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_max( $value = null ) {
			$this->set_prop( 'max', $value );
		}

		/**
		 * Set min.
		 *
		 * @since 3.3.0
		 * @param String $value min.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_min( $value = null ) {
			$this->set_prop( 'min', $value );
		}

		/**
		 * Set multiple.
		 *
		 * @since 3.3.0
		 * @param String $value multiple.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_multiple( $value = null ) {
			$this->set_prop( 'multiple', $value );
		}

		/**
		 * Set name.
		 *
		 * @since 3.3.0
		 * @param String $value name.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_name( $value = null ) {
			$this->set_prop( 'name', $value );
		}

		/**
		 * Set options.
		 *
		 * @since 3.3.0
		 * @param String $value options.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_options_attr( $value = null ) {
			$this->set_prop( 'options_attr', $value );
		}

		/**
		 * Set placeholder.
		 *
		 * @since 3.3.0
		 * @param String $value placeholder.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_placeholder( $value = null ) {
			$this->set_prop( 'placeholder', $value );
		}

		/**
		 * Set src.
		 *
		 * @since 3.3.0
		 * @param String $value src.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_src( $value = null ) {
			$this->set_prop( 'src', $value );
		}

		/**
		 * Set step.
		 *
		 * @since 3.3.0
		 * @param String $value step.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_step( $value = null ) {
			$this->set_prop( 'step', $value );
		}

		/**
		 * Set readonly.
		 *
		 * @since 3.3.0
		 * @param String $value readonly.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_readonly( $value = null ) {
			$this->set_prop( 'readonly', $value );
		}

		/**
		 * Set rel.
		 *
		 * @since 3.3.0
		 * @param String $value rel.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_rel( $value = null ) {
			$this->set_prop( 'rel', $value );
		}

		/**
		 * Set required.
		 *
		 * @since 3.3.0
		 * @param String $value required.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_required( $value = null ) {
			$this->set_prop( 'required', $value );
		}

		/**
		 * Set rows.
		 *
		 * @since 3.3.0
		 * @param String $value rows.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_rows( $value = null ) {
			$this->set_prop( 'rows', $value );
		}

		/**
		 * Set target.
		 *
		 * @since 3.3.0
		 * @param String $value target.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_target( $value = null ) {
			$this->set_prop( 'target', $value );
		}

		/**
		 * Set tool tip.
		 *
		 * @since 3.3.0
		 * @param String $value tool tip.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_tool_tip( $value = null ) {
			$this->set_prop( 'tool_tip', $value );
		}

		/**
		 * Set field type.
		 *
		 * @since 3.3.0
		 * @param String $value field type.
		 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
		 */
		public function set_field_type( $value = null ) {
			$this->set_prop( 'field_type', $value );
		}
	}
}
