<?php
/**
 * MMP Handling
 *
 * @package Lower Price for WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Handler' ) ) {

	/**
	 * Main MMP_Handler Class.
	 * */
	class MMP_Handler {

		/**
		 * Class Init function
		 *
		 * @since 1.0.0
		 */
		public static function init() {
			$single_product_position = get_option( 'mmp_general_form_position', 'woocommerce_before_add_to_cart_form' );
			$single_product_position = ( 'custom' === $single_product_position ) ? get_option( 'mmp_general_form_position_custom_hook', 'woocommerce_before_add_to_cart_form' ) : $single_product_position;
			$single_product_priority = get_option( 'mmp_general_form_position_hook_priority', 10 );

			$variable_product_position = get_option( 'mmp_general_form_position_variable', 'woocommerce_single_variation' );
			$variable_product_position = ( 'custom' === $variable_product_position ) ? get_option( 'mmp_general_form_position_variable', 'woocommerce_single_variation' ) : $variable_product_position;
			$variable_product_priority = get_option( 'mmp_general_form_position_variable_hook_priority', 10 );

			add_action( $single_product_position, array( __CLASS__, 'simple_product_form' ), $single_product_priority );
			add_action( $variable_product_position, array( __CLASS__, 'variable_product_form' ), $variable_product_priority );
			add_action( 'wp_head', array( __CLASS__, 'auto_apply_process' ) );
		}

		/**
		 * Auto Apply Process
		 *
		 * @since 1.0.0
		 */
		public static function auto_apply_process() {
			if ( ! isset( $_GET['auto_apply'] ) ) {
				return;
			}

			$list_obj = mmp_get_entry( wc_clean( wp_unslash( $_GET['auto_apply'] ) ) );

			WC()->cart->add_to_cart( $list_obj->get_pro_id() );

			$coupon_code = mmp_get_entry( wc_clean( wp_unslash( $_GET['auto_apply'] ) ) )->get_coupon_title();

			if ( WC()->cart->has_discount( $coupon_code ) ) {
				wp_safe_redirect( wc_get_cart_url() );
			}

			WC()->cart->add_discount( $coupon_code );
			wp_safe_redirect( wc_get_cart_url() );
		}

		/**
		 * Display Form in Simple Product
		 *
		 * @since 1.0.0
		 */
		public static function simple_product_form() {
			global $post, $product;

			if ( 'yes' !== get_option( 'mmp_general_mmp_enable' ) ) {
				return;
			}

			if ( ! is_user_logged_in() && 'yes' === get_option( 'mmp_advanced_link_guest_hide' ) ) {
				return;
			}

			if ( 'yes' !== get_option( 'mmp_general_display_var_show' ) && ! $product->is_type( 'simple' ) ) {
				return;
			}

			wp_localize_script(
				'mmp_scripts',
				'mmp_frontend_params',
				array(
					'mmp_form_nonce'   => wp_create_nonce( 'mmp-form-nonce' ),
					'ajaxurl'          => MMP_ADMIN_AJAX_URL,
					'post_id'          => $post->ID,
					'link_var_display' => get_option( 'mmp_general_display_var_show' ),
				)
			);

			wp_enqueue_script( 'mmp_scripts' );
			?>

			<div class="mmp_simple_datas">
				<?php self::mmp_form_fields( $post->ID ); ?>
			</div>
			<?php
		}

		/**
		 * Display Form in Variable Product
		 *
		 * @since 1.0.0
		 */
		public static function variable_product_form() {
			if ( 'yes' !== get_option( 'mmp_general_mmp_enable' ) ) {
				return;
			}

			if ( ! is_user_logged_in() && 'yes' === get_option( 'mmp_advanced_link_guest_hide' ) ) {
				return;
			}

			?>
			<div class="mmp_variation_datas"></div> 
			<?php
		}

		/**
		 * Display Form
		 *
		 * @since 1.0.0
		 * @param Integer $post_id Post ID.
		 * @param String  $type Product Type.
		 */
		public static function mmp_form_fields( $post_id, $type = 'simple' ) {
			global $post;

			$customer_email = '';
			$customer_name  = '';
			$user           = get_userdata( get_current_user_id() );

			if ( $user && $user->exists() ) {
				$customer_email = $user->user_email;
				$customer_name  = $user->display_name;
			}

			if ( ! MMP_Restriction_Handler::is_valid_for_mmp( $post_id ) ) {
				return;
			}

			if ( 'variable' !== $type && is_user_logged_in() ) {
				if ( 'yes' === get_option( 'mmp_advanced_overall_product_restriction' ) ) {
					$threshold     = get_option( 'mmp_advanced_pro_restriction_val' );
					$request_count = mmp_is_get_request_by_value( $customer_email );

					if ( ! empty( $threshold ) && ! empty( $request_count ) && count( $request_count ) >= $threshold ) {
						echo esc_html( get_option( 'mmp_messages_overall_pro_restriction_msg' ) );
						return;
					}
				}

				// Per product check.
				if ( 'yes' === get_option( 'mmp_advanced_per_product_restriction' ) && mmp_is_per_count_check( $customer_email, $post_id, 'all' ) ) {
					echo esc_html( get_option( 'mmp_messages_per_pro_restriction_msg' ) );
					return;
				}

				// if accept message.
				if ( mmp_is_per_count_check( $customer_email, $post_id, 'accept' ) ) {
					echo esc_html( get_option( 'mmp_messages_req_approved_msg' ) );
					return;
				}

				if ( mmp_is_per_count_check( $customer_email, $post_id ) ) {
					// default already submit message.
					echo esc_html( get_option( 'mmp_messages_req_submitted_msg' ) );
					return;
				}
			}

			$custom_fields_obj = mmp_get_custom_fields(
				array(
					'status'  => 'enabled',
					'orderby' => 'sort_order',
					'order'   => 'ASC',
				)
			);

			if ( ! $custom_fields_obj->has_custom_field ) {
				return;
			}

			mmp_get_template(
				'mmp-form.php',
				array(
					'post_id'           => $post_id,
					'custom_fields_obj' => $custom_fields_obj,
					'customer_email'    => $customer_email,
					'customer_name'     => $customer_name,
				)
			);
		}
	}

	MMP_Handler::init();
}
