<?php
/**
 * Custom Field Functions.
 *
 * @package Lower Price/Functions.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'mmp_default_custom_fields' ) ) {
	/**
	 * Check whether the given the value is Custom Field.
	 *
	 * @since 3.3.0
	 * @return Array
	 */
	function mmp_default_custom_fields_key() {
		/**
		 * Custom Field Types
		 *
		 * @since 3.3.0
		 */
		return array(
			'mmp_name',
			'mmp_mail',
			'mmp_description',
			'mmp_pref_price',
			'mmp_number',
		);
	}
}

if ( ! function_exists( 'mmp_custom_field_types' ) ) {
	/**
	 * Check whether the given the value is Custom Field.
	 *
	 * @since 3.3.0
	 * @return Array
	 */
	function mmp_custom_field_types() {
		/**
		 * Custom Field Types
		 *
		 * @since 3.3.0
		 */
		return apply_filters(
			'mmp_custom_field_types',
			array(
				'dropdown' => esc_html__( 'Select Box', 'lower-price-for-woocommerce' ),
				'text'     => esc_html__( 'Text', 'lower-price-for-woocommerce' ),
				'email'    => esc_html__( 'Email', 'lower-price-for-woocommerce' ),
				'number'   => esc_html__( 'Number', 'lower-price-for-woocommerce' ),
				'radio'    => esc_html__( 'Radio', 'lower-price-for-woocommerce' ),
				'checkbox' => esc_html__( 'Checkbox', 'lower-price-for-woocommerce' ),
				'textarea' => esc_html__( 'Textarea', 'lower-price-for-woocommerce' ),
			)
		);
	}
}

if ( ! function_exists( 'mmp_is_custom_field' ) ) {
	/**
	 * Check whether the given the value is Custom Field.
	 *
	 * @since 3.3.0
	 * @param  Mixed $custom_field Post object or post ID of the Custom Field.
	 * @return Boolean True on success.
	 */
	function mmp_is_custom_field( $custom_field ) {
		return $custom_field && is_a( $custom_field, 'MMP_Custom_Field' );
	}
}

if ( ! function_exists( 'mmp_get_custom_field' ) ) {
	/**
	 * Get Custom Field.
	 *
	 * @since 3.3.0
	 * @param MMP_Custom_Field $custom_field Custom Field.
	 * @param Boolean          $wp_error WordPress error.
	 * @return bool|\MMP_Custom_Field
	 */
	function mmp_get_custom_field( $custom_field, $wp_error = false ) {
		if ( ! $custom_field ) {
			return false;
		}

		try {
			$custom_field = new MMP_Custom_Field( $custom_field );
		} catch ( Exception $e ) {
			return $wp_error ? new WP_Error( 'error', $e->getMessage() ) : false;
		}

		return $custom_field;
	}
}

if ( ! function_exists( 'mmp_get_custom_field_statuses' ) ) {
	/**
	 * Get Custom Field statuses.
	 *
	 * @since 3.3.0
	 * @return Array
	 */
	function mmp_get_custom_field_statuses() {
		return array(
			'mmp_enabled'  => __( 'Enable', 'lower-price-for-woocommerce' ),
			'mmp_disabled' => __( 'Disable', 'lower-price-for-woocommerce' ),
		);
	}
}

if ( ! function_exists( 'mmp_get_custom_field_status_name' ) ) {
	/**
	 * Get the custom field status name.
	 *
	 * @since 3.3.0
	 * @param String $status Status name.
	 * @return String
	 */
	function mmp_get_custom_field_status_name( $status ) {
		$statuses = mmp_get_custom_field_statuses();
		$status   = mmp_trim_post_status( $status );

		return isset( $statuses[ "mmp_{$status}" ] ) ? $statuses[ "mmp_{$status}" ] : $status;
	}
}

if ( ! function_exists( 'mmp_is_custom_field_status' ) ) {
	/**
	 * See if a string is an custom field status.
	 *
	 * @since 3.3.0
	 * @param  String $maybe_status Status, including any mmp_ prefix.
	 * @return Boolean
	 */
	function mmp_is_custom_field_status( $maybe_status ) {
		$statuses = mmp_get_custom_field_statuses();

		return isset( $statuses[ $maybe_status ] );
	}
}

if ( ! function_exists( 'mmp_get_custom_fields' ) ) {
	/**
	 * Return the array of custom fields based upon the args requested.
	 *
	 * @since 3.3.0
	 * @param Array $args Arguments.
	 * @return Object
	 */
	function mmp_get_custom_fields( $args = array() ) {
		global $wpdb;
		$wpdb_ref = &$wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'status'      => array_keys( mmp_get_custom_field_statuses() ),
				'include_ids' => array(),
				'exclude_ids' => array(),
				's'           => '',
				'source_from' => '',
				'page'        => 1,
				'limit'       => -1,
				'fields'      => 'objects',
				'orderby'     => 'ID',
				'order'       => 'DESC',
				'date_before' => '',
				'date_after'  => '',
				'name'        => '',
				'field_type'  => '',
			)
		);

		// Add the 'mmp_' prefix to status if needed.
		if ( ! empty( $args['status'] ) ) {
			if ( is_array( $args['status'] ) ) {
				foreach ( $args['status'] as &$status ) {
					if ( mmp_is_custom_field_status( 'mmp_' . $status ) ) {
						$status = 'mmp_' . $status;
					}
				}
			} elseif ( mmp_is_custom_field_status( 'mmp_' . $args['status'] ) ) {
				$args['status'] = 'mmp_' . $args['status'];
			}
		}

		// Date Before.
		if ( ! empty( $args['date_before'] ) ) {
			$date_before = 'AND date_created <="' . gmdate( 'Y-m-d H:i:s', absint( $args['date_before'] ) ) . '"';
		} else {
			$date_before = '';
		}

		// Date after.
		if ( ! empty( $args['date_after'] ) ) {
			$date_after = ' AND date_created >= "' . gmdate( 'Y-m-d H:i:s', absint( $args['date_after'] ) ) . '"';
		} else {
			$date_after = '';
		}

		// Statuses.
		if ( is_array( $args['status'] ) ) {
			$allowed_statuses = " AND status IN ('" . implode( "','", $args['status'] ) . "') ";
		} else {
			$allowed_statuses = " AND status = '" . esc_sql( $args['status'] ) . "' ";
		}

		// Search term.
		if ( ! empty( $args['s'] ) ) {
			$term          = str_replace( '#', '', wc_clean( wp_unslash( $args['s'] ) ) );
			$search_fields = array();
			$search_where  = " AND ( 
                (ID LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR
				(name LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR 
				(field_type LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR 
				(status LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR 
				(created_via LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR 
				(source_from LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%')
                ) ";
		} else {
			$search_where = '';
		}

		// Includes.
		if ( ! empty( $args['include_ids'] ) ) {
			$include_ids = " AND ID IN ('" . implode( "','", $args['include_ids'] ) . "') ";
		} else {
			$include_ids = '';
		}

		// Excludes.
		if ( ! empty( $args['exclude_ids'] ) ) {
			$exclude_ids = " AND ID NOT IN ('" . implode( "','", $args['exclude_ids'] ) . "') ";
		} else {
			$exclude_ids = '';
		}

		// Name.
		if ( ! empty( $args['name'] ) ) {
			if ( is_array( $args['name'] ) ) {
				$allowed_name = " AND name IN ('" . implode( "','", $args['name'] ) . "') ";
			} else {
				$allowed_name = " AND name = '" . esc_sql( $args['name'] ) . "' ";
			}
		} else {
			$allowed_name = '';
		}

		// Field Type.
		if ( ! empty( $args['field_type'] ) ) {
			if ( is_array( $args['field_type'] ) ) {
				$allowed_field_type = " AND field_type IN ('" . implode( "','", $args['field_type'] ) . "') ";
			} else {
				$allowed_field_type = " AND field_type = '" . esc_sql( $args['field_type'] ) . "' ";
			}
		} else {
			$allowed_field_type = '';
		}

		// Source From.
		if ( ! empty( $args['source_from'] ) ) {
			if ( is_array( $args['source_from'] ) ) {
				$allowed_source_from = " AND source_from IN ('" . implode( "','", $args['source_from'] ) . "') ";
			} else {
				$allowed_source_from = " AND source_from = '" . esc_sql( $args['source_from'] ) . "' ";
			}
		} else {
			$allowed_source_from = '';
		}

		// Order by.
		$orderby = ' ORDER BY ' . esc_sql( $args['orderby'] ) . ' ';

		// Order.
		if ( ! empty( $args['order'] ) && 'desc' === strtolower( $args['order'] ) ) {
			$order = ' DESC ';
		} else {
			$order = ' ASC ';
		}

		// Paging.
		if ( $args['limit'] >= 0 ) {
			$page   = absint( $args['page'] );
			$page   = $page ? $page : 1;
			$offset = absint( ( $page - 1 ) * $args['limit'] );
			$limits = 'LIMIT ' . $offset . ', ' . $args['limit'];
		} else {
			$limits = '';
		}

		$all_custom_field_ids = $wpdb_ref->get_var(
			"SELECT COUNT(DISTINCT ID) FROM {$wpdb->prefix}mmp_custom_field AS c
			WHERE 1=1 {$allowed_statuses} {$search_where} {$include_ids} {$exclude_ids} {$allowed_name} {$allowed_field_type} {$allowed_source_from} {$date_before} {$date_after}"
		);

		$custom_field_ids = $wpdb_ref->get_col(
			"SELECT DISTINCT ID FROM {$wpdb->prefix}mmp_custom_field AS c
			WHERE 1=1 {$allowed_statuses} {$search_where} {$include_ids} {$exclude_ids} {$allowed_name} {$allowed_field_type} {$allowed_source_from} {$date_before} {$date_after}
			{$orderby} {$order} {$limits}"
		);

		if ( 'objects' === $args['fields'] ) {
			$custom_fields = array_filter( array_combine( $custom_field_ids, array_map( 'mmp_get_custom_field', $custom_field_ids ) ) );
		} else {
			$custom_fields = $custom_field_ids;
		}

		$query_object = (object) array(
			'custom_fields'       => $custom_fields,
			'total_custom_fields' => $all_custom_field_ids,
			'has_custom_field'    => count( $custom_field_ids ) > 0,
			'max_num_pages'       => $args['limit'] > 0 ? ceil( $all_custom_field_ids / $args['limit'] ) : 1,
		);

		return $query_object;
	}
}

if ( ! function_exists( 'mmp_output_custom_fields' ) ) {
	/**
	 * Return custom field.
	 *
	 * @since 3.3.0
	 * @param Array  $custom_field_obj Custom Field Object.
	 * @param String $field_value Field Value.
	 */
	function mmp_output_custom_fields( $custom_field_obj, $field_value = '' ) {
		ob_start();

		if ( empty( $field_value ) ) {
			$field_value = $custom_field_obj->get_default();
		}

		$wrapper_class_val = is_admin() ? 'form-field ' . $custom_field_obj->get_name() . '_field' : 'woocommerce-form-row mmp_form_row form-row';
		$class_val         = is_admin() ? 'mmp-admin-custom-field' : 'mmp-custom-field';
		$class_val        .= ' ' . $custom_field_obj->get_class_attr();
		?>
		<p class="<?php echo esc_html( $wrapper_class_val ); ?>">
			<label for="<?php echo esc_html( $custom_field_obj->get_name() ); ?>">
				<?php
				echo esc_html( $custom_field_obj->get_label() );

				if ( 'yes' === $custom_field_obj->get_required() ) :
					?>
					<span class="required">*</span> 
					<?php endif; ?>
			</label>
			<?php
			$name = ( 'custom' === $custom_field_obj->get_source_from() ) ? '_custom_fields[' . esc_html( $custom_field_obj->get_name() ) . ']' : esc_html( $custom_field_obj->get_name() );

			switch ( $custom_field_obj->get_field_type() ) {
				case 'text':
					?>
					<input 
					type="text" 
					class="mmp-form-fields <?php echo esc_html( $class_val ); ?>" 
					name="<?php echo esc_html( $name ); ?>"
					id="<?php echo esc_html( $custom_field_obj->get_id() ); ?>" 
					value="<?php echo esc_html( $field_value ); ?>" 
					placeholder="<?php echo esc_html( $custom_field_obj->get_placeholder() ); ?>">
					<?php
					break;
				case 'textarea':
					?>
					<textarea 
					class="mmp-form-fields <?php echo esc_html( $class_val ); ?>" 
					name="<?php echo esc_html( $name ); ?>"
					id="<?php echo esc_html( $custom_field_obj->get_id() ); ?>" 
					<?php if ( $custom_field_obj->get_cols() ) : ?>
						cols="<?php echo wp_kses_post( $custom_field_obj->get_cols() ); ?>" 
						<?php
						endif;
					if ( $custom_field_obj->get_rows() ) :
						?>
						rows="<?php echo wp_kses_post( $custom_field_obj->get_rows() ); ?>" 
					<?php endif; ?>
					placeholder="<?php echo esc_html( $custom_field_obj->get_placeholder() ); ?>"><?php echo esc_html( $field_value ); ?></textarea>
					<?php
					break;
				case 'number':
					?>
					<input 
					type="number" 
					min="<?php echo wp_kses_post( $custom_field_obj->get_min() ); ?>" 
					<?php if ( $custom_field_obj->get_max() ) : ?>
						max="<?php echo wp_kses_post( $custom_field_obj->get_max() ); ?>" 
					<?php endif; ?>
					step = "<?php echo wp_kses_post( $custom_field_obj->get_step() ); ?>" 
					class="mmp-form-fields <?php echo esc_html( $class_val ); ?>" 
					name="<?php echo esc_html( $name ); ?>"
					id="<?php echo esc_html( $custom_field_obj->get_id() ); ?>" 
					value="<?php echo esc_html( $field_value ); ?>" 
					placeholder="<?php echo esc_html( $custom_field_obj->get_placeholder() ); ?>">
					<?php
					break;
				case 'dropdown':
					?>
					<select id="mmp-form-fields <?php echo esc_html( $custom_field_obj->get_id() ); ?>" name="<?php echo esc_html( $name ); ?>" class="<?php echo wp_kses_post( $class_val ); ?>">
						<?php
						foreach ( json_decode( $custom_field_obj->get_options_attr() ) as $key => $value ) :
							echo '<option value="' . esc_attr( $key ) . '"' . esc_html( wc_selected( $key, $field_value ) ) . '>' . esc_html( $value ) . '</option>';
						endforeach;
						?>
					</select>
					<?php
					break;
				case 'radio':
					foreach ( json_decode( $custom_field_obj->get_options_attr() ) as $key => $value ) :
						$checked = ( $key == $field_value ) ? 'checked=checked' : '';
						?>
						<input type="radio" 
						name="<?php echo esc_html( $name ); ?>"
						id="<?php echo esc_attr( $key ); ?>" 
						value="<?php echo esc_attr( $key ); ?>" 
						class="mmp-form-fields <?php echo esc_html( $class_val ); ?>" 
						<?php echo esc_attr( $checked ); ?>> 
						<?php
						echo esc_attr( $value );
					endforeach;
					break;
				case 'checkbox':
					foreach ( json_decode( $custom_field_obj->get_options_attr() ) as $key => $value ) :
						$checked = mmp_check_is_array( $field_value ) && in_array( $key, $field_value ) ? 'checked=checked' : '';
						?>

					<input type="checkbox" 
						name="<?php echo esc_html( $name ) . '[' . esc_html( $key ) . ']'; ?>"
						id="<?php echo esc_attr( $key ); ?>"
						value="<?php echo esc_attr( $key ); ?>" 
						class="mmp-form-fields <?php echo esc_html( $class_val ); ?>" 
						<?php echo esc_attr( $checked ); ?>>
						<?php
						echo esc_attr( $value );
					endforeach;
					break;
				case 'email':
					?>
					<input 
					type="email" 
					class="mmp-form-fields <?php echo esc_html( $class_val ); ?>" 
					name="<?php echo esc_html( $name ); ?>"
					id="<?php echo esc_html( $custom_field_obj->get_id() ); ?>" 
					value="<?php echo esc_html( $field_value ); ?>" 
					placeholder="<?php echo esc_html( $custom_field_obj->get_placeholder() ); ?>">
					<?php
					break;
			}
			?>
		<span class="description"><?php echo wp_kses_post( $custom_field_obj->get_tool_tip() ); ?></span>
		</p>
		<?php
		ob_end_flush();
	}
}
