<?php
/**
 * Layout functions
 *
 * @package Lower Price
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'mmp_select2_html' ) ) {

	/**
	 * Return or display Select2 HTML
	 *
	 * @since 1.0.0
	 * @param Array   $args Arguments.
	 * @param Boolean $echo Whether to echo.
	 * @return string
	 */
	function mmp_select2_html( $args, $echo = true ) {
		$args = wp_parse_args(
			$args,
			array(
				'class'                   => '',
				'id'                      => '',
				'name'                    => '',
				'list_type'               => '',
				'action'                  => '',
				'placeholder'             => '',
				'exclude_global_variable' => 'no',
				'custom_attributes'       => array(),
				'multiple'                => true,
				'allow_clear'             => true,
				'selected'                => true,
				'options'                 => array(),
			)
		);

		$multiple     = $args['multiple'] ? 'multiple="multiple"' : '';
		$name         = esc_attr( '' !== $args['name'] ? $args['name'] : $args['id'] ) . '[]';
		$options      = array_filter( mmp_check_is_array( $args['options'] ) ? $args['options'] : array() );
		$allowed_html = array(
			'select' => array(
				'id'                           => array(),
				'class'                        => array(),
				'data-placeholder'             => array(),
				'data-allow_clear'             => array(),
				'data-exclude-global-variable' => array(),
				'data-action'                  => array(),
				'multiple'                     => array(),
				'name'                         => array(),
			),
			'option' => array(
				'value'    => array(),
				'selected' => array(),
			),
		);

		// Custom attribute handling.
		$custom_attributes = mmp_format_custom_attributes( $args );

		ob_start();
		?><select <?php echo esc_attr( $multiple ); ?> 
			name="<?php echo esc_attr( $name ); ?>" 
			id="<?php echo esc_attr( $args['id'] ); ?>" 
			data-action="<?php echo esc_attr( $args['action'] ); ?>" 
			data-exclude-global-variable="<?php echo esc_attr( $args['exclude_global_variable'] ); ?>" 
			class="mmp_select2_search <?php echo esc_attr( $args['class'] ); ?>" 
			data-placeholder="<?php echo esc_attr( $args['placeholder'] ); ?>" 
			<?php echo wp_kses( implode( ' ', $custom_attributes ), $allowed_html ); ?>
			<?php echo $args['allow_clear'] ? 'data-allow_clear="true"' : ''; ?> >
				<?php
				if ( is_array( $args['options'] ) ) {
					foreach ( $args['options'] as $option_id ) {
						$option_value = '';
						switch ( $args['list_type'] ) {
							case 'post':
								$option_value = get_the_title( $option_id );
								break;
							case 'products':
								$option_value = get_the_title( $option_id ) . ' (#' . absint( $option_id ) . ')';
								break;
							case 'customers':
								$user = get_user_by( 'id', $option_id );
								if ( $user ) {
									$option_value = $user->display_name . '(#' . absint( $user->ID ) . ' &ndash; ' . $user->user_email . ')';
								}
								break;
						}

						if ( $option_value ) {
							?>
						<option value="<?php echo esc_attr( $option_id ); ?>" <?php echo $args['selected'] ? 'selected="selected"' : ''; ?>><?php echo esc_html( $option_value ); ?></option>
							<?php
						}
					}
				}
				?>
		</select>
		<?php
		$html = ob_get_clean();

		if ( $echo ) {
			echo wp_kses( $html, $allowed_html );
		}

		return $html;
	}
}

if ( ! function_exists( 'mmp_format_custom_attributes' ) ) {

	/**
	 * Format Custom Attributes
	 *
	 * @since 1.0.0
	 * @param Array $value Custom attributes.
	 * @return array
	 */
	function mmp_format_custom_attributes( $value ) {
		$custom_attributes = array();

		if ( ! empty( $value['custom_attributes'] ) && is_array( $value['custom_attributes'] ) ) {
			foreach ( $value['custom_attributes'] as $attribute => $attribute_value ) {
				$custom_attributes[] = esc_attr( $attribute ) . '=' . esc_attr( $attribute_value ) . '';
			}
		}

		return $custom_attributes;
	}
}

if ( ! function_exists( 'mmp_get_datepicker_html' ) ) {

	/**
	 * Return or display Datepicker HTML
	 *
	 * @since 1.0.0
	 * @param Array   $args Array of Arguments.
	 * @param Boolean $echo Whether to echo.
	 * @return string
	 */
	function mmp_get_datepicker_html( $args, $echo = true ) {
		$args         = wp_parse_args(
			$args,
			array(
				'class'             => '',
				'id'                => '',
				'name'              => '',
				'placeholder'       => '',
				'custom_attributes' => array(),
				'value'             => '',
				'wp_zone'           => true,
			)
		);
		$name         = ( '' !== $args['name'] ) ? $args['name'] : $args['id'];
		$allowed_html = array(
			'input' => array(
				'id'          => array(),
				'type'        => array(),
				'placeholder' => array(),
				'class'       => array(),
				'value'       => array(),
				'name'        => array(),
				'min'         => array(),
				'max'         => array(),
				'style'       => array(),
			),
		);

		// Custom attribute handling.
		$custom_attributes = mmp_format_custom_attributes( $args );
		$value             = ! empty( $args['value'] ) ? FGF_Date_Time::get_date_object_format_datetime( $args['value'], 'date', $args['wp_zone'] ) : '';
		ob_start();

		?>
		<input type = "text" id="<?php echo esc_attr( $args['id'] ); ?>" value = "<?php echo esc_attr( $value ); ?>" class="mmp_datepicker <?php echo esc_attr( $args['class'] ); ?>" placeholder="<?php echo esc_attr( $args['placeholder'] ); ?>" <?php echo wp_kses( implode( ' ', $custom_attributes ), $allowed_html ); ?> />
		<input type = "hidden" class="mmp_alter_datepicker_value" name="<?php echo esc_attr( $name ); ?>" value = "<?php echo esc_attr( $args['value'] ); ?>" /> 
		<?php

		$html = ob_get_clean();

		if ( $echo ) {
			echo wp_kses( $html, $allowed_html );
		}

		return $html;
	}
}

if ( ! function_exists( 'mmp_display_status' ) ) {
	/**
	 * Display formatted status
	 *
	 * @since 1.0.0
	 * @param String  $status Status.
	 * @param Boolean $html Whether to display.
	 * @return string
	 */
	function mmp_display_status( $status, $html = true ) {
		$status_object = get_post_status_object( $status );

		if ( ! isset( $status_object ) ) {
			return '';
		}

		echo $html ? '<mark class="mmp_status_label ' . esc_attr( $status ) . '_status"><span >' . esc_html( $status_object->label ) . '</span></mark>' : esc_html( $status_object->label );
	}
}

if ( ! function_exists( 'mmp_get_list_status' ) ) {

	/**
	 * Display formatted status - Frontend
	 *
	 * @since 1.0.0
	 * @param String $status Status.
	 * @return HTML
	 */
	function mmp_get_list_status( $status ) {
		switch ( $status ) {
			case 'mmp_new_req':
				return '<mark class="mmp_status column-mmp_status"><span>' . esc_html__( 'Request Sent', 'lower-price-for-woocommerce' ) . '</span></mark>';
			case 'mmp_coupon_sent':
				return '<mark class="mmp_status column-mmp_status"><span>' . esc_html__( 'Coupon Received', 'lower-price-for-woocommerce' ) . '</span></mark>';
			case 'mmp_coupon_used':
				return '<mark class="mmp_status column-mmp_status"><span>' . esc_html__( 'Coupon Used', 'lower-price-for-woocommerce' ) . '</span></mark>';
			case 'mmp_coupon_expired':
				return '<mark class="mmp_status column-mmp_status"><span>' . esc_html__( 'Coupon Expired', 'lower-price-for-woocommerce' ) . '</span></mark>';
			case 'mmp_rejected':
				return '<mark class="mmp_status column-mmp_status"><span>' . esc_html__( 'Rejected', 'lower-price-for-woocommerce' ) . '</span></mark>';
		}
	}
}



if ( ! function_exists( 'mmp_get_template' ) ) {

	/**
	 *  Get other templates from themes
	 *
	 * @since 1.0.0
	 * @param string $template_name template name.
	 * @param Array  $args template arguments.
	 */
	function mmp_get_template( $template_name, $args = array() ) {
		wc_get_template( $template_name, $args, 'mmp/', MMP()->templates() );
	}
}

if ( ! function_exists( 'mmp_get_template_html' ) ) {
	/**
	 *  Like mmp_get_template, but returns the HTML instead of outputting.
	 *
	 * @since 1.0.0
	 * @param string $template_name template name.
	 * @param Array  $args template arguments.
	 *  @return string
	 */
	function mmp_get_template_html( $template_name, $args = array() ) {
		ob_start();

		mmp_get_template( $template_name, $args );

		return ob_get_clean();
	}
}

if ( ! function_exists( 'wc_mmp_get_template' ) ) {
	/**
	 *  Get other templates from themes
	 *
	 * @since 1.0.0
	 * @param string $template_name template name.
	 * @param Array  $args template arguments.
	 */
	function wc_mmp_get_template( $template_name, $args = array() ) {
		if ( ! $template_name ) {
			return;
		}

		wc_get_template( $template_name, $args, WC_MMP_BASENAME_DIR, WC_MMP_TEMPLATE_PATH );
	}
}



if ( ! function_exists( 'mmp_render_product_image' ) ) {

	/**
	 *  Display Product image
	 *
	 * @since 1.0.0
	 * @param WC_Product $product Product object.
	 * @param Boolean    $echo whether to echo the product image.
	 *  @return string
	 */
	function mmp_render_product_image( $product, $echo = true ) {
		$allowed_html = array(
			'a'   => array(
				'href' => array(),
			),
			'img' => array(
				'class'  => array(),
				'src'    => array(),
				'alt'    => array(),
				'srcset' => array(),
				'sizes'  => array(),
				'width'  => array(),
				'height' => array(),
			),
		);

		if ( $echo ) {
			echo wp_kses( $product->get_image( array( 50, 50 ) ), $allowed_html );
		}

		return $product->get_image();
	}
}

if ( ! function_exists( 'mmp_price' ) ) {

	/**
	 *  Display Price based wc_price function
	 *
	 * @since 1.0.0
	 * @param String  $price Price.
	 * @param Boolean $echo Whether to echo.
	 * @return string
	 */
	function mmp_price( $price, $echo = true ) {
		$allowed_html = array(
			'span' => array(
				'class' => array(),
			),
		);

		if ( $echo ) {
			echo wp_kses( wc_price( $price ), $allowed_html );
		}

		return wc_price( $price );
	}
}

if ( ! function_exists( 'get_thanks_layout' ) ) {

	/**
	 *  Display Thanks Page
	 *
	 * @since 1.0.0
	 */
	function get_thanks_layout() {
		ob_start();

		if ( 'mmp_feedback' === get_option( 'mmp_general_mode_of_operaton' ) ) {
			?>
			<div class="mmp_success"><img src="<?php echo esc_attr( MMP_PLUGIN_URL . '/assets/images/success-img.png' ); ?>"><br><?php echo esc_html( get_option( 'mmp_messages_ack_msg' ) ); ?></div>
				<?php
		} else {
			?>
			<div class="mmp_success"><img src="<?php echo esc_attr( MMP_PLUGIN_URL . '/assets/images/success-img.png' ); ?>"><br><?php echo esc_html( get_option( 'mmp_messages_ack_coupon_msg' ) ); ?></div>
			<?php
		}

		$contents = ob_get_contents();

		ob_end_clean();

		return $contents;
	}
}

add_action( 'mmp_get_pro_table', 'mmp_product_table_layout', 10, 2 );

/**
 * Display Product Table Layout
 *
 * @since 1.0.0
 * @param Object  $list_obj List object.
 * @param Integer $pro_id Product ID.
 */
function mmp_product_table_layout( $list_obj, $pro_id ) {
	$product_data = wc_get_product( $pro_id );
	?>
	<table width="100%" style="border-collapse:collapse;border:1px solid #ccc;">
		<tr>
			<th style="text-align: center;border-bottom:1px solid #ccc;border-right:1px solid #ccc;"><?php echo esc_html__( 'Image', 'lower-price-for-woocommerce' ); ?></th>
			<th style="text-align: center;border-bottom:1px solid #ccc;border-right:1px solid #ccc;"><?php echo esc_html__( 'Name', 'lower-price-for-woocommerce' ); ?></th>
			<th style="text-align: center;border-bottom:1px solid #ccc;border-right:1px solid #ccc;"><?php echo esc_html__( 'Price', 'lower-price-for-woocommerce' ); ?></th>
			<th style="text-align: center;border-bottom:1px solid #ccc;border-right:1px solid #ccc;"><?php echo esc_html__( 'Reported Price', 'lower-price-for-woocommerce' ); ?></th>
		</tr>
		<tr>
			<td style="text-align: center;border-right:1px solid #ccc;"><?php mmp_render_product_image( $product_data ); ?></td>
			<td style="text-align: center;border-right:1px solid #ccc;"><?php echo esc_html( $product_data->get_title() ); ?></td>
			<td style="text-align: center;border-right:1px solid #ccc;"><?php mmp_price( $product_data->get_price() ); ?></td>
			<td style="text-align: center;border-right:1px solid #ccc;"><?php mmp_price( $list_obj->get_pref_price() ); ?></td>
		</tr>
	</table>
	<?php
	$contents = ob_get_contents();
	ob_end_clean();
	echo wp_kses_post( $contents );
}

add_action( 'mmp_get_request_table', 'mmp_request_table_layout', 10, 2 );

/**
 * Display request Table Layout
 *
 * @since 1.0.0
 * @param Object  $list_obj List Object.
 * @param Integer $pro_id Product ID.
 */
function mmp_request_table_layout( $list_obj, $pro_id ) {
	$product_data = wc_get_product( $pro_id );
	?>
	<h2><?php echo esc_html__( 'Details', 'lower-price-for-woocommerce' ); ?></h2>
	<table width="100%">
		<tr>
			<th><?php echo esc_html__( 'Image', 'lower-price-for-woocommerce' ); ?></th>
			<td><?php mmp_render_product_image( $product_data ); ?></td>
		</tr>
		<tr>
			<th><?php echo esc_html__( 'Product Name', 'lower-price-for-woocommerce' ); ?></th>
			<th><?php echo esc_html( $product_data->get_title() ); ?></th>
		</tr>
		<tr>
			<th><?php echo esc_html__( 'Original Price', 'lower-price-for-woocommerce' ); ?></th>
			<th><?php mmp_price( $list_obj->get_org_price() ); ?></th>    
		</tr> 
		<tr>
			<th><?php echo esc_html__( 'Reported Price', 'lower-price-for-woocommerce' ); ?></th>
			<th><?php mmp_price( $list_obj->get_pref_price() ); ?></th>
		</tr>
		<tr>
			<th><?php echo esc_html__( 'Offered Price', 'lower-price-for-woocommerce' ); ?></th>
			<th><?php echo wp_kses_post( $list_obj->get_off_price_formatted() ); ?></th>
		</tr>
		<tr>
			<th><?php echo esc_html__( 'Coupon Code', 'lower-price-for-woocommerce' ); ?></th>
			<th><?php echo esc_html( mmp_is_empty_check( $list_obj->get_coupon_title() ) ); ?></th>
		</tr>  
		<tr>
			<th><?php echo esc_html__( 'Coupon Validity', 'lower-price-for-woocommerce' ); ?></th>
			<th><?php echo esc_html( $list_obj->get_coupon_expiry() ); ?></th>
		</tr>
	</table>
	<?php
	$contents = ob_get_contents();
	ob_end_clean();
	echo wp_kses_post( $contents );
}


add_action( 'mmp_get_guest_table', 'request_guest_layout', 10, 2 );

/**
 *  Display request Guest Table Layout
 *
 * @since 1.0.0
 * @param Object  $list_obj List Object.
 * @param Integer $pro_id Product ID.
 */
function request_guest_layout( $list_obj, $pro_id ) {
	ob_start();
	?>
	<h2><?php esc_html_e( 'Details', 'lower-price-for-woocommerce' ); ?></h2>
	<table width="100%">
		<?php
		if ( '-' !== $list_obj->get_name() ) {
			?>
			<tr>  
				<th><?php esc_html_e( 'Customer Name', 'lower-price-for-woocommerce' ); ?></th>
				<th><?php echo esc_html( $list_obj->get_name() ); ?></th>
			</tr> 
			<?php
		}

		if ( '-' !== $list_obj->get_description() ) {
			?>
			<tr>
				<th><?php esc_html_e( 'Description', 'lower-price-for-woocommerce' ); ?></th>
				<th><?php echo wp_kses_post( $list_obj->get_description() ); ?></th>
			</tr>
			<?php
		}

		if ( '-' !== $list_obj->get_pref_price() ) {
			?>
			<tr>
				<th><?php esc_html_e( 'Reported Price', 'lower-price-for-woocommerce' ); ?></th>
				<th><?php mmp_price( $list_obj->get_pref_price() ); ?></th>
			</tr>
			<?php
		}

		if ( '-' !== $list_obj->get_number() ) {
			?>
			<tr>
				<th><?php esc_html_e( 'Mobile Number', 'lower-price-for-woocommerce' ); ?></th>
				<th><?php echo esc_html( $list_obj->get_number() ); ?></th>
			</tr> 
			<?php
		}

		if ( '-' !== $list_obj->get_reason() ) {
			?>
		<tr>
			<th><?php echo esc_html__( 'Reason', 'lower-price-for-woocommerce' ); ?></th>
			<th><?php echo wp_kses_post( $list_obj->get_reason() ); ?></th>
		</tr>
		<?php } ?>
	</table>
	<?php
	$contents = ob_get_contents();

	ob_end_clean();

	echo wp_kses_post( $contents );
}
