<?php
/**
 * Admin Ajax.
 *
 * @package Lower Price|AJAX
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
if ( ! class_exists( 'MMP_Admin_Ajax' ) ) {

	/**
	 * Main Class.
	 */
	class MMP_Admin_Ajax {

		/**
		 * Class initialization.
		 */
		public static function init() {

			$actions = array(
				'product_search'                 => false,
				'products_and_variations_search' => false,
				'customers_search'               => false,
				'form_fields_sorting'            => false,
				'form_fields_save'               => true,
				'list_pagination'                => true,
				'get_form_fields'                => true,
				'update_viewed_once'             => false,
			);

			foreach ( $actions as $action => $nopriv ) {
				add_action( 'wp_ajax_mmp_' . $action, array( __CLASS__, $action ) );

				if ( $nopriv ) {
					add_action( 'wp_ajax_nopriv_mmp_' . $action, array( __CLASS__, $action ) );
				}
			}
		}

		/**
		 * Product search.
		 */
		public static function product_search() {
			check_ajax_referer( 'mmp-search-nonce', 'mmp_security' );

			try {
				$term = isset( $_GET['term'] ) ? (string) wc_clean( wp_unslash( $_GET['term'] ) ) : '';

				if ( empty( $term ) ) {
					throw new exception( esc_html__( 'No Product(s) found', 'lower-price-for-woocommerce' ) );
				}

				$data_store              = WC_Data_Store::load( 'product' );
				$ids                     = $data_store->search_products( $term, '', false );
				$product_objects         = array_filter( array_map( 'wc_get_product', $ids ), 'wc_products_array_filter_readable' );
				$products                = array();
				$exclude_global_variable = isset($_GET['exclude_global_variable']) ? wc_clean(wp_unslash($_GET['exclude_global_variable'])) : 'no'; // @codingStandardsIgnoreLine.

				foreach ( $product_objects as $product_object ) {
					if ( 'yes' === $exclude_global_variable && $product_object->is_type( 'variable' ) ) {
						continue;
					}

					$products[ $product_object->get_id() ] = rawurldecode( $product_object->get_formatted_name() );
				}
				wp_send_json( $products );
			} catch ( Exception $ex ) {
				wp_die();
			}
		}

		/**
		 * Search for product variations
		 *
		 * @since 1.0
		 * @param String  $term term to search.
		 * @param Boolean $include_variations Include Variations.
		 */
		public static function products_and_variations_search( $term = '', $include_variations = false ) {
			self::product_search( '', true );
		}

		/**
		 * Customer search.
		 */
		public static function customers_search() {
			check_ajax_referer( 'mmp-search-nonce', 'mmp_security' );

			try {
				$term = isset( $_GET['term'] ) ? (string) wc_clean( wp_unslash( $_GET['term'] ) ) : '';

				if ( empty( $term ) ) {
					throw new exception( esc_html__( 'No Customer(s) found', 'lower-price-for-woocommerce' ) );
				}

				$exclude = isset( $_GET['exclude'] ) ? (string) wc_clean( wp_unslash( $_GET['exclude'] ) ) : '';
				$exclude = ! empty( $exclude ) ? array_map( 'intval', explode( ',', $exclude ) ) : array();

				$found_customers = array();

				$customers_query = new WP_User_Query(
					array(
						'fields'         => 'all',
						'orderby'        => 'display_name',
						'search'         => '*' . $term . '*',
						'search_columns' => array( 'ID', 'user_login', 'user_email', 'user_nicename' ),
					)
				);

				$customers = $customers_query->get_results();

				if ( mmp_check_is_array( $customers ) ) {
					foreach ( $customers as $customer ) {
						if ( ! in_array( $customer->ID, $exclude ) ) {
							$found_customers[ $customer->ID ] = $customer->display_name . ' (#' . $customer->ID . ' &ndash; ' . sanitize_email( $customer->user_email ) . ')';
						}
					}
				}

				wp_send_json( $found_customers );
			} catch ( Exception $ex ) {
				wp_die();
			}
		}

		/**
		 * Form Fields Sorting
		 */
		public static function form_fields_sorting() {
			check_ajax_referer( 'mmp-field-sort-nonce', 'mmp_security' );

			try {

				if ( ! isset( $_POST ) || ! isset( $_POST['sort_order'] ) || ( isset( $_GET['term'] ) && ! mmp_check_is_array( wc_clean( wp_unslash( $_GET['term'] ) ) ) ) ) {
					throw new exception( __( 'Invalid Request', 'lower-price-for-woocommerce' ) );
				}

				$sort_ids = array();

				$sort_order = wc_clean( wp_unslash( $_POST['sort_order'] ) );

				foreach ( $sort_order as $key => $post_id ) {
					$sort_ids[ $key + 1 ] = str_replace( 'post-', '', $post_id );
				}

				foreach ( $sort_ids as $menu_order => $post_id ) {
					wp_update_post(
						array(
							'ID'         => $post_id,
							'menu_order' => $menu_order,
						)
					);
				}

				wp_send_json_success();
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => $e->getMessage() ) );
			}
		}

		/**
		 * Form Fields Save
		 *
		 * @since 1.0
		 */
		public static function form_fields_save() {
			check_ajax_referer( 'mmp-form-nonce', 'mmp_security' );

			try {
				if ( ! isset( $_REQUEST ) || ! isset( $_REQUEST['popup_form_data'] ) || ! isset( $_REQUEST['mmp_email'] ) ) {
					throw new exception( __( 'Invalid Request', 'lower-price-for-woocommerce' ) );
				}

				$form_data = array();
				parse_str( wc_clean( $_REQUEST['popup_form_data'] ), $form_data );

				$user_email      = sanitize_email( $_REQUEST['mmp_email'] );
				$mmp_description = isset( $_REQUEST['mmp_description'] ) ? wc_clean( $_REQUEST['mmp_description'] ) : '';

				if ( 'yes' == get_option( 'mmp_advanced_overall_product_restriction' ) ) {
					$request_count = mmp_is_get_request_by_value( $user_email );

					if ( '' !== get_option( 'mmp_advanced_pro_restriction_val' ) && ( $request_count ) && count( $request_count ) >= get_option( 'mmp_advanced_pro_restriction_val' ) ) {
						throw new exception( get_option( 'mmp_messages_overall_pro_restriction_msg' ) );
					}
				}

				// Per product check.
				if ( 'yes' === get_option( 'mmp_advanced_per_product_restriction' ) && mmp_is_per_count_check( $user_email, $form_data['mmp_post_id'] ) ) {
					throw new exception( get_option( 'mmp_messages_per_pro_restriction_msg' ) );
				}

				// If accept message.
				if ( mmp_is_per_count_check( $user_email, $form_data['mmp_post_id'], 'accept' ) ) {
					throw new exception( get_option( 'mmp_messages_req_approved_msg' ) );
				}

				// Default already submit message.
				if ( mmp_is_per_count_check( $user_email, $form_data['mmp_post_id'] ) ) {
					throw new exception( get_option( 'mmp_messages_req_submitted_msg' ) );
				}

				$post_args = array(
					'post_type'   => MMP_Register_Post_Types::MMP_FORM_FIELDS_POSTTYPE,
					'post_status' => 'mmp_enabled',
					'fields'      => 'ids',
					'orderby'     => 'menu_order',
					'order'       => 'ASC',
					'numberposts' => -1,
				);

				$form_fields = get_posts( $post_args );

				if ( ! $form_fields ) {
					return;
				}

				$meta_args = array( 'mmp_viewed_once' => 'no' );

				foreach ( $form_fields as $field_id ) {
					$form_field = mmp_get_fields_entry( $field_id );

					if ( 'yes' === $form_field->get_field_mandatory() && ( isset( $form_data[ $form_field->get_field_name() ] ) && empty( $form_data[ $form_field->get_field_name() ] ) ) ) {
						throw new exception( sprintf( '%s is Mandatory', $form_field->get_field_label() ) );
					}

					if ( ( 'mmp_mail' === $form_field->get_field_name() ) && 'yes' === $form_field->get_field_mandatory() && ! filter_var( $user_email, FILTER_VALIDATE_EMAIL ) ) {
						throw new exception( esc_html__( 'Invalid Email ID', 'lower-price-for-woocommerce' ) );
					}

					if ( ( 'mmp_number' === $form_field->get_field_name() ) && ! empty( $form_data[ $form_field->get_field_name() ] ) && ! is_numeric( $form_data[ $form_field->get_field_name() ] ) ) {
						throw new exception( sprintf( '%s - Please Enter Only Numbers', $form_field->get_field_label() ) );
					}

					if ( ( 'mmp_pref_price' === $form_field->get_field_name() ) && ! empty( $form_data[ $form_field->get_field_name() ] ) && isset( $form_data['mmp_post_id'] ) ) {
						$product       = wc_get_product( $form_data['mmp_post_id'] );
						$product_price = ( 'variable' === $product->get_type() ) ? $product->get_variation_price( 'max' ) : $product->get_price();

						/**
						 * Lower Price Product Original Price
						 *
						 * @since 3.0.0
						 */
						$product_price = apply_filters( 'mmp_product_original_price', $product_price, $product );

						if ( $form_data[ $form_field->get_field_name() ] >= $product_price ) {
							/* translators: %s: Product Price */
							throw new exception( sprintf( esc_html__( 'Enter the price lesser than the product price %s', 'lower-price-for-woocommerce' ), wp_kses_post( wc_price( $product_price ) ) ) );
						}
					}

					if ( 'mmp_mail' !== $form_field->get_field_name() ) {
						$meta_args[ $form_field->get_field_name() ] = $form_data[ $form_field->get_field_name() ];
					}
				}

				$post_author = ( is_user_logged_in() ) ? get_current_user_id() : 1;
				$user_type   = ( is_user_logged_in() ) ? 'user' : 'guest';

				$meta_args['mmp_mail']           = $user_email;
				$meta_args['mmp_description']    = $mmp_description;
				$meta_args['mmp_org_price']      = mmp_get_display_product_price( wc_get_product( $form_data['mmp_post_id'] ) );
				$meta_args['mmp_pro_id']         = $form_data['mmp_post_id'];
				$meta_args['mmp_submitted_date'] = current_time( 'mysql', true );
				$meta_args['mmp_lst_activity']   = current_time( 'mysql', true );
				$meta_args['mmp_request_type']   = get_option( 'mmp_general_mode_of_operaton' );
				$meta_args['mmp_user_type']      = $user_type;
				$entry_id                        = mmp_create_entry(
					$meta_args,
					array(
						'post_status' => 'mmp_new_req',
						'post_author' => $post_author,
					)
				);

				/**
				 * Admin ack.
				 *
				 * @since 1.0
				 */
				do_action( 'wc_mmp_admin_ack', $entry_id );

				if ( is_user_logged_in() ) {
					/**
					 * Customer ack.
					 *
					 * @since 1.0
					 */
					do_action( 'wc_mmp_customer_ack', $entry_id );
				} else {
					/**
					 * Customer ack guest.
					 *
					 * @since 1.0
					 */
					do_action( 'wc_mmp_customer_ack_guest', $entry_id );
				}

				wp_send_json_success(
					array(
						'success' => true,
						'layout'  => get_thanks_layout(),
					)
				);
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => $e->getMessage() ) );
			}
		}

		/**
		 * Display MMP list based on pagination
		 *
		 * @since 1.0
		 */
		public static function list_pagination() {
			check_ajax_referer( 'mmp-list-pagination', 'mmp_security' );

			try {
				if ( ! isset( $_POST ) || ! isset( $_POST[ 'page_number' ] ) ) { // @codingStandardsIgnoreLine.
					throw new exception( esc_html__( 'Invalid Request', 'lower-price-for-woocommerce' ) );
				}

				// Sanitize post values
				$html         = '';
				$current_page    = ! empty( $_POST[ 'page_number' ] ) ? absint( $_POST[ 'page_number' ] ) : 0 ; // @codingStandardsIgnoreLine.
				$page_url        = ! empty( $_POST[ 'page_url' ] ) ? wc_clean( wp_unslash( $_POST[ 'page_url' ] ) ) : '' ; // @codingStandardsIgnoreLine.
				$per_page     = 5;
				$offset       = ( $current_page - 1 ) * $per_page;

				// Get gift products based on per page count
				if ( isset( $_POST['is_logged_in'] ) && isset( $_POST['current_user'] ) ) {
					$user        = get_userdata( wc_clean( wp_unslash( $_POST['current_user'] ) ) );
					$user_email  = $user->user_email;
					$product_ids = mmp_is_get_request_by_value_display( $user_email );

					$product_ids = array_slice( $product_ids, $offset, $per_page );
					// Get gift products table body content
					$html = mmp_get_template_html(
						'mmp-price-list-data.php',
						array(
							'product_ids' => $product_ids,
							'permalink'   => esc_url( $page_url ),
						)
					);
				}
				wp_send_json_success( array( 'html' => $html ) );
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) );
			}
		}

		/**
		 * Get Form Fields
		 */
		public static function get_form_fields() {
			check_ajax_referer( 'mmp-form-nonce', 'mmp_security' );

			$variation_id = isset( $_POST['mmp_post_id'] ) ? wc_clean( wp_unslash( $_POST['mmp_post_id'] ) ) : '';

			if ( empty( $variation_id ) ) {
				return;
			}

			ob_start();
			MMP_Handler::mmp_form_fields( $variation_id, 'variable' );
			$contents = ob_get_contents();
			ob_end_clean();

			wp_send_json(
				array(
					'success' => true,
					'content' => $contents,
				)
			);
		}

		/**
		 * Update Viewed Once
		 *
		 * @since 2.3
		 */
		public static function update_viewed_once() {
			check_ajax_referer( 'mmp-viewed-once-nonce', 'mmp_security' );

			try {
				if ( ! isset( $_POST ) || ! isset( $_POST[ 'post_id' ] ) ) { // @codingStandardsIgnoreLine.
					throw new exception( esc_html__( 'Invalid Request', 'lower-price-for-woocommerce' ) );
				}

				$post_id = wc_clean( wp_unslash( $_POST['post_id'] ) );
				update_post_meta( $post_id, 'mmp_viewed_once', 'yes' );

				$unseen_count = mmp_get_product_list_count();

				wp_send_json_success( array( 'unseen_count' => $unseen_count ) );
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) );
			}
		}
	}

	MMP_Admin_Ajax::init();
}
