<?php
/**
 * MMP Post Table
 *
 * @package Lower Price
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Form_Fields_Post_Table' ) ) {

	/**
	 * MMP_Form_Fields_Post_Table Class.
	 */
	class MMP_Form_Fields_Post_Table {
		/*
		 * Object
		 */

		private static $object;
		/*
		 * Post type
		 */
		private static $post_type = MMP_Register_Post_Types::MMP_FORM_FIELDS_POSTTYPE;

		/*
		 * Plugin Slug
		 */
		private static $plugin_slug = 'mmp';

		/**
		 * Class initialization.
		 */
		public static function init() {
			add_action( 'admin_init', array( __CLASS__, 'remove_editor_and_title' ) );
			// body class.
			add_filter( 'admin_body_class', array( __CLASS__, 'custom_body_class' ), 10, 1 );
			add_filter( 'post_row_actions', array( __CLASS__, 'handle_post_row_actions' ), 10, 2 );
			add_filter( 'disable_months_dropdown', array( __CLASS__, 'remove_month_dropdown' ), 10, 2 );
			add_action( 'admin_action_mmp_disabled', array( __CLASS__, 'disable_enable_fields' ) );
			add_action( 'admin_action_mmp_enabled', array( __CLASS__, 'disable_enable_fields' ) );
			add_action( 'views_edit-' . self::$post_type, array( __CLASS__, 'remove_views' ) );
			add_filter( 'bulk_actions-edit-' . self::$post_type, array( __CLASS__, 'handle_bulk_actions' ), 10, 1 );
			add_filter( 'post_row_actions', array( __CLASS__, 'manage_post_row_actions' ), 10, 2 );
			// custom order by.
			add_action( 'posts_orderby', array( __CLASS__, 'manage_orderby_functionality' ), 10, 2 );
			// Adding Meta Boxes.
			add_action( 'add_meta_boxes', array( __CLASS__, 'add_meta_boxes' ), 10, 2 );
			// Post Where.
			add_action( 'posts_where', array( __CLASS__, 'adding_where' ), 10, 2 );
			// define column header.
			add_filter( 'manage_' . self::$post_type . '_posts_columns', array( __CLASS__, 'define_columns' ) );
			// display column value.
			add_action( 'manage_' . self::$post_type . '_posts_custom_column', array( __CLASS__, 'render_columns' ), 10, 2 );
			// define sortable column.
			add_filter( 'manage_edit-' . self::$post_type . '_sortable_columns', array( __CLASS__, 'sortable_columns' ) );
			add_action( 'save_post', array( __CLASS__, 'fields_save_post' ), 10, 2 );
		}

		/**
		 * Save Field.
		 *
		 * @since 1.0
		 * @param Integer $post_id Post ID.
		 */
		public static function fields_save( $post_id ) {
			if ( ! isset( $_POST['mmp_fields_meta_nonce'] ) ) {
				return;
			}

			if ( ! wp_verify_nonce( wc_clean( wp_unslash( $_POST['mmp_fields_meta_nonce'] ) ), '_mmp_save_fields_data' ) ) {
				return;
			}

			$meta_args = array();
			$fields    = array( 'mmp_field_mandatory', 'mmp_field_placeholder' );

			foreach ( $fields as $eachfield ) {
				if ( isset( $_REQUEST[ $eachfield ] ) ) {
					$meta_args[ $eachfield ] = wc_clean( wp_unslash( $_REQUEST[ $eachfield ] ) );
				}
			}

			$post_status = isset( $_POST['mmp_field_status'] ) ? wc_clean( wp_unslash( $_POST['mmp_field_status'] ) ) : 'mmp_disabled'; // phpcs:ignore WordPress.Security.NonceVerification.NoNonceVerification
			$post_label  = isset( $_POST['mmp_field_label'] ) ? wc_clean( wp_unslash( $_POST['mmp_field_label'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.NoNonceVerification

			mmp_update_fields_entry(
				$post_id,
				$meta_args,
				array(
					'post_title'  => $post_label,
					'post_status' => $post_status,
				)
			);
		}

		/**
		 * Save Post.
		 */
		public static function fields_save_post( $post_id, $post ) {
			if ( $post->post_type !== self::$post_type ) {
				return;
			}

			// $post_id and $post are required.
			if ( empty( $post_id ) || empty( $post ) || ! isset( $_POST['mmp_fields_meta_nonce'] ) ) {
				return;
			}

			if ( ! wp_verify_nonce( wc_clean( wp_unslash( $_POST['mmp_fields_meta_nonce'] ) ), '_mmp_save_fields_data' ) ) {
				return;
			}

			// Dont' save meta boxes for revisions or autosaves.
			if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || is_int( wp_is_post_revision( $post ) ) || is_int( wp_is_post_autosave( $post ) ) ) {
				return;
			}

			// Check user has permission to edit.
			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				return;
			}

			remove_action( 'save_post', array( 'MMP_Form_Fields_Post_Table', 'fields_save_post' ), 10, 2 );

			if ( isset( $_POST['mmp_field_placeholder'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.NoNonceVerification
				self::fields_save( $post_id );
			}

			add_action( 'save_post', array( 'MMP_Form_Fields_Post_Table', 'fields_save_post' ), 10, 2 );
		}

		/**
		 * Remove Editor and Title Meta boxes.
		 */
		public static function remove_editor_and_title() {
			$remove_fields = array( 'editor', 'title' );
			foreach ( $remove_fields as $remove_field ) {
				// Remove Supports for Request Post Type.
				remove_post_type_support( self::$post_type, $remove_field );
			}
		}

		/**
		 * Adding MMP meta box
		 */
		public static function add_meta_boxes( $post_type, $post ) {
			if ( self::$post_type != $post_type ) {
				return;
			}
			// Remove post submit metabox
			remove_meta_box( 'submitdiv', self::$post_type, 'side' );

			add_meta_box( 'mmp_field_customization', esc_html__( 'Field Customization', 'lower-price-for-woocommerce' ), array( __CLASS__, 'mmp_field_customization' ), self::$post_type, 'normal' );
		}

		/**
		 *  Order by Functionality
		 */
		public static function manage_orderby_functionality( $order_by, $wp_query ) {
			if ( ! isset( $wp_query->query['post_type'] ) || $wp_query->query['post_type'] != self::$post_type ) {
				return $order_by;
			}

			if ( ! isset( $_REQUEST['post_type'] ) || $_REQUEST['post_type'] != self::$post_type ) {
				return $order_by;
			}

			global $wpdb;
			if ( ! isset( $_REQUEST['order'] ) || ! isset( $_REQUEST['orderby'] ) ) {
				$order_by = "{$wpdb->posts}.menu_order ASC";
			}

			return $order_by;
		}

		/*
		 * Add custom class in body
		 */

		public static function mmp_field_customization( $post ) {
			$cfield_data = mmp_get_fields_entry( $post->ID );
			?> 
			<div>
				<?php
				if ( ! in_array( $cfield_data->get_field_name(), array( 'mmp_name', 'mmp_mail', 'mmp_description', 'mmp_pref_price' ) ) ) {
					?>
					<p class="form-field">
						<strong><?php esc_html_e( 'Field Status', 'lower-price-for-woocommerce' ); ?></strong>  
						<select id="mmp_field_status" name="mmp_field_status"> 
							<option value="mmp_enabled" <?php selected( 'mmp_enabled', $cfield_data->get_status() ); ?>><?php esc_html_e( 'Enabled', 'lower-price-for-woocommerce' ); ?></option>
							<option value="mmp_disabled" <?php selected( 'mmp_disabled', $cfield_data->get_status() ); ?>><?php esc_html_e( 'Disabled', 'lower-price-for-woocommerce' ); ?></option>
						</select> 
					</p>
					<?php
				} else {
					?>
					<input type="hidden" name="mmp_field_status" value="<?php echo esc_attr( $cfield_data->get_status() ); ?>" >
					<?php
				}
				if ( ! in_array( $cfield_data->get_field_name(), array( 'mmp_name', 'mmp_mail', 'mmp_description', 'mmp_pref_price' ) ) ) {
					?>
					<p class="form-field">
						<strong><?php esc_html_e( 'Field Type', 'lower-price-for-woocommerce' ); ?></strong>  
						<select id="mmp_field_mandatory" name="mmp_field_mandatory">
							<option value="no" <?php selected( 'no', $cfield_data->get_field_mandatory() ); ?>><?php esc_html_e( 'Optional', 'lower-price-for-woocommerce' ); ?></option>
							<option value="yes" <?php selected( 'yes', $cfield_data->get_field_mandatory() ); ?>><?php esc_html_e( 'Mandatory', 'lower-price-for-woocommerce' ); ?></option>
						</select>
					</p>
			<?php } ?>
				<p class="form-field">
					<strong><?php esc_html_e( 'Label', 'lower-price-for-woocommerce' ); ?></strong>  
					<input type="text" name="mmp_field_label" value="<?php echo esc_html( $cfield_data->get_field_label() ); ?>">
				</p>
				<p class="form-field">
					<strong><?php esc_html_e( 'Placeholder', 'lower-price-for-woocommerce' ); ?></strong>  
					<input type="text" name="mmp_field_placeholder" value="<?php echo esc_html( $cfield_data->get_field_placeholder() ); ?>">
				</p>
				<?php
				wp_nonce_field( '_mmp_save_fields_data', 'mmp_fields_meta_nonce' );
				submit_button( __( 'Save', 'lower-price-for-woocommerce' ), 'primary large', 'submit', false );
				?>
			</div>
			<?php
		}

		/*
		 * Add custom class in body
		 */

		public static function custom_body_class( $class ) {
			global $post;

			if ( ! is_object( $post ) ) {
				return $class;
			}

			if ( $post->post_type == self::$post_type ) {
				return $class . ' mmp_body_content';
			}
			return $class;
		}

		/*
		 * Handle Row Actions
		 */

		public static function handle_post_row_actions( $actions, $post ) {

			if ( $post->post_type == self::$post_type ) {
				return array();
			}

			return $actions;
		}

		/*
		 * Remove views
		 */

		public static function remove_views( $views ) {
			return array();
		}

		/**
		 * Remove month dropdown
		 */
		public static function remove_month_dropdown( $bool, $post_type ) {
			return $post_type == self::$post_type ? true : $bool;
		}

		/*
		 * Handle Bulk Actions
		 */

		public static function handle_bulk_actions( $actions ) {
			global $post;
			if ( $post->post_type == self::$post_type ) {
				return array();
			}

			return $actions;
		}

		/**
		 * Define custom columns
		 */
		public static function define_columns( $columns ) {

			if ( ! mmp_check_is_array( $columns ) ) {
				$columns = array();
			}

			$columns = array(
				'mmp_fname'   => esc_html__( 'Field Name', 'lower-price-for-woocommerce' ),
				'mmp_fstatus' => esc_html__( 'Field Status', 'lower-price-for-woocommerce' ),
				'mmp_ftype'   => esc_html__( 'Field Type', 'lower-price-for-woocommerce' ),
				'mmp_fsort'   => esc_html__( 'Sort', 'lower-price-for-woocommerce' ),
			);

			return $columns;
		}

		/*
		 * Remove views
		 */

		public static function prepare_row_data( $postid ) {
			if ( empty( self::$object ) || self::$object->get_id() != $postid ) {
				self::$object = mmp_get_fields_entry( $postid );
			}

			return self::$object;
		}

		/**
		 * Render each column
		 */
		public static function render_columns( $column, $postid ) {
			self::prepare_row_data( $postid );
			$function = 'render_' . $column . '_cloumn';

			if ( method_exists( __CLASS__, $function ) ) {
				self::$function();
			}
		}

		/**
		 * Render User Details column
		 */
		public static function render_mmp_fname_cloumn() {
			echo esc_html( self::$object->get_field_label() );
		}

		/**
		 * Render Total MMP column
		 */
		public static function render_mmp_fstatus_cloumn() {

			return mmp_display_status( self::$object->get_status() );
		}

		/**
		 * Render Current MMP column
		 */
		public static function render_mmp_ftype_cloumn() {
			if ( 'yes' == self::$object->get_field_mandatory() ) {
				echo esc_html__( 'Mandatory', 'lower-price-for-woocommerce' );
			} else {
				echo esc_html__( 'Optional', 'lower-price-for-woocommerce' );
			}
		}

		/**
		 * Render Purchased MMP column
		 */
		public static function render_mmp_fsort_cloumn() {
			echo wp_kses_post( "<div class='mmp_post_sort_handle'><img src=" . esc_attr( MMP_PLUGIN_URL . '\assets\images\drag-icon.png' ) . '></div>' );
		}

		/**
		 * Disable / Enable Fields
		 */
		public static function disable_enable_fields() {
			if ( empty( $_REQUEST['post'] ) ) {
				wp_die( esc_html__( 'No Fields to Disable/Enable!', 'lower-price-for-woocommerce' ) );
			}

			// Get post id
			$post_id = isset( $_REQUEST['post'] ) ? absint( $_REQUEST['post'] ) : '';

			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				wp_die( esc_html__( 'Sorry, you are not allowed to edit this item.', 'lower-price-for-woocommerce' ) );
			}

			check_admin_referer( 'mmp-action-' . $post_id ); // Check nonce

			$post_id = absint( $post_id );

			if ( ! $post_id || ! isset( $_REQUEST['action'] ) ) {
				return false;
			}

			$args = array(
				'ID'          => $post_id,
				'post_status' => wc_clean( wp_unslash( $_REQUEST['action'] ) ),
			);
			wp_update_post( $args );

			if ( isset( $_SERVER['HTTP_REFERER'] ) ) {
				$url = esc_url_raw( wc_clean( $_SERVER['HTTP_REFERER'] ) );
				wp_redirect( $url );
				exit();
			}
		}

		/**
		 * Managing the Row actions
		 */
		public static function manage_post_row_actions( $actions, $post ) {
			if ( $post->post_type != self::$post_type ) {
				return $actions;
			}

			unset( $actions ); // unset previous actions
			$post_status  = ( 'mmp_enabled' == $post->post_status ) ? 'mmp_disabled' : 'mmp_enabled';
			$edit_url     = add_query_arg(
				array(
					'post'   => $post->ID,
					'action' => 'edit',
				),
				admin_url( 'post.php' )
			);
			$status_url   = wp_nonce_url(
				add_query_arg(
					array(
						'post'   => $post->ID,
						'action' => $post_status,
					),
					admin_url( 'post.php' )
				),
				'mmp-action-' . $post->ID
			);
			$status_label = ( 'mmp_enabled' == $post->post_status ) ? __( 'Disable', 'lower-price-for-woocommerce' ) : __( 'Enable', 'lower-price-for-woocommerce' );
			$status       = ( 'mmp_enabled' == $post->post_status ) ? 'mmp_disabled' : 'mmp_enabled';
			$new_actions  = array();
			// create new actions
			$new_actions['edit'] = sprintf( '<a href=%s>%s</a>', $edit_url, __( 'Edit', 'lower-price-for-woocommerce' ) );

			if ( ! in_array( self::$object->get_field_name(), array( 'mmp_name', 'mmp_mail', 'mmp_description', 'mmp_pref_price' ) ) ) {
				$new_actions[ $status ] = sprintf( '<a href=%s>%s</a>', $status_url, $status_label );
			}

			return $new_actions;
		}

		/**
		 * Sortable columns
		 */
		public static function sortable_columns( $columns ) {

			$columns = array(
				'mmp_fname' => 'mmp_fname',
			);

			return $columns;
		}

		/**
		 * Sortable columns
		 */
		public static function orderby_columns( $order_by, $wp_query ) {
			global $wpdb;
			if ( isset( $wp_query->query['post_type'] ) && $wp_query->query['post_type'] == self::$post_type ) {
				if ( ! isset( $_REQUEST['order'] ) && ! isset( $_REQUEST['orderby'] ) ) {
					$order_by = "{$wpdb->posts}.ID " . 'DESC';
				} else {
					$decimal_column = array( 'hrw_available_balance', 'hrw_total_balance' );

					if ( in_array( sanitize_title( wp_unslash( $_REQUEST['orderby'] ) ), $decimal_column ) ) {
						$order_by = "CAST({$wpdb->postmeta}.meta_value AS DECIMAL) " . sanitize_title( wp_unslash( $_REQUEST['order'] ) );
					} elseif ( sanitize_title( wp_unslash( $_REQUEST['orderby'] ) ) == 'post_status' ) {
						$order_by = "{$wpdb->posts}.post_status " . sanitize_title( wp_unslash( $_REQUEST['order'] ) );
					} else {
						$order_by = "{$wpdb->postmeta}.meta_value " . sanitize_title( wp_unslash( $_REQUEST['order'] ) );
					}
				}
			}

			return $order_by;
		}

		/**
		 *  Sorting Functionality
		 */
		public static function orderby_filter_query( $query ) {
			if ( isset( $_REQUEST['post_type'] ) && sanitize_title( wp_unslash( $_REQUEST['post_type'] ) ) == self::$post_type && self::$post_type == $query->query['post_type'] ) {
				if ( isset( $_GET['orderby'] ) ) {
					$excerpt_array = array( 'ID', 'post_status' );
					if ( ! in_array( sanitize_title( wp_unslash( $_GET['orderby'] ) ), $excerpt_array ) ) {
						$query->query_vars['meta_key'] = sanitize_title( wp_unslash( $_GET['orderby'] ) );
					}
				}
			}
		}

		/**
		 * Distinct Functionality
		 *
		 * @since 1.0.0
		 * @param String $distinct Distinct.
		 * @return String
		 */
		public static function distinct_post( $distinct ) {
			$post_type = isset( $_REQUEST['post_type'] ) ? wc_clean( wp_unslash( $_REQUEST['post_type'] ) ) : '';

			if ( self::$post_type !== $post_type ) {
				return $distinct;
			}

			if ( isset( $_REQUEST['s'] ) || isset( $_REQUEST['orderby'] ) ) {
				return 'DISTINCT';
			}

			return $distinct;
		}

		public static function adding_where( $where ) {
			global $wpdb, $wp_query;

			if ( isset( $wp_query->query_vars['post_type'] ) && $wp_query->query_vars['post_type'] == self::$post_type ) {
				$where .= " OR {$wpdb->posts}.post_status IN ('mmp_disabled','mmp_enabled')";
			}

			return $where;
		}
	}

	MMP_Form_Fields_Post_Table::init();
}
