<?php
/**
 * MMP Post Table
 *
 * @package Lower Price
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Post_Table' ) ) {

	/**
	 * MMP_Post_Table Class.
	 */
	class MMP_Post_Table {
		/**
		 * Object
		 *
		 * @var Object
		 */
		private static $object;

		/**
		 * Object
		 *
		 * @var Object
		 */
		private static $pro_object;

		/**
		 * Post type
		 *
		 * @var String
		 */
		private static $post_type = MMP_Register_Post_Types::MMP_PRODUCT_POSTTYPE;

		/**
		 * Plugin Slug
		 *
		 * @var String
		 */
		private static $plugin_slug = 'mmp';

		/**
		 * Class initialization.
		 *
		 * @since 1.0
		 */
		public static function init() {
			add_action( 'admin_init', array( __CLASS__, 'remove_editor_and_title' ) );
			// body class.
			add_filter( 'admin_body_class', array( __CLASS__, 'custom_body_class' ), 10, 1 );
			add_filter( 'post_row_actions', array( __CLASS__, 'handle_post_row_actions' ), 10, 2 );
			add_filter( 'disable_months_dropdown', array( __CLASS__, 'remove_month_dropdown' ), 10, 2 );
			add_filter( 'bulk_actions-edit-' . self::$post_type, array( __CLASS__, 'handle_bulk_actions' ), 10, 1 );
			add_action( 'admin_action_mmp_delete', array( __CLASS__, 'delete_request' ) );
			// Adding Meta Boxes.
			add_action( 'add_meta_boxes', array( __CLASS__, 'add_meta_boxes' ), 10, 2 );
			add_filter( 'views_edit-' . self::$post_type, array( __CLASS__, 'get_views' ), 10, 1 );

			// Add Search Action Query.
			add_filter( 'posts_search', array( __CLASS__, 'search_action' ) );
			// Table Join.
			add_action( 'posts_join', array( __CLASS__, 'table_join_query' ), 10, 2 );
			// Display result column value in unique.
			add_action( 'posts_distinct', array( __CLASS__, 'distinct_post' ), 10, 2 );
			add_action( 'posts_where', array( __CLASS__, 'adding_where' ), 10, 2 );

			// Define column header.
			add_filter( 'manage_' . self::$post_type . '_posts_columns', array( __CLASS__, 'define_columns' ) );
			// Display column value.
			add_action( 'manage_' . self::$post_type . '_posts_custom_column', array( __CLASS__, 'render_columns' ), 10, 2 );
			// Define sortable column.
			add_filter( 'manage_edit-' . self::$post_type . '_sortable_columns', array( __CLASS__, 'sortable_columns' ) );
			add_action( 'save_post', array( __CLASS__, 'request_actions' ), 10, 2 );
		}

		/**
		 * Add custom class in body
		 *
		 * @since 1.0.0
		 * @param string $class Class name.
		 */
		public static function custom_body_class( $class ) {
			global $post;

			if ( ! is_object( $post ) ) {
				return $class;
			}

			if ( $post->post_type === self::$post_type ) {
				return $class . ' mmp_body_content';
			}

			return $class;
		}

		/**
		 * Handle Row Actions
		 *
		 * @since 1.0.0
		 * @param Array   $actions Actions.
		 * @param WC_Post $post Post object.
		 */
		public static function handle_post_row_actions( $actions, $post ) {
			if ( $post->post_type === self::$post_type ) {
				return array();
			}

			return $actions;
		}

		/**
		 * Get the list of views available on this table.
		 *
		 * @since 1.0.0
		 * @param Array $views Views.
		 * @return array
		 */
		public static function get_views( $views ) {
			global $wpdb, $post_id;

			$wpdb_ref     = &$wpdb;
			$status_links = array();
			$status_array = array(
				'all' => esc_html__( 'All', 'lower-price-for-woocommerce' ),
			);

			foreach ( $status_array as $status_key => $status_label ) {
				$status_count = $wpdb_ref->get_var( $wpdb_ref->prepare( "SELECT count(DISTINCT ID) FROM {$wpdb->posts} WHERE post_type=%s and post_status IN ('mmp_new_req' , 'mmp_coupon_sent' , 'mmp_coupon_used' , 'mmp_coupon_expired' , 'mmp_rejected')", esc_sql( self::$post_type ) ) );
				$class        = '';

				if ( isset( $_REQUEST['status'] ) && sanitize_text_field( wp_unslash( $_REQUEST['status'] ) ) === $status_key ) {
					$class = ' class="current"';
				}

				$label = array(
					'singular' => sprintf( '%s <span class="count">(%s)</span>', esc_html( $status_label ), $status_count ),
					'plural'   => sprintf( '%s <span class="count">(%s)</span>', esc_html( $status_label ), $status_count ),
					'context'  => '',
					'domain'   => 'lower-price-for-woocommerce',
				);

				$views[ $status_key ] = sprintf(
					"<a href='%s' %s>%s</a>",
					add_query_arg(
						array(
							'post_type' => self::$post_type,
							'status'    => $status_key,
						),
						admin_url( 'edit.php' )
					),
					$class,
					translate_nooped_plural( $label, $status_count )
				);
			}

			return $views;
		}

		/**
		 * Remove month dropdown
		 *
		 * @since 1.0.0
		 * @param Boolean $bool Boolean.
		 * @param String  $post_type Post type.
		 */
		public static function remove_month_dropdown( $bool, $post_type ) {
			return $post_type === self::$post_type ? true : $bool;
		}

		/**
		 * Handle Bulk Actions
		 *
		 * @since 1.0.0
		 * @param Array $actions Array of actions.
		 */
		public static function handle_bulk_actions( $actions ) {
			global $post;

			if ( $post->post_type === self::$post_type ) {
				$actions           = array();
				$actions['delete'] = __( 'Delete' );

				return $actions;
			}

			return $actions;
		}

		/**
		 * Define custom columns
		 *
		 * @since 1.0.0
		 * @param Array $columns Columns.
		 */
		public static function define_columns( $columns ) {
			if ( ! mmp_check_is_array( $columns ) ) {
				$columns = array();
			}

			$columns = array(
				'cb'                => $columns['cb'],
				'mmp_user_name'     => esc_html__( 'Username', 'lower-price-for-woocommerce' ),
				'mmp_pro_name'      => esc_html__( 'Product Name', 'lower-price-for-woocommerce' ),
				'mmp_org_price'     => esc_html__( 'Original Price', 'lower-price-for-woocommerce' ),
				'mmp_pref_price'    => esc_html__( 'Reported Price', 'lower-price-for-woocommerce' ),
				'mmp_off_price'     => esc_html__( 'Offered Price', 'lower-price-for-woocommerce' ),
				'mmp_req_mode'      => esc_html__( 'Mode of Operation', 'lower-price-for-woocommerce' ),
				'mmp_status'        => esc_html__( 'Status', 'lower-price-for-woocommerce' ),
				'mmp_coupon_expiry' => esc_html__( 'Coupon Expiry Date', 'lower-price-for-woocommerce' ),
				'mmp_lst_activity'  => esc_html__( 'Last Activity Date', 'lower-price-for-woocommerce' ),
				'mmp_actions'       => esc_html__( 'Actions', 'lower-price-for-woocommerce' ),
			);

			return $columns;
		}

		/**
		 * Remove views
		 *
		 * @since 1.0.0
		 * @param Integer $post_id Post ID.
		 */
		public static function prepare_row_data( $postid ) {
			self::$object     = mmp_get_entry( $postid );
			self::$pro_object = wc_get_product( self::$object->get_pro_id() );

			return self::$object;
		}

		/**
		 * Render each column
		 *
		 * @since 1.0.0
		 * @param String  $column Column name.
		 * @param Integer $post_id Post ID.
		 */
		public static function render_columns( $column, $post_id ) {
			self::prepare_row_data( $post_id );

			$function = 'render_' . $column . '_cloumn';

			if ( method_exists( __CLASS__, $function ) ) {
				self::$function();
			}
		}

		/**
		 * Render User Details column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_user_name_cloumn() {
			echo wp_kses_post( sprintf( '%1s(%2s)', self::$object->get_name(), self::$object->get_mail() ) );
		}

		/**
		 * Render User Details column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_pro_name_cloumn() {
			if ( wc_get_product( self::$object->get_pro_id() ) ) {
				if ( get_edit_post_link( self::$object->get_pro_id() ) ) {
					$product_link = get_edit_post_link( self::$object->get_pro_id() );
				}

				if ( wc_get_product( self::$object->get_pro_id() )->get_parent_id() ) {
					$parent_id    = wc_get_product( self::$object->get_pro_id() )->get_parent_id();
					$product_link = get_edit_post_link( $parent_id );
				}

				echo wp_kses_post( sprintf( '<a target="blank" href="%1s">%2s</a>', $product_link, self::$pro_object->get_name() ) );
			} else {
				mmp_delete_entry( self::$object->get_id() );
				echo esc_html__( 'NA', 'lower-price-for-woocommerce' );
			}
		}

		/**
		 * Render Total MMP column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_org_price_cloumn() {
			if ( self::$object->get_org_price() ) {
				echo wp_kses_post( wc_price( self::$object->get_org_price() ) );
			} else {
				echo esc_html__( 'NA', 'lower-price-for-woocommerce' );
			}
		}

		/**
		 * Render Current MMP column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_pref_price_cloumn() {
			mmp_price( self::$object->get_pref_price(), true );
		}

		/**
		 * Render Purchased MMP column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_off_price_cloumn() {
			echo wp_kses_post( self::$object->get_off_price_formatted() );
		}

		/**
		 * Render Purchased MMP column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_status_cloumn() {
			mmp_display_status( self::$object->get_status_formatted() );

			if ( 'no' === self::$object->get_viewed_once() ) {
				echo '<br/><span class="mmp-new-status-label">' . esc_html__( 'New', 'lower-price-for-woocommerce' ) . '</span>';
			}
		}

		/**
		 * Render Purchased MMP column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_req_mode_cloumn() {
			echo esc_html( self::$object->get_req_mode() );
		}

		/**
		 * Render Last Activity column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_coupon_expiry_cloumn() {
			if ( 'mmp_coupon_used' === self::$object->get_status() ) {
				echo '-';
			} else {
				echo wp_kses_post( mmp_is_empty_check( self::$object->get_coupon_expiry() ) );
			}
		}

		/**
		 * Render Last Activity column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_lst_activity_cloumn() {
			echo esc_html( mmp_is_empty_check( self::$object->get_formatted_activity_date() ) );
		}

		/**
		 * Render Actions column
		 *
		 * @since 1.0.0
		 */
		public static function render_mmp_actions_cloumn() {
			$allowed_html = array(
				'a' => array(
					'href'  => array(),
					'class' => array(),
				),
			);

			echo wp_kses(
				sprintf(
					'<span class="mmp_view"><a href="%s" class="%s">%s</a></span>',
					esc_url( get_edit_post_link( self::$object->get_id() ) ),
					esc_attr( 'edit_view' ),
					esc_html__( 'View', 'lower-price-for-woocommerce' )
				),
				$allowed_html
			);

			echo wp_kses(
				sprintf(
					'<span class="mmp_reject">   <a class="mmp_entry_delete" href="%s" >%s</a></span>',
					esc_url( mmp_get_base_url( self::$object->get_id(), 'mmp_delete' ) ),
					esc_html__( 'Delete', 'lower-price-for-woocommerce' )
				),
				$allowed_html
			);
		}

		/**
		 * Deleting The Request
		 *
		 * @since 1.0.0
		 */
		public static function delete_request() {
			if ( empty( $_REQUEST['post'] ) ) {
				wp_die( esc_html__( 'No Fields to Disable/Enable!', 'lower-price-for-woocommerce' ) );
			}

			// Get post id.
			$post_id = isset( $_REQUEST['post'] ) ? absint( $_REQUEST['post'] ) : '';

			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				wp_die( esc_html__( 'Sorry, you are not allowed to edit this item.', 'lower-price-for-woocommerce' ) );
			}

			// check_admin_referer( 'mmp-action-' . $post_id ) ; //Check nonce

			$post_id = absint( $post_id );

			if ( ! $post_id ) {
				return false;
			}

			wp_delete_post( $post_id );

			if ( isset( $_SERVER['HTTP_REFERER'] ) ) {
				$url = esc_url_raw( $_SERVER['HTTP_REFERER'], true );

				wp_safe_redirect( $url );
				exit();
			}
		}

		/**
		 * Save Post - Request Actions
		 */
		public static function request_actions( $post_id, $post ) {
			if ( $post->post_type !== self::$post_type ) {
				return;
			}

			// $post_id and $post are required.
			if ( empty( $post_id ) || empty( $post ) || ! isset( $_POST['mmp_req_meta_nonce'] ) ) {
				return;
			}

			if ( ! wp_verify_nonce( wc_clean( wp_unslash( $_POST['mmp_req_meta_nonce'] ) ), 'mmp_save_req_data' ) ) {
				return;
			}

			// Don't save meta boxes for revisions or autosaves.
			if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || is_int( wp_is_post_revision( $post ) ) || is_int( wp_is_post_autosave( $post ) ) ) {
				return;
			}

			// Check user has permission to edit.
			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				return;
			}

			remove_action( 'save_post', array( 'MMP_Post_Table', 'request_actions' ), 10, 2 );

			if ( isset( $_POST['mmp_reson_reject'] ) ) {
				self::request_reject( $post_id );
			}

			if ( isset( $_POST['mmp_accept_mail'] ) && isset( $_POST['mmp_offered_price'] ) && isset( $_POST['mmp_product_price'] ) ) {
				$offered_price = wc_clean( wp_unslash( $_POST['mmp_offered_price'] ) );
				$product_price = wc_clean( wp_unslash( $_POST['mmp_product_price'] ) );
				$coupon_amount = $product_price - $offered_price;

				self::coupon_mail_process( $post_id, $coupon_amount, $offered_price );
			}

			add_action( 'save_post', array( 'MMP_Post_Table', 'request_actions' ), 10, 2 );
		}

		/**
		 * Coupon Mail Process
		 *
		 * @since 1.0.0
		 * @param Integer $post_id Post identifier.
		 * @param Float   $amount Amount.
		 * @param Float   $offered_price Amount.
		 */
		public static function coupon_mail_process( $post_id, $amount, $offered_price ) {
			if ( ! isset( $_POST['mmp_req_meta_nonce'] ) ) {
				return;
			}

			if ( ! wp_verify_nonce( wc_clean( wp_unslash( $_POST['mmp_req_meta_nonce'] ) ), 'mmp_save_req_data' ) ) {
				return;
			}

			// create coupon.
			$coupon_title = self::mmp_create_coupon( $post_id, $amount );
			$request_type = isset( $_POST['mmp_request_type'] ) ? wc_clean( wp_unslash( $_POST['mmp_request_type'] ) ) : 'mmp_feedback';
			$reason       = isset( $_POST['mmp_reject_reason'] ) ? wc_clean( wp_unslash( $_POST['mmp_reject_reason'] ) ) : '';

			// Accept status update.
			mmp_update_entry(
				$post_id,
				array(
					'mmp_off_price'     => $offered_price,
					'mmp_accepted_date' => current_time( 'mysql', true ),
					'mmp_lst_activity'  => current_time( 'mysql', true ),
					'mmp_coupon_tilte'  => $coupon_title,
					'mmp_reject_reason' => $reason,
				),
				array( 'post_status' => 'mmp_coupon_sent' )
			);

			if ( 'mmp_auto_apply' === $request_type ) {
				/**
				 * Coupon apply.
				 *
				 * @since 1.0
				 */
				do_action( 'wc_mmp_coupon_apply', $post_id );
			} else {
				/**
				 * Sending coupon.
				 *
				 * @since 1.0
				 */
				do_action( 'wc_mmp_customer_sending_coupon', $post_id );
			}
		}

		/**
		 * Request Reject
		 *
		 * @since 1.0.0
		 * @param Integer $post_id Post ID.
		 */
		public static function request_reject( $post_id ) {
			if ( ! isset( $_POST['mmp_req_meta_nonce'] ) ) {
				return;
			}

			if ( ! wp_verify_nonce( wc_clean( wp_unslash( $_POST['mmp_req_meta_nonce'] ) ), 'mmp_save_req_data' ) ) {
				return;
			}

			$reson_reject    = isset( $_POST['mmp_reject_reason'] ) ? wc_clean( wp_unslash( $_POST['mmp_reject_reason'] ) ) : '';
			$mmp_post_author = isset( $_POST['mmp_post_author'] ) ? wc_clean( wp_unslash( $_POST['mmp_post_author'] ) ) : '';
			$get_user_type   = isset( $_POST['mmp_user_type'] ) ? wc_clean( wp_unslash( $_POST['mmp_user_type'] ) ) : '';

			// Reject status update.
			mmp_update_entry(
				$post_id,
				array(
					'mmp_rejected_date' => current_time( 'mysql', true ),
					'mmp_lst_activity'  => current_time( 'mysql', true ),
					'mmp_reject_reason' => $reson_reject,
				),
				array( 'post_status' => 'mmp_rejected' )
			);

			if ( 'user' === $get_user_type ) {
				/**
				 * Customer rejected.
				 *
				 * @since 1.0
				 */
				do_action( 'wc_mmp_customer_rejected', $post_id );
			} else {
				/**
				 * Customer rejected guest.
				 *
				 * @since 1.0
				 */
				do_action( 'wc_mmp_customer_rejected_guest', $post_id );
			}
		}

		/**
		 * Create Coupons
		 *
		 * @since 1.0.0
		 * @param Integer $post_id Post ID.
		 * @param Float   $amount Amount.
		 */
		public static function mmp_create_coupon( $post_id, $amount ) {
			$req_data           = mmp_get_entry( $post_id );
			$expiry_date        = gmdate( 'd-m-Y H:i:s', strtotime( '+' . get_option( 'mmp_general_coupon_validity', 30 ) . ' days' ) );
			$coupon_expiry_date = new WC_DateTime( $expiry_date, new DateTimeZone( 'UTC' ) );

			$coupon_expiry_date->setTimezone( new DateTimeZone( wc_timezone_string() ) );

			$coupon_title  = mmp_code_generation();
			$coupon        = array(
				'post_title'   => $coupon_title,
				'post_content' => '',
				'post_status'  => 'publish',
				'post_author'  => 1,
				'post_type'    => 'shop_coupon',
			);
			$new_coupon_id = wp_insert_post( $coupon );

			// Add meta.
			update_post_meta( $new_coupon_id, 'discount_type', 'fixed_product' );
			update_post_meta( $new_coupon_id, 'coupon_amount', $amount );
			update_post_meta( $new_coupon_id, 'product_ids', $req_data->get_pro_id() );
			update_post_meta( $new_coupon_id, 'customer_email', $req_data->get_mail() );
			update_post_meta( $new_coupon_id, 'usage_count', 0 );
			update_post_meta( $new_coupon_id, 'usage_limit', 1 );
			update_post_meta( $new_coupon_id, 'expiry_date', $coupon_expiry_date->date( 'd-m-Y H:i:s' ) );
			update_post_meta( $new_coupon_id, 'apply_before_tax', 'yes' );
			update_post_meta( $new_coupon_id, 'free_shipping', 'no' );

			if ( 'yes' === get_option( 'mmp_general_indvidual_use', 'yes' ) ) {
				update_post_meta( $new_coupon_id, 'individual_use', 'yes' );
			} else {
				update_post_meta( $new_coupon_id, 'individual_use', '' );
			}

			return $coupon_title;
		}

		/**
		 * Remove Editor and Title Meta boxes.
		 *
		 * @since 1.0.0
		 */
		public static function remove_editor_and_title() {
			$remove_fields = array( 'editor', 'title' );

			foreach ( $remove_fields as $remove_field ) {
				// Remove Supports for Request Post Type.
				remove_post_type_support( self::$post_type, $remove_field );
			}
		}

		/**
		 * Adding MMP meta box
		 *
		 * @since 1.0.0
		 * @param String  $post_type Post type.
		 * @param WC_Post $post Post object.
		 */
		public static function add_meta_boxes( $post_type, $post ) {
			if ( self::$post_type !== $post_type ) {
				return;
			}

			// Remove post submit metabox.
			remove_meta_box( 'submitdiv', self::$post_type, 'side' );

			add_meta_box( 'mmp_request_details', esc_html__( 'Request Details', 'lower-price-for-woocommerce' ), array( __CLASS__, 'mmp_request_details' ), self::$post_type, 'normal' );
		}

		/**
		 * Add Request Related Tables
		 *
		 * @since 1.0.0
		 * @param Object $post Post object.
		 */
		public static function mmp_request_details( $post ) {
			self::prepare_row_data( $post->ID );

			// User details.
			self::get_user_details( $post );
			// Product details.
			self::get_product_details();
			// Other details.
			self::get_other_details();

			if ( in_array( self::$object->get_status(), array( 'mmp_coupon_sent', 'mmp_coupon_used', 'mmp_coupon_expired' ), true ) ) {
				// Process Summary.
				self::get_process_summary();
			}

			if ( 'mmp_new_req' !== self::$object->get_status() ) {
				self::get_reason_details();
			}

			wp_nonce_field( 'mmp_save_req_data', 'mmp_req_meta_nonce' );
		}

		/**
		 * Process Summary
		 *
		 * @since 1.0
		 */
		public static function get_process_summary() {
			if ( ! in_array( self::$object->get_status(), array( 'mmp_rejected', 'mmp_coupon_send', 'mmp_feedback' ), true ) ) {
				?>
				<div class="mmp-request-table">
					<h2><?php echo esc_html__( 'Request Summary', 'lower-price-for-woocommerce' ); ?></h2>
					<table>
						<tr>
							<th><?php echo esc_html__( 'S.NO', 'lower-price-for-woocommerce' ); ?></th>
							<th><?php echo esc_html__( 'SUMMARY', 'lower-price-for-woocommerce' ); ?></th>
							<th><?php echo esc_html__( 'DATE', 'lower-price-for-woocommerce' ); ?></th>
						</tr>
						<?php if ( self::$object->get_accepted_date() ) { ?>
							<tr>
								<td style="width:10%;padding:5px;">
									<?php echo esc_html__( '1', 'lower-price-for-woocommerce' ); ?>
								</td>
								<td style="padding-right:5px;">
									<?php
									echo esc_html__( "You have accepted your user's request. Please check the below details,", 'lower-price-for-woocommerce' ) . '<br>';
									/* translators: %s: Coupon Code */
									printf( esc_html__( 'Code - %s', 'lower-price-for-woocommerce' ), esc_attr( self::$object->get_coupon_title() ) ) . '<br>';
									echo esc_html__( 'Status - Not Yet Used The Coupon', 'lower-price-for-woocommerce' ) . '<br>';
									/* translators: %s: Coupon Validity */
									printf( esc_html__( 'Validity - %s', 'lower-price-for-woocommerce' ), esc_attr( self::$object->get_coupon_expiry() ) ) . '<br>';
									?>
								</td>
								<td>
									<?php echo esc_html( self::$object->get_formatted_accepted_date() ); ?>
								</td>
							</tr>
							<?php
							if ( self::$object->get_coupon_used_order_id() ) {
								$order_link = '<a target="blank" href=' . esc_attr( get_edit_post_link( self::$object->get_coupon_used_order_id() ) ) . '>' . self::$object->get_coupon_used_order_id() . '</a>';
								?>
								<tr>
									<td style="width:10%;padding:5px;">
										<?php echo esc_html( '2' ); ?>
									</td>
									<td style="padding-right:5px;" >
										<?php echo wp_kses_post( 'The Coupon code <b>' . esc_attr( self::$object->get_coupon_title() ) . '</b> has been successfully used in the order ' . $order_link, 'lower-price-for-woocommerce' ) . '<br>'; ?>
									</td>
									<td>
										<?php echo get_the_date( '', self::$object->get_coupon_used_order_id() ); ?>
									</td>
								</tr>
								<?php
							}
							if ( ! mmp_is_coupon_valid( self::$object->get_coupon_expiry() ) ) {
								?>
								<tr>
									<td style="width:10%;padding:5px;">
										<?php echo esc_html( '3' ); ?>
									</td>
									<td style="padding-right:5px;" >
										<?php echo wp_kses_post( 'The Coupon code <b>' . esc_html__( self::$object->get_coupon_title() ) . '</b> has been expired.', 'lower-price-for-woocommerce' ) . '<br>'; ?>
									</td>
									<td>
										<?php echo esc_html( self::$object->get_coupon_expiry() ); ?>
									</td>
								</tr>
							<?php } ?>
						<?php } ?>
					</table>
				</div>
				<?php
			}
		}

		/**
		 * Product Details
		 *
		 * @since 1.0.0
		 */
		public static function get_product_details() {
			?>
			<div class="mmp-request-table">
				<h2><?php echo esc_html__( 'Product Details', 'lower-price-for-woocommerce' ); ?></h2>
				<table>
					<tr>
						<th><?php echo esc_html__( 'Product Name', 'lower-price-for-woocommerce' ); ?></th>
						<td><?php echo wp_kses_post( sprintf( '<a target="blank" href="%1s">%2s</a>', get_edit_post_link( self::$object->get_pro_id() ), self::$pro_object->get_name() ) ); ?></td>
					</tr>
					<tr>
						<th><?php echo esc_html__( 'Original Price', 'lower-price-for-woocommerce' ); ?></th>
						<td>
							<?php
							if ( ! empty( self::$object->get_org_price() ) ) {
								mmp_price( self::$object->get_org_price() );
							} else {
								echo esc_html__( 'NA', 'lower-price-for-woocommerce' );
							}
							?>
						</td>

					</tr>
					<tr>
						<th><?php echo esc_html__( 'Reported Price', 'lower-price-for-woocommerce' ); ?></th>
						<td><?php mmp_price( self::$object->get_pref_price() ); ?></td>
					</tr>
					<?php if ( in_array( self::$object->get_status(), array( 'mmp_coupon_sent', 'mmp_coupon_used', 'mmp_coupon_expired' ), true ) ) { ?>
						<tr>
							<th><?php echo esc_html__( 'Offered Price', 'lower-price-for-woocommerce' ); ?></th> 
							<td>
								<span><?php echo wp_kses_post( self::$object->get_off_price_formatted() ); ?></span>
						</tr>
					<?php } ?>
					<?php if ( in_array( self::$object->get_request_type(), array( 'mmp_coupon_send', 'mmp_auto_apply' ), true ) && 'mmp_new_req' === self::$object->get_status() ) { ?>
						<tr>
							<th><?php echo esc_html__( 'Your Offer Price', 'lower-price-for-woocommerce' ); ?></th> 
							<td>
								<input type='hidden' name='mmp_product_price' value="<?php echo esc_attr( self::$pro_object->get_price() ); ?>" >
								<input type='hidden' name='mmp_request_type' value="<?php echo esc_attr( self::$object->get_request_type() ); ?>" >
								<input type='text' name='mmp_offered_price' id='mmp_offered_price' > 
							</td>
						</tr>
						<tr>
							<th><?php echo esc_html__( 'Reason', 'lower-price-for-woocommerce' ); ?></th>
							<td><textarea name='mmp_reject_reason'></textarea></td> 
						</tr>
						<tr>
							<td colspan="2">
								<input type='hidden' name='mmp_user_type' value="<?php echo esc_attr( self::$object->get_user_type() ); ?>" >
								<input type='hidden' name='mmp_post_author' value="<?php echo esc_attr( self::$object->get_user_id() ); ?>" >
							</td>
						</tr>
						<tr>
							<td colspan="2">
								<?php
								submit_button( esc_html__( 'Accept & Send Email', 'lower-price-for-woocommerce' ), 'primary large', 'mmp_accept_mail', false );
								submit_button( esc_html__( 'Reject', 'lower-price-for-woocommerce' ), 'primary large', 'mmp_reson_reject', false, array( 'class' => 'mmp_reson_reject' ) );
								?>
							</td>
						</tr>
					<?php } ?>
				</table>
			</div>
			<?php
		}

		/**
		 * Other Details
		 *
		 * @since 1.0.0
		 */
		public static function get_other_details() {
			?>
			<div class="mmp-request-table">
				<h2><?php echo esc_html__( 'Other Details', 'lower-price-for-woocommerce' ); ?></h2>
				<table>
					<tr>
						<th><?php echo esc_html__( 'Status', 'lower-price-for-woocommerce' ); ?></th>
						<td>
							<?php mmp_display_status( self::$object->get_status_formatted() ); ?>
						</td>
					</tr>
					<tr>
						<th><?php echo esc_html__( 'Mode Of Operation', 'lower-price-for-woocommerce' ); ?></th>
						<td>
							<?php echo wp_kses_post( self::$object->get_req_mode() ); ?>
						</td>
					</tr>
					<tr>
						<th><?php echo esc_html__( 'Description', 'lower-price-for-woocommerce' ); ?></th>
						<td>
							<?php echo wp_kses_post( self::$object->get_description() ); ?>
						</td>
					</tr>
					<tr>
						<th><?php echo esc_html__( 'Submitted Date', 'lower-price-for-woocommerce' ); ?></th>
						<td><?php echo wp_kses_post( self::$object->get_formatted_submitted_date() ); ?></td>
					</tr>
				</table>
			</div>
			<?php
		}

		/**
		 * Reason Details
		 *
		 * @since 1.0.0
		 */
		public static function get_reason_details() {
			?>
			<div class="mmp-request-table">
				<h2><?php echo esc_html__( 'Reason', 'lower-price-for-woocommerce' ); ?></h2>
				<table>
					<tr>
						<td colspan="2"><?php echo wp_kses_post( sprintf( '<b>%s</b>', self::$object->get_reason() ) ); ?></td>
					</tr>
				</table>
			</div>
			<?php
		}

		/**
		 * User Details
		 *
		 * @since 1.0
		 * @param Object $post Post object.
		 */
		public static function get_user_details( $post ) {
			?>
			<div class="mmp-request-table">
				<h2><?php echo esc_html__( 'User Details', 'lower-price-for-woocommerce' ); ?></h2>
				<table>
					<tr>
						<th><?php echo esc_html__( 'Username', 'lower-price-for-woocommerce' ); ?></th>
						<td><?php echo esc_attr( self::$object->get_name() ); ?></td>
					</tr>
					<tr>
						<th><?php echo esc_html__( 'Email ID', 'lower-price-for-woocommerce' ); ?></th>
						<td><?php echo esc_attr( self::$object->get_mail() ); ?></td>
					</tr>
					<tr>
						<th><?php echo esc_html__( 'Mobile Number', 'lower-price-for-woocommerce' ); ?></th>
						<td><?php echo esc_attr( self::$object->get_number() ); ?></td>
					</tr>
				</table>
				<?php

				if ( 'no' === self::$object->get_viewed_once() ) {
					?>
					<input type="hidden" class="mmp-viewed-count" value=<?php echo esc_attr( $post->ID ); ?>>
					<?php
				}

				?>
			</div>
			<?php
		}

		/**
		 * Sortable columns
		 */
		public static function search_action( $where ) {
			global $wpdb, $wp_query;

			if ( ! is_search() || ! isset( $_REQUEST['s'] ) || $wp_query->query_vars['post_type'] != self::$post_type ) {
				return $where;
			}

			$search_ids = array();
			$terms      = explode( ',', sanitize_title( wp_unslash( $_REQUEST['s'] ) ) );

			foreach ( $terms as $term ) {
				$term       = $wpdb->esc_like( wc_clean( $term ) );
				$post_query = new MMP_Query( $wpdb->posts, 'p' );
				$post_query->select( 'DISTINCT `p`.post_author' )
						->where( '`p`.post_type', self::$post_type )
						->where( '`p`.post_status', 'publish' );

				$search_ids = $post_query->fetchCol( 'post_author' );

				$post_query = new MMP_Query( $wpdb->posts, 'p' );
				$post_query->select( '`um`.user_id' )
						->leftJoin( $wpdb->users, 'u', '`u`.ID = `p`.ID' )
						->leftJoin( $wpdb->usermeta, 'um', '`p`.ID = `um`.user_id' )
						->whereIn( '`u`.ID', $search_ids )
						->whereIn( '`um`.meta_key', array( 'first_name', 'last_name', 'billing_email', 'nickname' ) )
						->wherelike( '`um`.meta_value', '%' . $term . '%' )
						->orderBy( '`p`.ID' );

				$search_ids = $post_query->fetchCol( 'user_id' );
			}
			$search_ids = array_filter( array_unique( $search_ids ) );

			if ( count( $search_ids ) > 0 ) {
				$where = str_replace( 'AND (((', "AND ( ({$wpdb->posts}.post_author IN (" . implode( ',', $search_ids ) . ')) OR ((', $where );
			}

			return $where;
		}

		/**
		 * Join query
		 *
		 * @since 1.0.0
		 * @param String $join Join query string.
		 */
		public static function table_join_query( $join ) {
			global $wp_query;

			if ( is_admin() && ! isset( $_GET['post'] ) && isset( $wp_query->query_vars['post_type'] ) && $wp_query->query_vars['post_type'] == self::$post_type ) {
				if ( isset( $_REQUEST['s'] ) && isset( $_REQUEST['post_type'] ) && sanitize_title( wp_unslash( $_REQUEST['post_type'] ) ) == self::$post_type ) {
					global $wpdb;
					$join .= " INNER JOIN $wpdb->postmeta ON ($wpdb->posts.ID = $wpdb->postmeta.post_id)";
				}
			}

			return $join;
		}

		/**
		 * Sortable columns
		 *
		 * @since 1.0.0
		 * @param Array $columns Columns.
		 */
		public static function sortable_columns( $columns ) {
			$columns = array(
				'mmp_org_price'    => 'mmp_org_price',
				'mmp_pref_price'   => 'mmp_pref_price',
				'mmp_off_price'    => 'mmp_off_price',
				'mmp_lst_activity' => 'mmp_lst_activity',
			);

			return $columns;
		}

		/**
		 * Sortable columns
		 *
		 * @since 1.0.0
		 * @param String   $order_by Order by.
		 * @param WP_Query $wp_query Query Object.
		 */
		public static function orderby_columns( $order_by, $wp_query ) {
			global $wpdb;
			if ( isset( $wp_query->query['post_type'] ) && $wp_query->query['post_type'] == self::$post_type ) {
				if ( ! isset( $_REQUEST['order'] ) && ! isset( $_REQUEST['orderby'] ) ) {
					$order_by = "{$wpdb->posts}.ID " . 'DESC';
				} else {
					$decimal_column = array( 'hrw_available_balance', 'hrw_total_balance' );

					if ( in_array( sanitize_title( wp_unslash( $_REQUEST['orderby'] ) ), $decimal_column ) ) {
						$order_by = "CAST({$wpdb->postmeta}.meta_value AS DECIMAL) " . sanitize_title( wp_unslash( $_REQUEST['order'] ) );
					} elseif ( sanitize_title( wp_unslash( $_REQUEST['orderby'] ) ) == 'post_status' ) {
						$order_by = "{$wpdb->posts}.post_status " . sanitize_title( wp_unslash( $_REQUEST['order'] ) );
					} else {
						$order_by = "{$wpdb->postmeta}.meta_value " . sanitize_title( wp_unslash( $_REQUEST['order'] ) );
					}
				}
			}

			return $order_by;
		}

		/**
		 * Sorting Functionality
		 *
		 * @since 1.0.0
		 * @param WP_Query $query The query object.
		 */
		public static function orderby_filter_query( $query ) {
			if ( isset( $_REQUEST['post_type'] ) && sanitize_title( wp_unslash( $_REQUEST['post_type'] ) ) == self::$post_type && self::$post_type == $query->query['post_type'] ) {
				if ( isset( $_GET['orderby'] ) ) {
					$excerpt_array = array( 'ID', 'post_status' );
					if ( ! in_array( sanitize_title( wp_unslash( $_GET['orderby'] ) ), $excerpt_array ) ) {
						$query->query_vars['meta_key'] = sanitize_title( wp_unslash( $_GET['orderby'] ) );
					}
				}
			}
		}

		/**
		 *  Distinct Functionality
		 *
		 * @since 1.0.0
		 * @param String $distinct Distinct function.
		 * @return String
		 */
		public static function distinct_post( $distinct ) {
			if ( ( isset( $_REQUEST['s'] ) || isset( $_REQUEST['orderby'] ) ) && ( isset( $_REQUEST['post_type'] ) && sanitize_title( wp_unslash( $_REQUEST['post_type'] ) ) ) == self::$post_type ) {
				$distinct .= empty( $distinct ) ? 'DISTINCT' : $distinct;
			}

			return $distinct;
		}
		/**
		 * Where Condition
		 *
		 * @since 1.0.0
		 * @param String $where where function.
		 * @return String
		 */
		public static function adding_where( $where ) {
			global $wpdb, $wp_query;

			if ( isset( $wp_query->query_vars['post_type'] ) && $wp_query->query_vars['post_type'] === self::$post_type ) {
				$where .= " OR {$wpdb->posts}.post_status IN ('mmp_new_req' , 'mmp_coupon_sent' , 'mmp_coupon_used' , 'mmp_coupon_expired' , 'mmp_rejected')";
			}

			return $where;
		}
	}

	MMP_Post_Table::init();
}
