<?php
/**
 * MMP Handling
 *
 * @package Lower Price for WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Handler' ) ) {

	/**
	 * Main MMP_Handler Class.
	 * */
	class MMP_Handler {

		/**
		 * Class Init function
		 *
		 * @since 1.0.0
		 */
		public static function init() {
			$single_product_position = get_option( 'mmp_general_form_position', 'woocommerce_before_add_to_cart_form' );
			$single_product_position = ( 'custom' === $single_product_position ) ? get_option( 'mmp_general_form_position_custom_hook', 'woocommerce_before_add_to_cart_form' ) : $single_product_position;
			$single_product_priority = get_option( 'mmp_general_form_position_hook_priority', 10 );

			$variable_product_position = get_option( 'mmp_general_form_position_variable', 'woocommerce_single_variation' );
			$variable_product_position = ( 'custom' === $variable_product_position ) ? get_option( 'mmp_general_form_position_variable', 'woocommerce_single_variation' ) : $variable_product_position;
			$variable_product_priority = get_option( 'mmp_general_form_position_variable_hook_priority', 10 );

			add_action( $single_product_position, array( __CLASS__, 'simple_product_form' ), $single_product_priority );
			add_action( $variable_product_position, array( __CLASS__, 'variable_product_form' ), $variable_product_priority );
			add_action( 'wp_head', array( __CLASS__, 'auto_apply_process' ) );
		}

		/**
		 * Auto Apply Process
		 *
		 * @since 1.0.0
		 */
		public static function auto_apply_process() {
			if ( ! isset( $_GET['auto_apply'] ) ) {
				return;
			}

			$list_obj = mmp_get_entry( wc_clean( wp_unslash( $_GET['auto_apply'] ) ) );

			WC()->cart->add_to_cart( $list_obj->get_pro_id() );

			$coupon_code = mmp_get_entry( wc_clean( wp_unslash( $_GET['auto_apply'] ) ) )->get_coupon_title();

			if ( WC()->cart->has_discount( $coupon_code ) ) {
				wp_redirect( wc_get_cart_url() );
			}

			WC()->cart->add_discount( $coupon_code );
			wp_safe_redirect( wc_get_cart_url() );
		}

		/**
		 * Display Form in Simple Product
		 *
		 * @since 1.0.0
		 */
		public static function simple_product_form() {
			global $post, $product;

			if ( 'yes' !== get_option( 'mmp_general_mmp_enable' ) ) {
				return;
			}

			if ( ! is_user_logged_in() && 'yes' === get_option( 'mmp_advanced_link_guest_hide' ) ) {
				return;
			}

			if ( 'yes' !== get_option( 'mmp_general_display_var_show' ) && ! $product->is_type( 'simple' ) ) {
				return;
			}

			wp_localize_script(
				'mmp_scripts',
				'mmp_frontend_params',
				array(
					'mmp_form_nonce'   => wp_create_nonce( 'mmp-form-nonce' ),
					'ajaxurl'          => MMP_ADMIN_AJAX_URL,
					'post_id'          => $post->ID,
					'link_var_display' => get_option( 'mmp_general_display_var_show' ),
				)
			);

			wp_enqueue_script( 'mmp_scripts' );
			?>

			<div class="mmp_simple_datas">
				<?php self::mmp_form_fields( $post->ID ); ?>
			</div>
			<?php
		}

		/**
		 * Display Form in Variable Product
		 *
		 * @since 1.0.0
		 */
		public static function variable_product_form() {
			if ( 'yes' !== get_option( 'mmp_general_mmp_enable' ) ) {
				return;
			}

			if ( ! is_user_logged_in() && 'yes' === get_option( 'mmp_advanced_link_guest_hide' ) ) {
				return;
			}

			?>
			<div class="mmp_variation_datas"></div> 
			<?php
		}

		/**
		 * Display Form
		 *
		 * @since 1.0.0
		 * @param Integer $post_id Post ID.
		 * @param String  $type Product Type.
		 */
		public static function mmp_form_fields( $post_id, $type = 'simple' ) {
			global $post;

			if ( ! MMP_Restriction_Handler::is_valid_for_mmp( $post_id ) ) {
				return;
			}

			if ( 'variable' !== $type && is_user_logged_in() ) {
				$user       = get_userdata( get_current_user_id() );
				$user_email = $user->user_email;

				if ( 'yes' === get_option( 'mmp_advanced_overall_product_restriction' ) ) {
					$threshold     = get_option( 'mmp_advanced_pro_restriction_val' );
					$request_count = mmp_is_get_request_by_value( $user_email );

					if ( ! empty( $threshold ) && ! empty( $request_count ) && count( $request_count ) >= $threshold ) {
						echo esc_html( get_option( 'mmp_messages_overall_pro_restriction_msg' ) );
						return;
					}
				}

				// Per product check.
				if ( 'yes' === get_option( 'mmp_advanced_per_product_restriction' ) && mmp_is_per_count_check( $user_email, $post_id, 'all' ) ) {
					echo esc_html( get_option( 'mmp_messages_per_pro_restriction_msg' ) );
					return;
				}

				// if accept message.
				if ( mmp_is_per_count_check( $user_email, $post_id, 'accept' ) ) {
					echo esc_html( get_option( 'mmp_messages_req_approved_msg' ) );
					return;
				}

				if ( mmp_is_per_count_check( $user_email, $post_id ) ) {
					// default already submit message.
					echo esc_html( get_option( 'mmp_messages_req_submitted_msg' ) );
					return;
				}
			}

			$post_args   = array(
				'post_type'   => MMP_Register_Post_Types::MMP_FORM_FIELDS_POSTTYPE,
				'post_status' => 'mmp_enabled',
				'fields'      => 'ids',
				'orderby'     => 'menu_order',
				'suppress_filters' => false,
				'order'       => 'ASC',
				'numberposts' => -1,
			);
			$form_fields = get_posts( $post_args );

			if ( ! $form_fields ) {
				return;
			}
			?>
			<a class="mmp_min_price_link"><?php echo esc_html( get_option( 'mmp_messages_link_msg' ) ); ?></a>
			<div class="mmp_popup_overall_wrapper mmp_hide" id="mmp_popup_overall_wrapper">
				<div class="mmp_popup_form_wrapper" id="mmp_popup_form_wrapper">
					<p class="mmp_popup_title"><?php echo esc_html( get_option( 'mmp_localization_form_title' ) ); ?></p>
					<p class="mmp_popup_desc"><?php echo esc_html( get_option( 'mmp_localization_form_description' ) ); ?></p>
					<div id='mmp_error'></div>
				<?php
				foreach ( $form_fields as $field_id ) {
					/**
					 * WPML Support.
					 *
					 * @since 3.2.0
					 * */
					$field_id   = apply_filters( 'wpml_object_id', $field_id, 'post' );
					$form_field = mmp_get_fields_entry( $field_id );
					$required   = ( $form_field->get_field_mandatory() == 'yes' ) ? 'required="required"' : '';
					$class_name = 'mmp_form_row form-row ' . $form_field->get_field_name();

					$data_args = array(
						'form_field' => $form_field,
						'required'   => $required,
						'class_name' => $class_name,
					);

					mmp_get_template( 'mmp-form-fields.php', $data_args );
				}

				mmp_get_template( 'mmp-save-button.php', array( 'post_id' => $post_id ) );
				?>
				</div>
			</div>
				<?php
		}
	}

	MMP_Handler::init();
}
