<?php
/**
 * Menu Management
 *
 * @package Lower Price
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MMP_Menu_Management' ) ) {

	include_once 'class-mmp-settings.php';

	/**
	 * Main Class.
	 * */
	class MMP_Menu_Management {

		/**
		 * Plugin slug.
		 *
		 * @var String
		 */
		protected static $plugin_slug = 'mmp';

		/**
		 * Menu slug.
		 *
		 * @var String
		 */
		protected static $menu_slug = 'mmp-product-list';

		/**
		 * Form Fields slug.
		 *
		 * @var String
		 */
		protected static $form_fields_slug = 'mmp_form_fields';

		/**
		 * Settings slug.
		 *
		 * @var String
		 */
		protected static $settings_slug = 'mmp_settings';

		/**
		 * Init Functions
		 *
		 * @since 1.0
		 */
		public static function init() {
			// Add Admin Menu Page.
			add_action( 'admin_menu', array( __CLASS__, 'add_menu_pages' ) );
			// Add Custom Screen Ids.
			add_filter( 'woocommerce_screen_ids', array( __CLASS__, 'add_custom_wc_screen_ids' ), 9, 1 );
			// Save Custom Field in Settings.
			add_filter( 'woocommerce_admin_settings_sanitize_option', array( __CLASS__, 'save_custom_fields' ), 10, 3 );
		}

		/**
		 * Add Custom Screen IDs in WooCommerce.
		 *
		 * @since 1.0
		 */
		public static function add_custom_wc_screen_ids( $wc_screen_ids ) {
			$screen_ids     = mmp_page_screen_ids();
			$new_screen_ids = get_current_screen();
			$screen_id      = str_replace( 'edit-', '', $new_screen_ids->id );

			// Return if current page is not refund page.
			if ( ! in_array( $screen_id, $screen_ids, true ) ) {
				return $wc_screen_ids;
			}

			$wc_screen_ids[] = $screen_id;

			return $wc_screen_ids;
		}

		/**
		 * Adding MMP Menu
		 */
		public static function add_menu_pages() {
			$dash_icon_url = MMP_PLUGIN_URL . '/assets/images/dash-icon.png';

			/**
			 * Display restriction.
			 *
			 * @since 1.0
			 */
			if ( apply_filters( 'mmp_display_unseen_count_in_menu', true ) && mmp_get_product_list_count() > 0 ) {
				/**
				 * Unseen Count
				 *
				 * @since 3.1.0
				 */
				$unseen_count = apply_filters( 'mmp_product_list_count', mmp_get_product_list_count() );
				$menu_name    = esc_html__( 'Lower Price', 'lower-price-for-woocommerce' ) . ' <span class="mmp-unseen-count-wrap awaiting-mod update-plugins count-' . esc_attr( $unseen_count ) . '" data-count="' . esc_attr( $unseen_count ) . '"><span class="mmp-unseen-count processing-count">' . number_format_i18n( $unseen_count ) . '</span></span>';
			} else {
				$menu_name = esc_html__( 'Lower Price', 'lower-price-for-woocommerce' );
			}

			add_menu_page( $menu_name, $menu_name, 'manage_options', self::$menu_slug, '', $dash_icon_url );

			$form_fields = add_submenu_page( self::$menu_slug, esc_html__( 'Frontend Form', 'lower-price-for-woocommerce' ), esc_html__( 'Frontend Form', 'lower-price-for-woocommerce' ), 'manage_options', self::$form_fields_slug, array( __CLASS__, 'form_fields_page' ) );
			add_action( sanitize_key( 'load-' . $form_fields ), array( __CLASS__, 'form_fields_page_init' ) );

			$settings_page = add_submenu_page( self::$menu_slug, esc_html__( 'Settings', 'lower-price-for-woocommerce' ), esc_html__( 'Settings', 'lower-price-for-woocommerce' ), 'manage_options', self::$settings_slug, array( __CLASS__, 'settings_page' ) );
			add_action( sanitize_key( 'load-' . $settings_page ), array( __CLASS__, 'settings_page_init' ) );
		}

		/**
		 * Settings page init.
		 *
		 * @since 1.0
		 */
		public static function settings_page_init() {
			global $current_tab, $current_section, $current_action;

			// Include settings pages.
			$settings        = MMP_Settings::get_settings_pages();
			$tabs            = mmp_get_allowed_setting_tabs();
			$current_tab     = ( ! isset( $_GET['tab'] ) || empty( $_GET['tab'] ) || ! array_key_exists( wc_clean( wp_unslash( $_GET['tab'] ) ), $tabs ) ) ? key( $tabs ) : wc_clean( wp_unslash( $_GET['tab'] ) );
			$section         = isset( $settings[ $current_tab ] ) ? $settings[ $current_tab ]->get_sections() : array();
			$current_section = empty( $_REQUEST['section'] ) ? key( $section ) : wc_clean( wp_unslash( $_REQUEST['section'] ) );
			$current_section = empty( $current_section ) ? $current_tab : $current_section;
			$current_action  = empty( $_REQUEST['action'] ) ? '' : wc_clean( wp_unslash( $_REQUEST['action'] ) );

			/**
			 * Settings save.
			 *
			 * @since 1.0
			 */
			do_action( sanitize_key( self::$plugin_slug . '_settings_save_' . $current_tab ), $current_section );

			/**
			 * Settings reset.
			 *
			 * @since 1.0
			 */
			do_action( sanitize_key( self::$plugin_slug . '_settings_reset_' . $current_tab ), $current_section );

			// Add Custom Field in Settings.
			add_action( 'woocommerce_admin_field_mmp_custom_fields', array( __CLASS__, 'custom_fields_output' ) );
		}

		/**
		 * Settings page output.
		 *
		 * @since 3.3.0
		 */
		public static function form_fields_page_init() {
			global $current_tab, $current_section, $current_action;

			$current_action = empty( $_REQUEST['action'] ) ? '' : wc_clean( wp_unslash( $_REQUEST['action'] ) );
		}

		/**
		 * Settings page output.
		 *
		 * @since 3.3.0
		 */
		public static function form_fields_page() {
			global $current_tab, $current_section, $current_action;

			MMP_Admin_Custom_Field::output();
		}

		/**
		 * Settings page output.
		 *
		 * @since 1.0
		 */
		public static function settings_page() {
			MMP_Settings::output();
		}

		/**
		 * Output the custom field settings.
		 *
		 * @since 1.0
		 * @param Array $options Options.
		 */
		public static function custom_fields_output( $options ) {
			MMP_Settings::output_fields( $options );
		}

		/**
		 * Save Custom Field settings.
		 *
		 * @since 1.0
		 * @param String $value Value.
		 * @param String $option Option Name.
		 * @param String $raw_value Value.
		 */
		public static function save_custom_fields( $value, $option, $raw_value ) {
			return MMP_Settings::save_fields( $value, $option, $raw_value );
		}
	}

	MMP_Menu_Management::init();
}
