<?php

/*
Plugin Name: Parcelz
Plugin URI: http://wordpress.org/plugins/Parcelz/
Description: Parcelz Shipping Plugin
Author: Noviship Inc
Version: 1.42
Author URI: http://http://www.parcelz.com
*/

include_once(plugin_dir_path(__FILE__) . 'noviship-config.php');
include_once(plugin_dir_path(__FILE__) . 'noviship-client.php');

// Translator Handling
function __t($s, $a = null)
{
    return $s;
}

function noviship_init()
{
    if (!class_exists('WC_Noviship_Shipping_Method'))
    {
        class WC_Noviship_Shipping_Method extends WC_Shipping_Method
        {
            protected function get_setting($name, $boolean, $instance = false)
            {
                $value = isset($this->settings[$name]) ? $this->settings[$name] : null;
                if ($instance)
                {
                    $value = isset($this->instance_settings[$name]) ? $this->instance_settings[$name] : $value;
                }
                if ($boolean)
                {
                    return $value === 'yes' || $value === '1' || $value === true;
                }
                return $value;
            }

            protected function get_instance_setting($name, $boolean)
            {
                return $this->get_setting($name, $boolean, true);
            }

            /**
                * Constructor for Noviship
                *
                * @access public
                * @return void
                */
            public function __construct($instance_id = 0)
            {
                $this->id = 'noviship';
                $this->title = NOVISHIP_BRANDING;
                $this->method_title = NOVISHIP_BRANDING;
                $this->method_description = __t(NOVISHIP_BRANDING . ' Shipping System', 'woocommerce');
                $this->enabled = 'yes';
                $this->method_title = NOVISHIP_BRANDING;

                $this->form_fields = array(
                    'enabled' => array(
                        'title' => __t('Enabled', 'woocommerce'),
                        'type' => 'checkbox',
                        'description' => __t('Enable this plugin', 'woocommerce'),
                    ),
                    'username' => array(
                        'title' => __t('User Name', 'woocommerce'),
                        'type' => 'text',
                        'description' => __t('Your ' . NOVISHIP_BRANDING . ' User Name', 'woocommerce'),
                    ),
                    'password' => array(
                        'title' => __t('Password', 'woocommerce'),
                        'type' => 'password',
                        'description' => __t('Your ' . NOVISHIP_BRANDING . ' Account Password', 'woocommerce'),
                    ),
                    'commercial_invoice' => array(
                        'title' => __t('Enable Commercial Invoices', 'woocommerce'),
                        'type' => 'checkbox',
                        'description' => __t('Create a Commercial Invoice for International Shipments', 'woocommerce'),
                    ),
                    'email_shipper' => array(
                        'title' => __t('Notify Shipper by Email', 'woocommerce'),
                        'type' => 'checkbox',
                        'description' => __t('Send a notification email to the shipment sender including labels and other documents', 'woocommerce'),
                    ),
                    'email_destination' => array(
                        'title' => __t('Notify Consignee by Email', 'woocommerce'),
                        'type' => 'checkbox',
                        'description' => __t('Send a notification email to the consignee including a link to track the shipment', 'woocommerce'),
                    ),
                    'include_tax' => array(
                        'title' => __t('Include Tax in Shipping Price', 'woocommerce'),
                        'type' => 'checkbox',
                        'description' => __t('Select this option to have the shipping price include taxes.', 'woocommerce'),
                    ),
                    'markup' => array(
                        'title' => __t('Percentage to add to shipping price', 'woocommerce'),
                        'type' => 'text',
                        'description' => __t('Optionally enter an amount (percentage) to add to the shipping price as it appears in the customer shopping cart.', 'woocommerce'),
                    ),
                    'dry_ice' => array(
                        'title' => __t('Add Dry Ice to each shipment', 'woocommerce'),
                        'type' => 'text',
                        'description' => __t('Optionally enter a weight (in your selected units) representing dry ice to be added to each shipment.  Note that not all carriers will accept dry ice shipments.', 'woocommerce'),
                    ),
                    'multiple_packages' => array(
                        'title' => __t('Enable Multiple Packages', 'woocommerce'),
                        'type' => 'checkbox',
                        'description' => __t('Select this option to have each cart item in a separate package', 'woocommerce'),
                    ),
                    'display_date' => array(
                        'title' => __t('Include Estimated Delivery Date', 'woocommerce'),
                        'type' => 'checkbox',
                        'description' => __t('Select this option to include the estimated delivery date (if available) in the service label'),
                    ),
                    'packaging_type' => array(
                        'title' => __t('Packaging Type', 'woocommerce'),
                        'type' => 'select',
                        'description' => __t('Select the packaging type that will be used for shipments'),
                        'options' => array(
                            'CUST' => 'Customer Packaging',
                            'ENVELOPE' => 'Courier Provided Envelope',
                            'PAK' => 'Courier Provided Pak',
                        ),
                    ),
                    'shipping_days' => array(
                        'title' => __t('Shipping Days', 'woocommerce'),
                        'type' => 'select',
                        'description' => __t('Select which days you ship packages.  Estimates and pricing will be based on the next available shipping day.'),
                        'options' => array(
                            '' => 'Every Day (Including Weekends)',
                            '1,2,3,4,5' => 'Week Days (Monday-Friday)',
                            '2,4' => 'Tuesday and Thursday',
                        ),
                    ),
                );

                $this->instance_form_fields = array(
                    'instance_markup' => array(
                        'title' => __t('Percentage to add to shipping price', 'woocommerce'),
                        'type' => 'text',
                        'description' => __t('Optionally enter an amount (percentage) to add to the shipping price as it appears in the customer shopping cart.', 'woocommerce'),
                    ),
                    'instance_add_price' => array(
                        'title' => __t('Adjustment to the shipping price', 'woocommerce'),
                        'type' => 'text',
                        'description' => __t('Optionally enter an price (in current currency) to add to each shipping quote', 'woocommerce'),
                    ),
                    'instance_add_weight' => array(
                        'title' => __t('Adjustment to the shipping weight', 'woocommerce'),
                        'type' => 'text',
                        'description' => __t('Optionally enter a weight (in current weight unit) to add to each shipment for calculating shipping prices.', 'woocommerce'),
                    ),
                    'instance_signature_required' => array(
                        'title' => __t('Add Signature Required option to shipments', 'woocommerce'),
                        'type' => 'select',
                        'description' => __t('Optionally add Signature Required option to all shipments', 'woocommerce'),
                        'options' => array(
                          '0' => 'Default (determined by account and carrier)',
                          '1' => 'YES (Signature Required)',
                          '2' => 'NO (Origin Signature Not Required)',
                        ),
                    ),
                );

                //$this->instance_form_fields = $this->form_fields;
                //unset($this->form_fields);

                $this->instance_id = absint($instance_id);
                $this->init_settings();
                $this->init_instance_settings();

                $this->api_key = $this->settings['username'] . ':' . $this->settings['password'];
                $this->commercial_invoice = $this->get_setting('commercial_invoice', true);
                $this->email_shipper = $this->get_setting('email_shipper', true);
                $this->email_destination = $this->get_setting('email_destination', true);
                $this->include_tax = $this->get_setting('include_tax', true);
                $this->markup = floatval($this->get_setting('markup', false));
                $this->dry_ice = $this->get_setting('dry_ice', true);
                $this->multiple_packages = $this->get_setting('multiple_packages', true);
                $this->display_date = $this->get_setting('display_date', true);
                $this->packaging_type = $this->get_setting('packaging_type', false);
                $this->shipping_days = $this->get_setting('shipping_days', false);

                $this->instance_markup = $this->get_instance_setting('instance_markup', true);
                $this->instance_add_price = floatval($this->get_instance_setting('instance_add_price', false));
                $this->instance_add_weight = floatval($this->get_instance_setting('instance_add_weight', false));
                $this->instance_signature_required = intval($this->get_instance_setting('instance_signature_required', false));

                $this->supports = array('settings', 'shipping-zones', 'instance-settings');

                add_action('woocommerce_update_options_shipping_methods', array(&$this, 'process_admin_options'));
                add_action('woocommerce_update_options_shipping_' . $this->id, array( $this, 'process_admin_options' ) );
            }

            function _admin_options()
            {
?>
                <h2>
                    <?php _e('Noviship','woocommerce'); ?>
                </h2>
                <table class="form-table">
                    <?php $this->generate_settings_html(); ?>
                </table>
                <?php
            }

            function get_key()
            {
                return $this->api_key;
            }

            function get_commercial_invoice()
            {
                return $this->commercial_invoice ? '1' : '0';
            }

            function get_email_shipper()
            {
                return $this->email_shipper ? '1' : '0';
            }

            function get_email_destination()
            {
                return $this->email_shipper ? '1' : '0';
            }

            function get_include_tax()
            {
                return $this->include_tax ? '1' : '0';
            }

            function get_markup()
            {
                return floatval($this->markup) + floatval($this->instance_markup);
            }

            function get_weight_adjustment()
            {
                return floatval($this->instance_add_weight);
            }

            function get_price_adjustment()
            {
                return floatval($this->instance_add_price);
            }

            function get_dry_ice()
            {
                return $this->dry_ice;
            }

            function get_signature_required()
            {
                return $this->instance_signature_required;
            }

            function get_multiple_packages()
            {
                return $this->multiple_packages;
            }

            function get_display_date()
            {
                return $this->display_date ? '1' : '0';
            }

            function get_packaging_type()
            {
                return $this->packaging_type ? $this->packaging_type : 'CUST';
            }

            function get_shipping_days()
            {
                return $this->shipping_days ? $this->shipping_days : '';
            }

            function calculate_shipping($package = array())
            {
                if (in_array($package['destination']['country'], array('US', 'CA')) && $package['destination']['postcode'] == '')
                {
                    // don't quote US or Canada without a postal code
                    return;
                }

                $woocommerce = function_exists('WC') ? WC() : $GLOBALS['woocommerce'];
                $weight = $woocommerce->cart->get_cart_contents_weight();
                if ($this->get_dry_ice())
                {
                    $weight += $this->get_dry_ice();
                }
                $weight += $this->get_weight_adjustment();
                $weight_in_lb = wc_get_weight($weight, 'lbs');

                $shipment_date = date('Y-m-d');
                if ($this->get_shipping_days() != '')
                {
                    $date = time();
                    $allowed = explode(',', $this->get_shipping_days());
                    while (!in_array(date('N', $date), $allowed))
                    {
                        $date = strtotime('+1 day', $date);
                    }
                    $shipment_date = date('Y-m-d', $date);
                }

                $shipment = array(
                    'shipmentdate' => $shipment_date,
                    'weightunit' => 'LB',
                    'dimensionunit' => 'IN',
                    'pkgtype' => $this->get_packaging_type(),
                    'packages' => array(
                        array(
                            'weight' => $weight_in_lb,
                        )
                    ),
                    'destination' => array(
                        'country' => $package['destination']['country'],
                        'province' => '',//$package['destination']['state'],
                        'postalcode' => $package['destination']['postcode'],
                        'city' => '',//$package['destination']['city'],
                        'street1' => $package['destination']['address'],
                        'street2' => $package['destination']['address_2'],
                        'residential' => '1',
                    ),
                    'options' => array(),
                );

                $cart = $woocommerce->cart->get_cart();
                if ($this->get_multiple_packages())
                {
                    $shipment['packages'] = array();
                    foreach ($cart as $item)
                    {
                        $product = new WC_Product($item['product_id']);
                        $package = array(
                            'qty' => isset($item['quantity']) ? $item['quantity'] : 1,
                            'weight' => wc_get_weight($product->get_weight(), 'lbs'),
                            'length' => wc_get_dimension($product->get_length(), 'in'),
                            'width' => wc_get_dimension($product->get_width(), 'in'),
                            'height' => wc_get_dimension($product->get_height(), 'in'),
                            'description' => $product->get_sku() . ' ' . $product->get_title(),
                        );
                        $shipment['packages'][] = $package;
                    }
                }

                foreach ($cart as $item)
                {
                    $product = new WC_Product($item['product_id']);
                    $shipper_fields = array(
                        'shipper_attn' => 'attn',
                        'shipper_company' => 'orgname',
                        'shipper_orgname' => 'orgname',
                        'shipper_country' => 'country',
                        'shipper_state' => 'province',
                        'shipper_province' => 'province',
                        'shipper_street1' => 'street1',
                        'shipper_street2' => 'street2',
                        'shipper_zip' => 'postalcode',
                        'shipper_postalcode' => 'postalcode',
                        'shipper_tel' => 'tel',
                        'shipper_email' => 'email',
                        'shipper_city' => 'city',
                    );
                    foreach ($shipper_fields as $wc_field => $card_field)
                    {
                        if ($product->get_attribute($wc_field))
                        {
                            $shipment['shipper'][$card_field] = strtoupper($product->get_attribute($wc_field));
                        }
                    }
                }

                if (isset($package['destination']['first_name']) && isset($package['destination']['last_name']))
                {
                    $shipment['destination']['attn'] = $package['destination']['first_name'] . ' ' . $package['destination']['last_name'];
                }
                if (isset($package['destination']['company']))
                {
                    $shipment['destination']['orgname'] = $package['destination']['company'];
                }
                if ($this->get_dry_ice())
                {
                    $shipment['options'][] =
                        array(
                            'code' => 'dry_ice',
                            'params' => array(
                                array(
                                    'name' => 'enable',
                                    'value' => 1,
                                ),
                                array(
                                    'name' => 'dry_ice_weight',
                                    'value' => wc_get_weight($this->dry_ice, 'lbs'),
                                ),
                            )
                        );
                }
                if ($this->instance_signature_required > 0)
                {
                    $shipment['options'][] =
                        array(
                            'code' => 'signature_required',
                            'params' => array(
                                array(
                                    'name' => 'option',
                                    'value' => $this->instance_signature_required,
                                ),
                            )
                        );
                }
                $noviship = new Noviship($this->api_key);
                $error = $noviship->GetQuotes($shipment, $quotes, $messages);

                if ($error == 0)
                {
                    foreach ($quotes as $quote)
                    {
                        $cost = $this->get_include_tax() ? $quote['grossprice'] : $quote['netprice'];
                        $ratio = 1 + ($this->get_markup() / 100);
                        $rate = array(
                            'id' => 'noviship_' . $quote['carrier'] . '_' . $quote['serviceid'],
                            'label' => $quote['carriername'] . ' ' . $quote['servicename'],
                            'cost' => round(($cost * $ratio) + $this->get_price_adjustment(), 2),
                            'calc_tax' => 'per_order',
                            'taxes' => false,
                        );
                        if ($this->get_display_date() && $quote['deliverydate'])
                        {
                            $rate['label'] .= ' (' . $quote['deliverydate'] . ')';
                        }

                        if ($this->get_include_tax() == false)
                        {
                            //$taxes = array();
                            /*
                            if ($quote['tax1amount'])
                            {
                            $taxes[$quote['tax1name']] = round($quote['tax1amount'] * $ratio, 2);
                            }
                            if ($quote['tax2amount'])
                            {
                            $taxes[$quote['tax2name']] = round($quote['tax2amount'] * $ratio, 2);
                            }
                             */
                            $rate['taxes'] = true;
                        }
                        $this->add_rate($rate);
                    }
                }

            }
        }
    }
}

function noviship_shipping_method($methods)
{
    $methods['noviship'] = 'WC_Noviship_Shipping_Method';
    return $methods;
}

function noviship_submit_shipment($order)
{
    try
    {
        noviship_init();
        $order_id = $order->id;

        // check if this shipment has already been manifested
        $tracking_number = get_post_meta($order_id, '_noviship_tracking', true);

        // check if this shipment has already been created
        $shipment_number = get_post_meta($order_id, '_noviship_number', true);

        $selected_quote_service = null;
        $waybills = array();

        // measure products and prepare commodities for international shipments
        $items = $order->get_items();
        $total_weight = 0;
        $commodities = array();
        foreach ($items as $item)
        {
             if ($item->get_variation_id() > 0)
             {
                 $product = new WC_Product_Variation($item->get_variation_id());
             }
             else
             {
                 $product = wc_get_product($item->get_product_id());
             }
             //$product = new WC_Product($item['product_id']);
            $weight = wc_get_weight($product->get_weight(), 'lbs');
            $qty = $item['qty'];
            $product_weight = $weight;
            $total_weight += $weight * $qty;
            $hscode = $product->get_attribute('hscode');
            if ($hscode == '')
            {
                $hscode = '0.0.0.0';
            }
            $origin = $product->get_attribute('origin');
            if ($origin == '')
            {
                $origin = 'US';
            }
            $commodity = array(
                'description' => trim($product->get_sku() . ' ' . $product->get_title()),
                'qty' => $qty,
                'weight' => $product_weight,
                'hscode' => $hscode,
                'unitofmeasure' => 'EA',
                'unitprice' => $product->get_price_excluding_tax() ? $product->get_price_excluding_tax() : '1',
                'origin' => $origin,
            );
            $commodities[] = $commodity;
        }

        $shipping_method = new WC_Noviship_Shipping_Method();
        $noviship = new Noviship($shipping_method->get_key());

        $dry_ice_weight = $shipping_method->get_dry_ice();
        if ($dry_ice_weight)
        {
            $total_weight += $dry_ice_weight;
        }
        $total_weight += wc_get_weight($shipping_method->get_weight_adjustment(), 'lbs');

        $weight_in_lb = $total_weight;
        $shipping = $order->get_items('shipping');
        $shipping_id = null;
        $shipping_service = null;
        if ($shipping)
        {
            $shipping_id = current($shipping)['method_id'];
            $shipping_service = current($shipping)['method_title'];
            $i = strpos($shipping_service, ' (');
            if ($i > 0) {
                $shipping_service = substr($shipping_service, 0, $i);
            }
        }

        // set up address
        $billing_address = $order->get_address('billing');
        $shipping_address = $order->get_address('shipping');
        $attn = trim($shipping_address['first_name'] . ' ' . $shipping_address['last_name']);
        $company = $shipping_address['company'];
        $destination = array(
            'attn' => $attn ? $attn : $company,
            'orgname' => $company ? $company : $attn,
            'country' => $shipping_address['country'],
            'province' => $shipping_address['state'],
            'postalcode' => $shipping_address['postcode'],
            'city' => $shipping_address['city'],
            'street1' => $shipping_address['address_1'],
            'street2' => $shipping_address['address_2'],
            'tel' => empty($shipping_address['phone']) ? (empty($billing_address['phone']) ? '' : $billing_address['phone']) : $shipping_address['phone'],
            'email' => empty($shipping_address['email']) ? (empty($billing_address['email']) ? '' : $billing_address['email']) : $shipping_address['email'],
            'residential' => '1',
        );
        $shipment = array(
            'destination' => $destination,
            'packages' => array(
                array(
                    'weight' => $weight_in_lb,
                    'description' => 'Order #' . $order_id,
                    'qty' => 1,
                    'width' => 1,
                    'height' => 1,
                    'length' => 1,
                )
            ),
            'commodities' => $commodities,
            'weightunit' => 'LB',
            'dimensionunit' => 'IN',
            'pkgtype' => $shipping_method->get_packaging_type(),
            'custrefcode' => 'Order #' . $order_id,
            'customs' => array(
                array('name' => 'commercial_invoice', 'value' => $shipping_method->get_commercial_invoice()),
            ),
            'emailshipper' => $shipping_method->get_email_shipper(),
            'emaildestination' => $shipping_method->get_email_destination(),
            'options' => array(),
        );

        //if ($shipping_method->get_multiple_packages() == 'yes')
        if ($shipping_method->get_multiple_packages())
        {
            $shipment['packages'] = array();
            foreach ($items as $item)
            {
                if ($item->get_variation_id() > 0)
                {
                    $product = new WC_Product_Variation($item->get_variation_id());
                }
                else
                {
                    $product = wc_get_product($item->get_product_id());
                }

                $package = array(
                    'qty' => $item['qty'],
                    'weight' => wc_get_weight($product->get_weight(), 'lbs'),
                    'length' => wc_get_dimension($product->get_length(), 'in'),
                    'width' => wc_get_dimension($product->get_width(), 'in'),
                    'height' => wc_get_dimension($product->get_height(), 'in'),
                    'description' => $product->get_sku() . ' ' . $product->get_title(),
                );
                $shipment['packages'][] = $package;
            }

        }

        if ($shipping_method->dry_ice)
        {
            $shipment['options'][] =
                array(
                    'code' => 'dry_ice',
                    'params' => array(
                        array(
                            'name' => 'enable',
                            'value' => 1,
                        ),
                        array(
                            'name' => 'dry_ice_weight',
                            'value' => wc_get_weight($shipping_method->dry_ice, 'lbs'),
                        ),
                    )
                );
        }

        if ($shipping_method->instance_signature_required > 0)
        {
            $shipment['options'][] =
                array(
                    'code' => 'signature_required',
                    'params' => array(
                        array(
                            'name' => 'option',
                            'value' => $shipping_method->instance_signature_required,
                        ),
                    )
                );
        }

        // create shipment if we don't have a shipment_number
        if ($shipment_number == null)
        {
            $result = $noviship->CreateShipment($shipment, $messages, $shipment_number);
            if ($result != 0)
            {
                foreach ($messages as $message)
                {
                    $order->add_order_note($message['text']);
                }
            }
        }
        else
        {
            $result = $noviship->ReadShipment($shipment_number, $read_messages, $read_shipment);
            if ($result == 0)
            {
                $tracking_number = $read_shipment['trackingnumber'];
                if ($tracking_number)
                {
                    foreach ($shipment['quotes'] as $quote)
                    {
                        if ($quote['id'] == $read_shipment['selectedquote'])
                        {
                            $selected_quote_service = $quote['carriername'] . ' ' . $quote['servicename'];
                        }
                    }
                }
                else
                {
                    $modify_messages = array();
                    $shipment['shipper'] = $read_shipment['shipper'];
                    $noviship->ModifyShipment($shipment_number, $shipment, $modify_messages);
                }
            }
        }

        // quote and submit shipment if we don't have a tracking number
        if ($shipment_number && !$tracking_number)
        {
            $selected_quote_id = null;
            $result = $noviship->QuoteShipment($shipment_number, $messages, $quotes);
            foreach ($quotes as $quote)
            {
                $key = 'noviship_' . $quote['carrier'] . '_' . $quote['serviceid'];
                // revision after method_id is being set to "noviship" in WC 3.4
                $key2 = $quote['carriername'] . ' ' . $quote['servicename'];
                if ($key == $shipping_id || $key2 == $shipping_service)
                {
                    $selected_quote_id = $quote['id'];
                    $selected_quote_service = $quote['carriername'] . ' ' . $quote['servicename'];
                    break;
                }
            }
            /*
            if ($selected_quote_id == null && count($quotes))
            {
                $selected_quote_id = $quotes[0]['id'];
                $selected_quote_service = $quotes[0]['carriername'] . ' ' . $quotes[0]['servicename'];
            }
            */
            foreach ($messages as $message)
            {
                $order->add_order_note($message['text']);
            }
            if ($selected_quote_id)
            {
                $result = $noviship->SubmitShipment($shipment_number, $selected_quote_id, $messages, $documents);
                if ($result == 0)
                {
                    $order->add_order_note('Shipment ' . $shipment_number . ' submitted (' . $selected_quote_service . ') Tracking #' . $shipment['trackingnumber']);
                    $result = $noviship->ReadShipment($shipment_number, $read_messages, $shipment);
                    if ($result == 0)
                    {
                        $tracking_number = $shipment['trackingnumber'];
                    }
                }
                foreach ($messages as $message)
                {
                    $order->add_order_note($selected_quote_service . ': ' . $message['text']);
                }
            }
        }

        // save data as order meta
        if (isset($shipment['documents']))
        {
            $waybills = array();
            foreach ($shipment['documents'] as $document)
            {
                if ($document['type'] == 'waybill' || $document['type'] == '1')
                {
                    $waybills[] = $document['url'];
                }
            }
            update_post_meta($order_id, '_noviship_waybills', implode('|', $waybills));
        }

        if ($tracking_number)
        {
            update_post_meta($order_id, '_noviship_tracking', $tracking_number);
        }

        if ($shipment_number)
        {
            update_post_meta($order_id, '_noviship_number', $shipment_number);
        }

        if ($selected_quote_service)
        {
            update_post_meta($order_id, '_noviship_service', $selected_quote_service);
        }

    }
    catch (Exception $e)
    {
        $order->add_order_note($e->getMessage());
    }

}

function noviship_admin_order_data_after_order_details($order)
{
    $order_id = $order->get_id();
    $number = get_post_meta($order_id, '_noviship_number', true);
    if ($number)
    {
        echo '<p><strong>' . __t('Shipment ID', 'woocommerce') . ':</strong> <a href="' . NOVISHIP_CUSTOMER_URL . '/ref/' . $number . '">' . $number . '</a></p>';
        $service = get_post_meta($order_id, '_noviship_service', true);
        if ($service)
        {
            echo '<p><strong>' . __t('Service', 'woocommerce') . ':</strong> ' . $service . '</p>';
        }
        $waybills = get_post_meta($order_id, '_noviship_waybills', true);
        if ($waybills)
        {
            $list = explode('|', $waybills);
            echo '<p><strong>' . __t('Shipping Labels', 'woocommerce') . ':</strong> ';
            $index = 1;
            foreach ($list as $url)
            {
                echo '<a href="' . $url . '">' . __t('Label', 'woocommerce') . ' #' . $index++ . '</a> ';
            }
            echo '</p>';
        }
        $tracking = get_post_meta($order_id, '_noviship_tracking', true);
        if ($tracking)
        {
            echo '<p><strong>' . __t('Tracking Number', 'woocommerce') . ':</strong> ' . $tracking . '</p>';
        }
    }
}

function noviship_add_order_action($actions) {
    $actions['noviship_submit'] = __t('Create Shipping Label', 'woocommerce');
    return $actions;
}

if (in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins'))))
{
    add_action('woocommerce_shipping_init', 'noviship_init');
    add_filter('woocommerce_shipping_methods', 'noviship_shipping_method');
    add_action('woocommerce_order_actions', 'noviship_add_order_action');
    add_action('woocommerce_order_action_noviship_submit', 'noviship_submit_shipment');
    //add_action('woocommerce_admin_order_data_after_order_details', 'noviship_admin_order_data_after_order_details');
    add_action('woocommerce_admin_order_data_after_shipping_address', 'noviship_admin_order_data_after_order_details');
}
