<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://oxymade.com
 * @since      1.0.0
 *
 * @package    Oxymade
 * @subpackage Oxymade/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Oxymade
 * @subpackage Oxymade/includes
 * @author     Anvesh <support@oxymade.com>
 */
class Oxymade
{
  const PREFIX = "oxymade_";
  const TITLE = "OxyMade";

  static function init()
  {
    OxyMadeLicense::init(
      self::PREFIX,
      self::TITLE,
      OXYMADE_STORE_URL,
      OXYMADE_ITEM_ID
    );
    add_action(
      "activate_" . plugin_basename(__FILE__),
      [__CLASS__, "activate"],
      10,
      2
    );

    if (OxyMadeLicense::is_activated_license() === true) {
      // add_action("wp_enqueue_scripts", [__CLASS__, "scripts"], 11);
    }

    add_action("admin_menu", [__CLASS__, "admin_menu"], 11);
    add_action("admin_init", [__CLASS__, "plugin_updater"], 0);
  }

  static function admin_menu()
  {
    // add_submenu_page(
    //   "oxymade",
    //   "OxyMade",
    //   "OxyMade",
    //   "manage_options",
    //   self::PREFIX . "menu",
    //   [__CLASS__, "menu_item"]
    // );

    // add_submenu_page(
    //   "oxymade",
    //   "License",
    //   "License",
    //   "manage_options",
    //   "oxymade&tab=license",
    //   [__CLASS__, "menu_item"]
    // );
  }

  static function menu_item()
  {
    $tab = isset($_GET["tab"]) ? sanitize_text_field($_GET["tab"]) : false; ?>
    <div class="wrap">
      
    <?php if (OxyMadeLicense::is_activated_license() === true) {
      if ($tab === "license") {
        OxyMadeLicense::license_page();
      }
    } else {
      OxyMadeLicense::license_page();
    } ?>
    </div>
    <?php
  }

  static function plugin_updater()
  {
    // require( 'includes/plugin_updater.php' );
    // retrieve our license key from the DB.
    $license_key = trim(get_option(self::PREFIX . "license_key"));

    // setup the updater.
    $edd_updater = new OxyMade_Plugin_Updater(OXYMADE_STORE_URL, OXYMADE_PLUGIN_FILE, [
      "version" => OXYMADE_VERSION, // current version number
      "license" => $license_key, // license key (used get_option above to retrieve from DB)
      "item_id" => OXYMADE_ITEM_ID, // ID of the product
      "item_name" => self::TITLE,
      "author" => "OxyMade", // author of this plugin
      "url" => home_url(),
      "beta" => false,
    ]);
  }

  /**
   * The loader that's responsible for maintaining and registering all hooks that power
   * the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      Oxymade_Loader    $loader    Maintains and registers all hooks for the plugin.
   */
  protected $loader;

  /**
   * The unique identifier of this plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $plugin_name    The string used to uniquely identify this plugin.
   */
  protected $plugin_name;

  /**
   * The current version of the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $version    The current version of the plugin.
   */
  protected $version;

  /**
   * Define the core functionality of the plugin.
   *
   * Set the plugin name and the plugin version that can be used throughout the plugin.
   * Load the dependencies, define the locale, and set the hooks for the admin area and
   * the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function __construct()
  {
    if (defined("OXYMADE_VERSION")) {
      $this->version = OXYMADE_VERSION;
    } else {
      $this->version = "1.0.0";
    }
    $this->plugin_name = "oxymade";

    $this->load_dependencies();
    $this->set_locale();
    $this->define_admin_hooks();
    $this->define_public_hooks();
  }

  /**
   * Load the required dependencies for this plugin.
   *
   * Include the following files that make up the plugin:
   *
   * - Oxymade_Loader. Orchestrates the hooks of the plugin.
   * - Oxymade_i18n. Defines internationalization functionality.
   * - Oxymade_Admin. Defines all hooks for the admin area.
   * - Oxymade_Public. Defines all hooks for the public side of the site.
   *
   * Create an instance of the loader which will be used to register the hooks
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function load_dependencies()
  {
    /**
     * The class responsible for orchestrating the actions and filters of the
     * core plugin.
     */
    require_once plugin_dir_path(dirname(__FILE__)) .
      "includes/class-oxymade-loader.php";

    /**
     * The class responsible for defining internationalization functionality
     * of the plugin.
     */
    require_once plugin_dir_path(dirname(__FILE__)) .
      "includes/class-oxymade-i18n.php";

    /**
     * The class responsible for defining all actions that occur in the admin area.
     */
    require_once plugin_dir_path(dirname(__FILE__)) .
      "admin/class-oxymade-admin.php";

    /**
     * The class responsible for defining all actions that occur in the public-facing
     * side of the site.
     */
    require_once plugin_dir_path(dirname(__FILE__)) .
      "public/class-oxymade-public.php";

    $this->loader = new Oxymade_Loader();
  }

  /**
   * Define the locale for this plugin for internationalization.
   *
   * Uses the Oxymade_i18n class in order to set the domain and to register the hook
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function set_locale()
  {
    $plugin_i18n = new Oxymade_i18n();

    $this->loader->add_action(
      "plugins_loaded",
      $plugin_i18n,
      "load_plugin_textdomain"
    );
  }

  /**
   * Register all of the hooks related to the admin area functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_admin_hooks()
  {
    $plugin_admin = new Oxymade_Admin(
      $this->get_plugin_name(),
      $this->get_version()
    );

    $this->loader->add_action(
      "admin_enqueue_scripts",
      $plugin_admin,
      "enqueue_styles"
    );
    $this->loader->add_action(
      "admin_enqueue_scripts",
      $plugin_admin,
      "enqueue_scripts"
    );

    $this->loader->add_action(
      "admin_menu",
      $plugin_admin,
      "register_dashboard_pages"
    );
  }

  /**
   * Register all of the hooks related to the public-facing functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_public_hooks()
  {
    $plugin_public = new Oxymade_Public(
      $this->get_plugin_name(),
      $this->get_version()
    );

    $this->loader->add_action(
      "wp_enqueue_scripts",
      $plugin_public,
      "enqueue_styles"
    );
    $this->loader->add_action(
      "wp_enqueue_scripts",
      $plugin_public,
      "enqueue_scripts"
    );
    $this->loader->add_action(
      "wp_enqueue_scripts",
      $plugin_public,
      "enqueue_scripts"
    );
    $this->loader->add_action(
      "oxygen_enqueue_ui_scripts",
      $plugin_public,
      "oxy_enqueue_scripts"
    );
    
    $oxymade_hoversPanel = get_option("oxymade_hoverstyles");
    if (isset($oxymade_hoversPanel) && $oxymade_hoversPanel == "Disable") {
      } else {
        
    $this->loader->add_action( 
      "oxygen_after_toolbar", 
      $plugin_public, 
      "oxymade_hovers_panel" 
    );
  }
    
    $this->loader->add_action( 
      "oxygen_add_plus_oxymade_helpers", 
      $plugin_public, 
      "om_side_helpers" 
    );
    
    // $this->loader->add_action( 
    //   "oxygen_after_toolbar", 
    //   $plugin_public, 
    //   "checkRequired" 
    // );
    //TODO: chek if it is necessary above.
  }

  /**
   * Run the loader to execute all of the hooks with WordPress.
   *
   * @since    1.0.0
   */
  public function run()
  {
    $this->loader->run();
  }

  /**
   * The name of the plugin used to uniquely identify it within the context of
   * WordPress and to define internationalization functionality.
   *
   * @since     1.0.0
   * @return    string    The name of the plugin.
   */
  public function get_plugin_name()
  {
    return $this->plugin_name;
  }

  /**
   * The reference to the class that orchestrates the hooks with the plugin.
   *
   * @since     1.0.0
   * @return    Oxymade_Loader    Orchestrates the hooks of the plugin.
   */
  public function get_loader()
  {
    return $this->loader;
  }

  /**
   * Retrieve the version number of the plugin.
   *
   * @since     1.0.0
   * @return    string    The version number of the plugin.
   */
  public function get_version()
  {
    return $this->version;
  }
}

OxyMade::init();
?>
