<?php
/**
 * @package Polylang-WC
 */

/**
 * Expose the product language and translations in the REST API.
 *
 * @since 1.1
 */
class PLLWC_REST_Product extends PLL_REST_Translated_Object {

	/**
	 * Constructor.
	 *
	 * @since 1.1
	 */
	public function __construct() {
		// FIXME Backward compatibility with Polylang < 2.6.
		$instance = version_compare( POLYLANG_VERSION, '2.6-dev', '<' ) ? PLL()->model : PLL()->rest_api;

		parent::__construct( $instance, array( 'product' => array( 'filters' => false ) ) );

		$this->type = 'post';
		$this->id   = 'ID';

		$this->data_store = PLLWC_Data_Store::load( 'product_language' );

		// FIXME Backward compatibility with Polylang Pro < 2.7.
		if ( method_exists( 'PLL_REST_Translated_Object', 'query' ) ) {
			add_filter( 'woocommerce_rest_product_object_query', array( $this, 'query' ), 10, 2 );
		}

		add_filter( 'get_terms_args', array( $this, 'get_terms_args' ) ); // Before Auto translate.

		add_filter( 'pllwc_language_for_unique_sku', array( $this, 'language_for_unique_sku' ) );
	}

	/**
	 * Returns the object language.
	 *
	 * @since 1.1
	 *
	 * @param array $object Product array.
	 * @return string
	 */
	public function get_language( $object ) {
		return $this->data_store->get_language( $object['id'] );
	}

	/**
	 * Sets the object language.
	 *
	 * @since 1.1
	 *
	 * @param string $lang   Language code.
	 * @param object $object Instance of WC_Product.
	 * @return bool
	 */
	public function set_language( $lang, $object ) {
		if ( $object instanceof WC_Product ) {
			$this->data_store->set_language( $object->get_id(), $lang );
		} else {
			parent::set_language( $lang, $object );
		}
		return true;
	}

	/**
	 * Returns the object translations.
	 *
	 * @since 1.1
	 *
	 * @param array $object Product array.
	 * @return array
	 */
	public function get_translations( $object ) {
		return $this->data_store->get_translations( $object['id'] );
	}

	/**
	 * Save the translations.
	 *
	 * @since 1.1
	 *
	 * @param array  $translations Array of translations with language codes as keys and object ids as values.
	 * @param object $object       Instance of WC_Product.
	 * @return bool
	 */
	public function save_translations( $translations, $object ) {
		if ( $object instanceof WC_Product ) {
			$translations[ $this->data_store->get_language( $object->get_id() ) ] = $object->get_id();
			$this->data_store->save_translations( $translations );
		} else {
			parent::save_translations( $translations, $object );
		}
		return true;
	}

	/**
	 * Deactivate Auto translate to allow queries of attribute terms in the right language.
	 *
	 * @since 1.1
	 *
	 * @param array $args WP_Term_Query arguments.
	 * @return array
	 */
	public function get_terms_args( $args ) {
		if ( ! empty( $args['include'] ) ) {
			$args['lang'] = '';
		}
		return $args;
	}

	/**
	 * Returns the language to use when searching if a sku is unique.
	 * Requires Polylang Pro 2.7+
	 *
	 * @since 1.3
	 *
	 * @param PLL_Language $language Language for unique sku.
	 * @return PLL_Language
	 */
	public function language_for_unique_sku( $language ) {
		if ( isset( $this->request['lang'] ) && in_array( $this->request['lang'], $this->model->get_languages_list( array( 'fields' => 'slug' ) ) ) ) {
			$language = PLL()->model->get_language( $this->request['lang'] );
		} elseif ( isset( $this->params['lang'] ) && in_array( $this->params['lang'], $this->model->get_languages_list( array( 'fields' => 'slug' ) ) ) ) {
			// Backward compatibility with Polylang Pro < 3.2.
			$language = PLL()->model->get_language( $this->params['lang'] );
		}

		return $language;
	}
}
