<?php

/**
 * General Tab.
 * 
 * @class QTS_Settings_General
 * @package Class
 */
class QTS_Settings_General extends QTS_Abstract_Settings {

	/**
	 * QTS_Settings_General constructor.
	 */
	public function __construct() {
		$this->id            = 'general';
		$this->label         = __( 'General', 'quote-request-for-woocommerce' );
		$this->custom_fields = array(
			'get_quote_basket_menu',
			'get_shortcodes_and_its_usage',
			'get_include_product_selector',
			'get_exclude_product_selector',
			'get_include_product_category_selector',
			'get_exclude_product_category_selector',
			'get_include_product_tag_selector',
			'get_exclude_product_tag_selector',
			'get_include_product_brand_selector',
			'get_exclude_product_brand_selector',
			'get_included_users',
			'get_excluded_users',
		);
		$this->settings      = $this->get_settings();
		$this->init();
	}

	/**
	 * Get settings array.
	 *
	 * @return array
	 */
	public function get_settings() {
		/**
		 * Get the tab settings.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_' . $this->id . '_settings', array(
			array( 'type' => $this->get_custom_field_type( 'get_shortcodes_and_its_usage' ) ),
			array(
				'name' => __( 'General Settings', 'quote-request-for-woocommerce' ),
				'type' => 'title',
				'id'   => $this->prefix . 'general_settings',
			),
			array(
				'name'    => __( 'Enable Quote Request', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'quote_request_enabled',
				'type'    => 'checkbox',
				'default' => 'no',
				'desc'    => __( 'When enabled, your customers can start submitting Quote Requests for products in your shop', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'     => __( 'Working Mode', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'quote_request_mode',
				'type'     => 'select',
				'options'  => array(
					'approx'      => __( 'Approximate Quote(Excluding Shipping and Other Costs)', 'quote-request-for-woocommerce' ),
					'accurate'    => __( 'Accurate Quote(Final Amount to be Paid)', 'quote-request-for-woocommerce' ),
					'user-choice' => __( 'Let the User Decide', 'quote-request-for-woocommerce' ),
				),
				'default'  => 'approx',
				'desc'     => __( '<b>Note:</b> "Quote Request" Payment Gateway has to be enabled in order for the user to submit a Quote in "Accurate Quote(Final Amount to be Paid)" mode or "Let the User Decide" mode.', 'quote-request-for-woocommerce' ),
				'desc_tip' => __( 'Approximate Quote - The amount mentioned in the quote will be excluding the shipping and other costs. Accurate Quote - The amount mentioned is the final amount which needs to be paid by your customer.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'     => __( 'Product Selection', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'get_product_select_type',
				'type'     => 'select',
				'options'  => array(
					'all-products'       => __( 'All Products', 'quote-request-for-woocommerce' ),
					'include-products'   => __( 'Include Products', 'quote-request-for-woocommerce' ),
					'exclude-products'   => __( 'Exclude Products', 'quote-request-for-woocommerce' ),
					'include-categories' => __( 'Include Categories', 'quote-request-for-woocommerce' ),
					'exclude-categories' => __( 'Exclude Categories', 'quote-request-for-woocommerce' ),
					'include-tags'       => __( 'Include Tag(s)', 'quote-request-for-woocommerce' ),
					'exclude-tags'       => __( 'Exclude Tag(s)', 'quote-request-for-woocommerce' ),
					'include-brands'     => __( 'Include Brand(s)', 'quote-request-for-woocommerce' ),
					'exclude-brands'     => __( 'Exclude Brand(s)', 'quote-request-for-woocommerce' ),
				),
				'default'  => 'all-products',
				'desc_tip' => __( 'By default, the "Add to Quote" button will be visible to all products. You can also restrict the display of "Add to Quote" button to specific products and categories.', 'quote-request-for-woocommerce' ),
			),
			array(
				'type' => $this->get_custom_field_type( 'get_include_product_selector', array() ),
			),
			array(
				'type' => $this->get_custom_field_type( 'get_exclude_product_selector', array() ),
			),
			array(
				'type' => $this->get_custom_field_type( 'get_include_product_category_selector' ),
			),
			array(
				'type' => $this->get_custom_field_type( 'get_exclude_product_category_selector' ),
			),
			array(
				'type' => $this->get_custom_field_type( 'get_include_product_tag_selector', array() ),
			),
			array(
				'type' => $this->get_custom_field_type( 'get_exclude_product_tag_selector', array() ),
			),
			array(
				'type' => $this->get_custom_field_type( 'get_include_product_brand_selector', array() ),
			),
			array(
				'type' => $this->get_custom_field_type( 'get_exclude_product_brand_selector', array() ),
			),
			array(
				'name'     => __( 'User Selection', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'get_user_select_type',
				'type'     => 'select',
				'options'  => array(
					'all-users'         => __( 'All Users', 'quote-request-for-woocommerce' ),
					'include-users'     => __( 'Include Users', 'quote-request-for-woocommerce' ),
					'exclude-users'     => __( 'Exclude Users', 'quote-request-for-woocommerce' ),
					'include-userroles' => __( 'Include User Roles', 'quote-request-for-woocommerce' ),
					'exclude-userroles' => __( 'Exclude User Roles', 'quote-request-for-woocommerce' ),
				),
				'default'  => 'all-users',
				'desc_tip' => __( 'By default, all logged-in users can request for a quote. You can also restrict the quote request functionality to specific Users and User Roles.', 'quote-request-for-woocommerce' ),
			),
			array( 'type' => $this->get_custom_field_type( 'get_included_users' ) ),
			array( 'type' => $this->get_custom_field_type( 'get_excluded_users' ) ),
			array(
				'name'    => __( 'Include Userrole', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'get_included_userroles',
				'type'    => 'multiselect',
				'class'   => 'wc-enhanced-select',
				'options' => _qts_get_user_roles(),
				'default' => array(),
			),
			array(
				'name'    => __( 'Exclude Userrole', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'get_excluded_userroles',
				'type'    => 'multiselect',
				'class'   => 'wc-enhanced-select',
				'options' => _qts_get_user_roles(),
				'default' => array(),
			),
			array(
				'name'    => __( 'Display Add to Quote Button for Guests', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'display_add_to_quote_for_guests',
				'type'    => 'checkbox',
				'default' => 'yes',
			),
			array(
				'name'     => __( 'When Guests Click the "Add to Quote" Button', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'when_guest_clicked_add_to_quote',
				'type'     => 'select',
				'default'  => 'allow',
				'options'  => array(
					'allow'       => __( 'Allow them to Submit Quote', 'quote-request-for-woocommerce' ),
					'force-login' => __( 'Force the User to Login / Signup to the Site', 'quote-request-for-woocommerce' ),
				),
				'desc_tip' => __( 'By default, guest users can submit quote. If you want to force your customers to register before submiting a quote, select "Force the User to Login / Signup to the Site".', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'     => __( 'Minimum Quote Request Price Percentage', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'min_quote_req_price_percentage',
				'type'     => 'number',
				'default'  => '50',
				'desc_tip' => __( 'While requesting for a quote, your customers will be not be able to request less than the value set in this option for a product.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'     => __( 'Allow Customers to Add More than the Available Inventory', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'allow_add_to_quote_more_than_available_inventory',
				'type'     => 'checkbox',
				'default'  => 'yes',
				'desc_tip' => __( 'When enabled, your customers can add more than the available inventory while submitting a quote request.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'    => __( 'Allow Customer to Make Payment when the Product(s) is Low in Stock/Out of Stock', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'make_payment_when_not_having_enough_stock',
				'type'    => 'checkbox',
				'default' => 'no',
			),
			array( 'type' => 'sectionend', 'id' => $this->prefix . 'general_settings' ),
			array(
				'name' => __( 'Quote Form Settings', 'quote-request-for-woocommerce' ),
				'type' => 'title',
				'id'   => $this->prefix . 'form_settings',
			),
			array(
				'name'     => __( 'Quote Request Page', 'quote-request-for-woocommerce' ),
				'id'       => 'woocommerce_qts_quote_page_id',
				'type'     => 'single_select_page',
				'class'    => 'wc-enhanced-select-nostd',
				'css'      => 'min-width:300px;',
				'default'  => wc_get_page_id( 'qts_quote' ),
				/* translators: 1: quote page shortcode */
				'desc'     => sprintf( __( 'This page will used for submitting Quote Requests in "Approximate Quote" Working Mode. Page contents: [%1$s]', 'quote-request-for-woocommerce' ), 'qts_quote' ),
				'desc_tip' => true,
			),
			array(
				'name'    => __( 'Display Request Price Field', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'display_request_price_field',
				'type'    => 'checkbox',
				'default' => 'yes',
				'desc'    => __( 'When enabled, your customers will be able to enter their preferred price/percentage while submitting for a quote.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'    => __( 'Request Price Field Type', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'request_price_field_type',
				'type'    => 'select',
				'options' => array(
					'fixed'            => __( 'Fixed', 'quote-request-for-woocommerce' ),
					'percent'          => __( 'Percentage', 'quote-request-for-woocommerce' ),
					'percent-discount' => __( 'Percentage Discount', 'quote-request-for-woocommerce' ),
				),
				'default' => 'fixed',
			),
			array( 'type' => 'sectionend', 'id' => $this->prefix . 'form_settings' ),
			array(
				'name' => __( 'Display Settings', 'quote-request-for-woocommerce' ),
				'type' => 'title',
				'id'   => $this->prefix . 'display_settings',
			),
			array( 'type' => $this->get_custom_field_type( 'get_quote_basket_menu' ) ),
			array(
				'name'     => __( 'Request Quote Input Type', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'request_quote_input_type',
				'type'     => 'select',
				'options'  => array(
					'button' => __( 'Button', 'quote-request-for-woocommerce' ),
					'link'   => __( 'Link', 'quote-request-for-woocommerce' ),
				),
				'default'  => 'button',
				'desc_tip' => __( 'Choose whether you want to display the "Add to Quote" button as a Button or as a hyperlink.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'     => __( 'Display Request Quote Button/Link on', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'display_request_quote_page_in',
				'type'     => 'multiselect',
				'class'    => 'wc-enhanced-select',
				'options'  => array(
					'single-product' => __( 'Single Product Pages', 'quote-request-for-woocommerce' ),
					'other-pages'    => __( 'Other Pages', 'quote-request-for-woocommerce' ),
				),
				'default'  => array( 'single-product', 'other-pages' ),
				'desc_tip' => __( 'The "Add to Quote" button will be displayed on the selected locations.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'     => __( 'Hide Add to Cart for the Products', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'hide_add_to_cart',
				'type'     => 'checkbox',
				'default'  => 'no',
				'desc_tip' => __( 'When enabled, the "Add to Cart" button will be hidden for the products in which the Quote is enabled.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'     => __( 'Hide Preferred Subtotal', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'hide_preferred_subtotal',
				'type'     => 'checkbox',
				'default'  => 'no',
				'desc_tip' => __( 'When enabled, the preferred subtotal field/column will be hidden in quotation page.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'     => __( 'Hide Price for the Products', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'hide_product_price',
				'type'     => 'checkbox',
				'default'  => 'no',
				'desc_tip' => __( 'When enabled, the "Price" will be hidden for the products in which the Quote is enabled.', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'    => __( 'Display "Add to Quote" Button/Link for Out of Stock Products', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'display_add_to_quote_for_out_of_stock',
				'type'    => 'checkbox',
				'default' => 'no',
			),
			array(
				'name'    => __( 'Display "Add to Quote" Button/Link Only for Out of Stock Products', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'display_add_to_quote_only_for_out_of_stock',
				'type'    => 'checkbox',
				'default' => 'no',
			),
			array(
				'name'    => __( 'Add to Quote Behaviour', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'quote_redirect_after_add',
				'type'    => 'checkbox',
				'default' => 'no',
				'desc'    => __( 'Redirect to the Quote Page after Successful Addition', 'quote-request-for-woocommerce' ),
			),
			array(
				'id'      => $this->prefix . 'ajax_add_to_quote_enabled',
				'type'    => 'checkbox',
				'default' => 'no',
				'desc'    => __( 'Enable AJAX Add to Quote Buttons on Archives', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'              => __( 'Custom CSS', 'quote-request-for-woocommerce' ),
				'id'                => $this->prefix . 'inline_style',
				'type'              => 'textarea',
				'default'           => '',
				'custom_attributes' => array(
					'rows' => 10,
					'cols' => 10,
				),
			),
			array( 'type' => 'sectionend', 'id' => $this->prefix . 'display_settings' ),
			array(
				'name' => __( 'Convert to Quote', 'quote-request-for-woocommerce' ),
				'type' => 'title',
				'id'   => $this->prefix . 'convert_to_quote',
			),
			array(
				'title' => __('Convert Cart Items to Quote', 'quote-request-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',              
				'id' => $this->prefix . 'enable_convert_to_quote',
			),
			array(
				'title' => __('Display Convert to Quote Button in Cart Page', 'quote-request-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'yes',              
				'id' => $this->prefix . 'display_convert_to_quote_button_in_cart_page', 
				'desc_tip' => __( 'This option works only for the classic cart page. The button will always be displayed for the block cart.', 'quote-request-for-woocommerce' ),             
			),
			array(
				'title' => __('Convert to Quote Button Position in Cart Page', 'quote-request-for-woocommerce'),
				'id' => $this->prefix . 'convert_to_quote_button_position_in_cart_page',                
				'type' => 'select',
				'default' => 'proceed_to_checkout',             
				'options' => array(
					'before_cart_totals'             => 'Woocommerce Before Cart Totals',
					'cart_totals_before_order_total' => 'Woocommerce Cart Totals Before Order Total',
					'proceed_to_checkout'            => 'Woocommerce Proceed to Checkout',
				),
				'desc' => __('Positioning will work only for the classic cart page. For positioning in block based cart page, edit the cart page and move the "Convert to Quote" block as per your needs.', 'quote-request-for-woocommerce'),
			),
			array(
				'title' => __('Display Convert to Quote Button in Checkout Page', 'quote-request-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'yes',              
				'id' => $this->prefix . 'display_convert_to_quote_button_in_checkout_page',  
				'desc_tip' => __( 'This option works only for the classic checkout page. The button will always be displayed for the block checkout.', 'quote-request-for-woocommerce' ),            
			),
			array(
				'title' => __('Convert to Quote Button Position in Checkout Page', 'quote-request-for-woocommerce'),
				'id' => $this->prefix . 'convert_to_quote_button_position_in_checkout_page',                
				'type' => 'select',
				'default' => 'checkout_order_review',
				'options' => array(
					'checkout_order_review'           => 'Woocommerce Checkout Order Review',
					'checkout_after_customer_details' => 'Woocommerce Checkout After Customer Details',
					'before_checkout_form'            => 'Woocommerce Before Checkout Form',
					'checkout_after_order_review'     => 'Woocommerce Checkout After Order Review',
				),
				'desc' => __('Positioning will work only for the classic checkout page. For positioning in block based checkout page, edit the checkout page and move the "Convert to Quote" block as per your needs.', 'quote-request-for-woocommerce'),
			),
			array(
				'title' => __('Display Back to Cart Button in Quote Page', 'quote-request-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'yes',             
				'id' => $this->prefix . 'display_back_to_cart_button_in_quote_page',
			),
			array( 'type' => 'sectionend', 'id' => $this->prefix . 'convert_to_quote' ),
			array(
				'name' => __( 'Validity Settings', 'quote-request-for-woocommerce' ),
				'type' => 'title',
				'id'   => $this->prefix . 'validity_settings',
			),
			array(
				'name'              => __( 'Customer Response Time', 'quote-request-for-woocommerce' ),
				'id'                => $this->prefix . 'customer_response_time',
				'type'              => 'number',
				'default'           => '7',
				'custom_attributes' => array(
					'min' => '1',
					'max' => '30',
				),
				'desc'              => __( 'In Days', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'    => __( 'Remind Customer to Respond in', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'remind_customer_to_respond_in',
				'type'    => 'text',
				'default' => '1,2,3',
				'desc'    => __( 'Days From Receiving an Reply from the Admin', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'              => __( 'No. Of Days for Making Payment for Approved Quote', 'quote-request-for-woocommerce' ),
				'id'                => $this->prefix . 'due_days_to_make_payment',
				'type'              => 'number',
				'default'           => '7',
				'custom_attributes' => array(
					'min' => '1',
					'max' => '30',
				),
				'desc'              => __( 'Days from Accepting the Offer', 'quote-request-for-woocommerce' ),
			),
			array(
				'name'    => __( 'Send Payment Reminders', 'quote-request-for-woocommerce' ),
				'id'      => $this->prefix . 'remind_customer_to_pay',
				'type'    => 'text',
				'default' => '1,2,3',
				'desc'    => __( 'From the Date of Accepting the Offer', 'quote-request-for-woocommerce' ),
			),
			array( 'type' => 'sectionend', 'id' => $this->prefix . 'validity_settings' ),
			array(
				'name' => __( 'Payment Settings', 'quote-request-for-woocommerce' ),
				'type' => 'title',
				'id'   => $this->prefix . 'payment_settings',
			),
			array(
				'name'     => __( 'Disable Payment Gateways while Making Payment for Quote Requests', 'quote-request-for-woocommerce' ),
				'id'       => $this->prefix . 'disabled_payment_gateways_while_making_payment',
				'type'     => 'multiselect',
				'class'    => 'wc-enhanced-select',
				'options'  => _qts_get_valid_payment_gateways(),
				'default'  => array(),
				'desc_tip' => __( 'This option controls what are the payment gateways which should be displayed to the customers while making payment for their accepted Quote', 'quote-request-for-woocommerce' ),
			),
			array( 'type' => 'sectionend', 'id' => $this->prefix . 'payment_settings' ),
				) );
	}

	/**
	 * Custom type field.
	 */
	public function get_shortcodes_and_its_usage() {
		$shortcodes = array(
			'[qts_quote]' => __( 'Shortcode to display the Quote Request Page.', 'quote-request-for-woocommerce' ),
		);
		?>
		<table class="widefat">
			<thead>
				<tr>
					<th><?php esc_html_e( 'Shortcode', 'quote-request-for-woocommerce' ); ?></th>
					<th><?php esc_html_e( 'Purpose', 'quote-request-for-woocommerce' ); ?></th>
				</tr>
			</thead>
			<tbody>                
				<?php foreach ( $shortcodes as $shortcode => $purpose ) : ?>
					<tr>
						<td><?php echo esc_html( $shortcode ); ?></td>
						<td><?php echo esc_html( $purpose ); ?></td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
		<?php
	}

	/**
	 * Custom type field.
	 */
	public function get_include_product_selector() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_include_product_selector"><?php esc_html_e( 'Include Products(s)', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td class="forminp forminp-select">
				<?php
				_qts_search_field( array(
					'class'       => 'wc-product-search',
					'id'          => $this->prefix . 'get_include_product_selector',
					'type'        => 'product',
					'action'      => 'woocommerce_json_search_products_and_variations',
					'title'       => __( 'Select Product(s) to Include', 'quote-request-for-woocommerce' ),
					'placeholder' => __( 'Search for a product&hellip;', 'quote-request-for-woocommerce' ),
					'options'     => get_option( $this->prefix . 'get_include_product_selector' ),
				) );
				?>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field.
	 */
	public function get_exclude_product_selector() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_exclude_product_selector"><?php esc_html_e( 'Exclude Product(s)', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td class="forminp forminp-select">
				<?php
				_qts_search_field( array(
					'class'       => 'wc-product-search',
					'id'          => $this->prefix . 'get_exclude_product_selector',
					'type'        => 'product',
					'action'      => 'woocommerce_json_search_products_and_variations',
					'title'       => __( 'Select Product(s) to Include', 'quote-request-for-woocommerce' ),
					'placeholder' => __( 'Search for a product&hellip;', 'quote-request-for-woocommerce' ),
					'options'     => get_option( $this->prefix . 'get_exclude_product_selector' ),
				) );
				?>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field.
	 */
	public function get_include_product_category_selector() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_include_product_category_selector"><?php esc_html_e( 'Select Categorie(s) to Include', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td>                
				<select class="wc-enhanced-select" name="_qts_get_include_product_category_selector[]" id="_qts_get_include_product_category_selector" multiple="multiple" style="min-width:350px;">
					<?php
					$option_value = get_option( "{$this->prefix}get_include_product_category_selector", array() );

					foreach ( _qts_get_product_terms() as $key => $val ) {
						?>
						<option value="<?php echo esc_attr( $key ); ?>"
						<?php
						if ( is_array( $option_value ) ) {
							selected( in_array( ( string ) $key, $option_value, true ), true );
						} else {
							selected( $option_value, ( string ) $key );
						}
						?>
								>
									<?php echo esc_html( $val ); ?>
						</option>
						<?php
					}
					?>
				</select>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field.
	 */
	public function get_exclude_product_category_selector() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_exclude_product_category_selector"><?php esc_html_e( 'Select Categorie(s) to Exclude', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td>                
				<select class="wc-enhanced-select" name="_qts_get_exclude_product_category_selector[]" id="_qts_get_exclude_product_category_selector" multiple="multiple" style="min-width:350px;">
					<?php
					$option_value = get_option( "{$this->prefix}get_exclude_product_category_selector", array() );

					foreach ( _qts_get_product_terms() as $key => $val ) {
						?>
						<option value="<?php echo esc_attr( $key ); ?>"
						<?php
						if ( is_array( $option_value ) ) {
							selected( in_array( ( string ) $key, $option_value, true ), true );
						} else {
							selected( $option_value, ( string ) $key );
						}
						?>
								>
									<?php echo esc_html( $val ); ?>
						</option>
						<?php
					}
					?>
				</select>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field for include product tag selector.
	 *
	 * @since 3.1.0
	 */
	public function get_include_product_tag_selector() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_include_product_tag_selector"><?php esc_html_e( 'Select Tag(s) to Include', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td>                
				<select class="wc-enhanced-select" name="_qts_get_include_product_tag_selector[]" id="_qts_get_include_product_tag_selector" multiple="multiple" style="min-width:350px;">
					<?php
					$option_value = get_option( "{$this->prefix}get_include_product_tag_selector", array() );

					foreach ( _qts_get_product_terms( 'product_tag' ) as $key => $val ) {
						?>
						<option value="<?php echo esc_attr( $key ); ?>"
						<?php
						if ( is_array( $option_value ) ) {
							selected( in_array( (string) $key, $option_value, true ), true );
						} else {
							selected( $option_value, (string) $key );
						}
						?>
								>
									<?php echo esc_html( $val ); ?>
						</option>
						<?php
					}
					?>
				</select>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field for exclude product tag selector.
	 */
	public function get_exclude_product_tag_selector() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_exclude_product_tag_selector"><?php esc_html_e( 'Select Tag(s) to Exclude', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td>                
				<select class="wc-enhanced-select" name="_qts_get_exclude_product_tag_selector[]" id="_qts_get_exclude_product_tag_selector" multiple="multiple" style="min-width:350px;">
					<?php
					$option_value = get_option( "{$this->prefix}get_exclude_product_tag_selector", array() );

					foreach ( _qts_get_product_terms( 'product_tag' ) as $key => $val ) {
						?>
						<option value="<?php echo esc_attr( $key ); ?>"
						<?php
						if ( is_array( $option_value ) ) {
							selected( in_array( (string) $key, $option_value, true ), true );
						} else {
							selected( $option_value, (string) $key );
						}
						?>
								>
									<?php echo esc_html( $val ); ?>
						</option>
						<?php
					}
					?>
				</select>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field for include product brand selector.
	 *
	 * @since 3.1.0
	 */
	public function get_include_product_brand_selector() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_include_product_brand_selector"><?php esc_html_e( 'Select Brand(s) to Include', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td>                
				<select class="wc-enhanced-select" name="_qts_get_include_product_brand_selector[]" id="_qts_get_include_product_brand_selector" multiple="multiple" style="min-width:350px;">
					<?php
					$option_value = get_option( "{$this->prefix}get_include_product_brand_selector", array() );

					foreach ( _qts_get_product_terms( 'product_brand' ) as $key => $val ) {
						?>
						<option value="<?php echo esc_attr( $key ); ?>"
						<?php
						if ( is_array( $option_value ) ) {
							selected( in_array( (string) $key, $option_value, true ), true );
						} else {
							selected( $option_value, (string) $key );
						}
						?>
								>
									<?php echo esc_html( $val ); ?>
						</option>
						<?php
					}
					?>
				</select>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field for exclude product brand selector.
	 * 
	 * @since 3.1.0
	 */
	public function get_exclude_product_brand_selector() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_exclude_product_brand_selector"><?php esc_html_e( 'Select Brand(s) to Exclude', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td>                
				<select class="wc-enhanced-select" name="_qts_get_exclude_product_brand_selector[]" id="_qts_get_exclude_product_brand_selector" multiple="multiple" style="min-width:350px;">
					<?php
					$option_value = get_option( "{$this->prefix}get_exclude_product_brand_selector", array() );

					foreach ( _qts_get_product_terms( 'product_brand' ) as $key => $val ) {
						?>
						<option value="<?php echo esc_attr( $key ); ?>"
						<?php
						if ( is_array( $option_value ) ) {
							selected( in_array( (string) $key, $option_value, true ), true );
						} else {
							selected( $option_value, (string) $key );
						}
						?>
								>
									<?php echo esc_html( $val ); ?>
						</option>
						<?php
					}
					?>
				</select>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field.
	 */
	public function get_included_users() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_included_users"><?php esc_html_e( 'Include User(s)', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td class="forminp forminp-select">
				<?php
				_qts_search_field( array(
					'class'       => 'wc-customer-search',
					'id'          => $this->prefix . 'get_included_users',
					'type'        => 'customer',
					'multiple'    => true,
					'placeholder' => __( 'Search for a user&hellip;', 'quote-request-for-woocommerce' ),
					'options'     => get_option( $this->prefix . 'get_included_users' ),
				) );
				?>
			</td>
		</tr>
		<?php
	}

	/**
	 * Custom type field.
	 */
	public function get_excluded_users() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="get_excluded_users"><?php esc_html_e( 'Exclude User(s)', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td class="forminp forminp-select">
				<?php
				_qts_search_field( array(
					'class'       => 'wc-customer-search',
					'id'          => $this->prefix . 'get_excluded_users',
					'type'        => 'customer',
					'multiple'    => true,
					'placeholder' => __( 'Search for a user&hellip;', 'quote-request-for-woocommerce' ),
					'options'     => get_option( $this->prefix . 'get_excluded_users' ),
				) );
				?>
			</td>
		</tr>
		<?php
	}

	/**
	 * Quote Basket Menu
	 * 
	 * @since 2.7.0
	 */
	public function get_quote_basket_menu() {
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $this->prefix ); ?>get_quote_basket_menu"><?php esc_html_e( 'Mini Quote Menu', 'quote-request-for-woocommerce' ); ?></label>
			</th>
			<td class="forminp forminp-select">
				<?php
				$menus      = ( array ) get_terms( 'nav_menu' );
				$quote_menu = get_option( $this->prefix . 'get_quote_basket_menu' );

				foreach ( $menus as $menu ) :
					?>
					<input type="checkbox" name="<?php echo esc_attr( $this->prefix ); ?>get_quote_basket_menu[]" id="<?php echo esc_attr( $this->prefix ); ?>get_quote_basket_menu" value="<?php echo intval( esc_attr( $menu->term_id ) ); ?>" <?php echo in_array( $menu->term_id, ( array ) $quote_menu ) ? 'checked' : ''; ?> />
					<?php echo esc_html( $menu->name ); ?>
					<br>
				<?php endforeach; ?>
				<p class="description">
					<?php echo wp_kses_post( __( "Select the menu where the Mini-Quote has to be displayed at Frontend. Please create a new menu in WordPress[Appearance -> Menus] if there is no menu available here. <br> <b>Note:</b> Some themes doesn't support all the menu types. If there is any conflict in Mini Quote, then try changing to a different menu type and check.", 'quote-request-for-woocommerce' ) ); ?>
				</p>
			</td>
		</tr>
		<?php
	}

	/**
	 * Delete the custom options.
	 */
	public function custom_types_delete_options( $posted = null ) {
		delete_option( $this->prefix . 'get_include_product_selector' );
		delete_option( $this->prefix . 'get_exclude_product_selector' );
		delete_option( $this->prefix . 'get_include_product_category_selector' );
		delete_option( $this->prefix . 'get_exclude_product_category_selector' );
		delete_option( $this->prefix . 'get_include_product_tag_selector' );
		delete_option( $this->prefix . 'get_exclude_product_tag_selector' );
		delete_option( $this->prefix . 'get_include_product_brand_selector' );
		delete_option( $this->prefix . 'get_exclude_product_brand_selector' );
		delete_option( $this->prefix . 'get_included_users' );
		delete_option( $this->prefix . 'get_excluded_users' );
		delete_option( $this->prefix . 'get_quote_basket_menu' );
	}

	/**
	 * Save custom settings.
	 */
	public function custom_types_save( $posted ) {
		$metakeys = array(
			'get_include_product_selector'          => 'multisearch',
			'get_exclude_product_selector'          => 'multisearch',
			'get_include_product_category_selector' => 'multisearch',
			'get_exclude_product_category_selector' => 'multisearch',
			'get_include_product_tag_selector'      => 'multisearch',
			'get_exclude_product_tag_selector'      => 'multisearch',
			'get_include_product_brand_selector'    => 'multisearch',
			'get_exclude_product_brand_selector'    => 'multisearch',
			'get_included_users'                    => 'multisearch',
			'get_excluded_users'                    => 'multisearch',
			'get_quote_basket_menu'                 => 'checkbox',
		);

		foreach ( $metakeys as $key => $type ) {
			if ( ! isset( $posted[ "{$this->prefix }{$key}" ] ) ) {
				continue;
			}

			$posted_value = wc_clean( $posted[ "{$this->prefix }{$key}" ] );

			if ( 'multisearch' === $type ) {
				if ( ! is_array( $posted_value ) ) {
					$value = array_filter( array_map( 'absint', explode( ',', $posted_value ) ) );
				} else {
					$value = $posted_value;
				}
			} else {
				$value = $posted_value;
			}

			update_option( "{$this->prefix }{$key}", wc_clean( $value ) );
		}
	}

	/**
	 * Save the custom options once.
	 */
	public function custom_types_add_options( $posted = null ) {
		add_option( $this->prefix . 'get_include_product_selector', array() );
		add_option( $this->prefix . 'get_exclude_product_selector', array() );
		add_option( $this->prefix . 'get_include_product_category_selector', array() );
		add_option( $this->prefix . 'get_exclude_product_category_selector', array() );
		add_option( $this->prefix . 'get_include_product_tag_selector', array() );
		add_option( $this->prefix . 'get_exclude_product_tag_selector', array() );
		add_option( $this->prefix . 'get_include_product_brand_selector', array() );
		add_option( $this->prefix . 'get_exclude_product_brand_selector', array() );
		add_option( $this->prefix . 'get_included_users', array() );
		add_option( $this->prefix . 'get_excluded_users', array() );
		add_option( $this->prefix . 'get_quote_basket_menu', array() );
	}
}

return new QTS_Settings_General();
