<?php

defined( 'ABSPATH' ) || exit;

/**
 * Handle enqueues.
 * 
 * @class QTS_Enqueues
 * @package Class
 */
class QTS_Enqueues {

	/**
	 * Init QTS_Enqueues.
	 */
	public static function init() {
		add_action( 'admin_enqueue_scripts', __CLASS__ . '::admin_script' );
		add_action( 'admin_enqueue_scripts', __CLASS__ . '::admin_style' );
		add_action( 'wp_enqueue_scripts', __CLASS__ . '::frontend_script' );
		add_action( 'wp_enqueue_scripts', __CLASS__ . '::frontend_style' );
		add_filter( 'woocommerce_screen_ids', __CLASS__ . '::load_wc_enqueues', 1 );
	}

	/**
	 * Register and enqueue a script for use.
	 *
	 * @uses   wp_enqueue_script()
	 * @param  string   $handle
	 * @param  string   $path
	 * @param  array   $localize_data
	 * @param  string[] $deps
	 * @param  string   $version
	 * @param  boolean  $in_footer
	 */
	public static function enqueue_script( $handle, $path = '', $localize_data = array(), $deps = array( 'jquery' ), $version = QTS_VERSION, $in_footer = false ) {
		wp_register_script( $handle, $path, $deps, $version, $in_footer );

		$name = str_replace( '-', '_', $handle );
		wp_localize_script( $handle, $name, $localize_data );
		wp_enqueue_script( $handle );
	}

	/**
	 * Register and enqueue a styles for use.
	 *
	 * @uses   wp_enqueue_style()
	 * @param  string   $handle
	 * @param  string   $path
	 * @param  string[] $deps
	 * @param  string   $version
	 * @param  string   $media
	 * @param  boolean  $has_rtl
	 */
	public static function enqueue_style( $handle, $path = '', $deps = array(), $version = QTS_VERSION, $media = 'all', $has_rtl = false ) {
		wp_register_style( $handle, $path, $deps, $version, $media, $has_rtl );
		wp_enqueue_style( $handle );
	}

	/**
	 * Enqueue jQuery Modal.
	 * 
	 * @since 3.0.0
	 */
	public static function enqueue_jQuery_modal() {
		self::enqueue_script( 'jquery-modal', self::get_asset_url( 'js/jquery-modal/jquery.modal.min.js' ) );
		self::enqueue_style( 'jquery-modal', self::get_asset_url( 'js/jquery-modal/jquery.modal.min.css' ) );
	}

	/**
	 * Return asset URL.
	 *
	 * @param string $path
	 * @return string
	 */
	public static function get_asset_url( $path ) {
		return QTS_URL . "/assets/{$path}";
	}

	/**
	 * Perform script localization in backend.
	 */
	public static function admin_script() {
		global $post, $theorder;

		$screen        = get_current_screen();
		$screen_id     = $screen ? $screen->id : '';
		$wc_screen_id  = sanitize_title( __( 'WooCommerce', 'woocommerce' ) );
		$quote_request = $theorder instanceof WC_Order ? $theorder : $post;
		$tiptip = _qts_is_wc_version( '<' , '10.3.0' ) ? 'jquery-tiptip' : 'wc-jquery-tiptip';
		
		// Quote Request Post screens.
		if ( in_array( $screen_id, _qts_get_screen_ids() ) ) {
			self::enqueue_script( 'qts-admin', self::get_asset_url( 'js/admin/qts-admin.js' ), array(
				'ajax_url'                                => admin_url( 'admin-ajax.php' ),
				'post_id'                                 => _qts_is_screen( $screen_id, $wc_screen_id . '_page_wc-orders--qts_quote_request' ) ? \Automattic\WooCommerce\Utilities\OrderUtil::get_post_or_order_id( $quote_request ) : ( isset( $post->ID ) ? $post->ID : '' ),
				'add_private_note_nonce'                  => wp_create_nonce( 'qts-add-quote-request-private-note' ),
				'note_to_customer_nonce'                  => wp_create_nonce( 'qts-send-quote-request-note-to-customer' ),
				'delete_conversation_note_nonce'          => wp_create_nonce( 'delete-order-note' ),
				'i18n_prompt_admin_upon_note_to_customer' => esc_attr__( 'Ensure that you have modified the Product\'s price as per your preference before replying.', 'quote-request-for-woocommerce' ),
				'i18n_went_wrong'                         => esc_attr__( 'Something went wrong!!', 'quote-request-for-woocommerce' ),
				'il8n_confirm_min_offer_price'            => esc_attr__( 'The price you have offered for is making the product as free. Are you wish to continue?', 'quote-request-for-woocommerce' ),
				'il8n_confirm_max_offer_price'            => esc_attr__( 'The price you have offered for is above the original price. Are you wish to continue?', 'quote-request-for-woocommerce' ),
				'i18n_delete_conversation_note'           => __( 'Are you sure you wish to delete this conversation? This action cannot be undone.', 'quote-request-for-woocommerce' ),
					), array( $tiptip ) );

			wp_dequeue_script( 'autosave' ); // Disable WP Auto Save on Edit Page.
		}

		if ( 'edit-qts_form_field' === $screen_id ) {
			self::enqueue_script( 'qts-post-ordering', self::get_asset_url( 'js/admin/qts-post-ordering.js' ), array(), array( 'jquery-ui-sortable' ) );
		}

		// Quote Request Settings screens
		if ( in_array( $screen_id, _qts_get_screen_ids() ) ) {
			self::enqueue_script( 'qts-admin-settings-general', self::get_asset_url( 'js/admin/qts-admin-settings-general.js' ), array(
				'create_virtual_product_nonce' => wp_create_nonce( 'qts-create-virtual-product' ),
				'get_repayment_nonce'          => wp_create_nonce( 'qts-get-repayment-day-of-month' ),
				'i18n_went_wrong'              => esc_attr__( 'Something went wrong!!', 'quote-request-for-woocommerce' ),
			) );
		}
	}

	/**
	 * Load style in backend.
	 */
	public static function admin_style() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';

		if ( in_array( $screen_id, _qts_get_screen_ids() ) ) {
			self::enqueue_style( 'qts-admin', self::get_asset_url( 'css/qts-admin.css' ), array( 'wc-admin-layout' ) );
		}
	}

	/**
	 * Perform script localization in frontend.
	 */
	public static function frontend_script() {
		global $wp;
		self::enqueue_jQuery_modal();
		self::enqueue_script( 'qts-frontend', self::get_asset_url( 'js/frontend/qts-frontend.js' ), array(
			'ajax_url'                             => admin_url( 'admin-ajax.php' ),
			'quote_request_id'                     => isset( $wp->query_vars[ 'view-quote' ] ) ? absint( $wp->query_vars[ 'view-quote' ] ) : 0,
			'add_to_quote_nonce'                   => wp_create_nonce( 'qts-add-to-quote' ),
			'remove_from_quote_nonce'              => wp_create_nonce( 'qts-remove-from-quote' ),
			'quote_mode_selected_nonce'            => wp_create_nonce( 'qts-quote-mode-selected' ),
			'save_customer_note_nonce'             => wp_create_nonce( 'qts-save-customer-note' ),
			'note_to_admin_nonce'                  => wp_create_nonce( 'qts-send-quote-request-note-to-admin' ),
			'i18n_view_quote_label'                => esc_attr__( 'View Quote', 'quote-request-for-woocommerce' ),
			'i18n_went_wrong'                      => esc_attr__( 'Something went wrong!!', 'quote-request-for-woocommerce' ),
			'i18n_confirm_before_flushing_cart'    => esc_attr( trim( get_option( QTS_PREFIX . 'cannot_modify_cart_while_making_payment_notice' ) ) ),
			'i18n_prompt_guest_to_log_in'          => esc_attr( trim( get_option( QTS_PREFIX . 'guest_login_notice' ) ) ),
			'i18n_confirm_before_flushing_mode'    => esc_attr( trim( get_option( QTS_PREFIX . 'when_user_tries_to_use_both_modes_notice' ) ) ),
			'i18n_confirm_before_accept'           => esc_attr( trim( get_option( QTS_PREFIX . 'accept_quote_confirmation_notice' ) ) ),
			'i18n_confirm_before_reject'           => esc_attr( trim( get_option( QTS_PREFIX . 'reject_quote_confirmation_notice' ) ) ),
			'prompt_guest_to_log_in'               => QTS_Add_To_Quote::prompt_guest_to_log_in() ? 'yes' : '',
			'session_mode'                         => QTS_Add_To_Quote::get_mode( true ),
			'empty_cart_when_awaiting_for_payment' => QTS_Cart::awaiting_for_payment() && count( WC()->cart->cart_contents ) > 1 ? 'yes' : '',
			'quote_mode'                           => QTS_Add_To_Quote::get_mode(),
			'back_to_cart_nonce'                   => wp_create_nonce( 'qts-back-to-cart' ),
			'convert_to_quote_nonce'               => wp_create_nonce( 'qts-convert-to-quote' ),
		)  );
	}

	/**
	 * Load style in frontend.
	 */
	public static function frontend_style() {
		global $post, $product;

		self::enqueue_style( 'qts-frontend-inline' );
		self::enqueue_style( 'qts-frontend', self::get_asset_url( 'css/qts-frontend.css' ), array( 'dashicons' ) );

		$css = get_option( QTS_PREFIX . 'inline_style', '' );
		$css .= QTS_Add_To_Quote::maybe_hide_add_to_cart_in_single( $post );

		if ( '' !== $css ) {
			wp_add_inline_style( 'qts-frontend-inline', $css );
		}
	}

	/**
	 * Load WC enqueues.
	 * 
	 * @param array $screen_ids
	 * @return array
	 */
	public static function load_wc_enqueues( $screen_ids ) {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';

		if ( in_array( $screen_id, _qts_get_screen_ids() ) ) {
			$screen_ids[] = $screen_id;
		}

		return $screen_ids;
	}
}

QTS_Enqueues::init();
