<?php

defined( 'ABSPATH' ) || exit;

/**
 * Quote Request.
 * 
 * @class QTS_Quote_Request
 * @package Class
 */
class QTS_Quote_Request extends WC_Order {

	/**
	 * Which data store to load.
	 *
	 * @var string
	 */
	protected $data_store_name = 'qts_quote_request';

	/**
	 * This is the name of this object type.
	 *
	 * @var string
	 */
	protected $object_type = 'qts_quote_request';

	/**
	 * Extra data for this object. Name value pairs.
	 * Used to add additional information to an inherited class.
	 *
	 * @var array
	 */
	protected $extra_data = array(
		'order_id'                   => 0,
		'price_negotiations_started' => 'no',
		'date_expiry'                => null,
		'form_fields'                => array(),
	);

	/**
	 * Get internal type.
	 *
	 * @return string
	 */
	public function get_type() {
		return 'qts_quote_request';
	}

	/**
	 * Get all valid statuses for this quote request.
	 *
	 * @return array Internal status keys e.g. QTS_PREFIX. 'new'
	 */
	public function get_valid_statuses() {
		return array_keys( _qts_get_quote_request_statuses() );
	}

	/*
	  |--------------------------------------------------------------------------
	  | Helper Methods
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Get the requested price type.
	 * 
	 * @return string
	 */
	public function get_requested_price_type() {
		$requested_price_type = 'fixed';

		foreach ( $this->get_items( 'line_item' ) as $item ) {
			if ( is_numeric( $item->get_meta( '_requested_price_percent' ) ) ) {
				$requested_price_type = 'percent';
			} else if ( is_numeric( $item->get_meta( '_requested_discount_percent' ) ) ) {
				$requested_price_type = 'percent-discount';
			}
		}

		return $requested_price_type;
	}

	/**
	 * Returns the customer billing address in html format.
	 * 
	 * @param bool $tips
	 * @return string
	 */
	public function get_customer_billing_address_html( $tips = true ) {
		$billing_address = array(
			'first_name' => '&nbsp;',
			'last_name'  => ",\n",
			'email'      => ",\n",
			'company'    => ",\n",
			'address_1'  => ",\n",
			'address_2'  => ",\n",
			'city'       => ',&nbsp;',
			'state'      => ',&nbsp;',
			'country'    => ",&nbsp;\n",
			'postcode'   => ",\n",
			'phone'      => '',
		);

		$out = '<p>';
		foreach ( $billing_address as $internal_prop => $rep ) {
			$getter = "get_billing_$internal_prop";

			if ( ! is_callable( array( $this, $getter ) ) ) {
				continue;
			}

			if ( '' !== $this->{$getter}() ) {
				$out .= $this->{$getter}();
				$out .= $rep;
			}
		}
		$out .= '</p>';

		if ( $tips ) {
			/* translators: 1: class name 2: data-tip 3: email */
			return sprintf( __( '<a href="#" class="%1$s" data-tip="%2$s">%3$s</a>' ), esc_attr( QTS_PREFIX . 'tips' ), esc_attr( nl2br( $out ) ), esc_html( $this->get_billing_email() ) );
		}

		return nl2br( $out );
	}

	/**
	 * When a payment is complete this function is called.
	 *
	 * @param string $transaction_id Optional
	 * @return bool success
	 */
	public function payment_complete( $transaction_id = '' ) {
		if ( ! $this->get_id() ) {
			return false;
		}

		try {
			/**
			 * Trigger before the quote request payment complete.
			 * 
			 * @param int $quote_request_id 
			 * @since 1.0
			 */
			do_action( 'qts_quote_request_pre_payment_complete', $this->get_id() );

			/**
			 * Valid quote request statuses for payment complete.
			 * 
			 * @since 1.0
			 */
			if ( $this->has_status( apply_filters( 'qts_valid_quote_request_statuses_for_payment_complete', array( 'accepted', 'approved' ), $this ) ) ) {
				if ( ! empty( $transaction_id ) ) {
					$this->set_transaction_id( $transaction_id );
				}

				if ( ! $this->get_date_paid( 'edit' ) ) {
					$this->set_date_paid( time() );
				}

				$this->update_status( 'completed' );

				/**
				 * Trigger after the quote request payment complete.
				 * 
				 * @param int $quote_request_id 
				 * @since 1.0
				 */
				do_action( 'qts_quote_request_payment_complete', $this->get_id() );
			} else {
				/**
				 * Trigger after the quote request already completed.
				 * 
				 * @param int $quote_request_id 
				 * @since 1.0
				 */
				do_action( 'qts_payment_complete_quote_request_status_' . $this->get_status(), $this->get_id() );
			}
		} catch ( Exception $e ) {
			if ( function_exists( 'wc_get_logger' ) ) {
				$logger = wc_get_logger();
				$logger->error( sprintf( 'Error completing payment for quote request #%d', $this->get_id() ), array(
					'quote_request' => $this,
					'error'         => $e,
				) );
			}

			$this->add_order_note( __( 'Quote Request payment complete event failed.', 'quote-request-for-woocommerce' ) . ' ' . $e->getMessage() );
			return false;
		}

		return true;
	}

	/**
	 * Handle the status transition.
	 */
	protected function status_transition() {
		$status_transition = $this->status_transition;

		// Reset status transition variable.
		$this->status_transition = false;

		if ( ! $status_transition ) {
			return;
		}

		try {
			/**
			 * Trigger after the quote request status has been updated to respective status.
			 * 
			 * @param int $quote_request_id 
			 * @param QTS_Quote_Request $this 
			 * @since 1.0
			 */
			do_action( 'qts_quote_request_status_' . $status_transition[ 'to' ], $this->get_id(), $this );

			if ( 'system' !== $status_transition[ 'action_by' ] ) {
				/**
				 * Trigger after the quote request status has been updated to respective status manually.
				 * 
				 * @param int $quote_request_id 
				 * @param QTS_Quote_Request $this 
				 * @since 1.0
				 */
				do_action( 'qts_quote_request_' . $status_transition[ 'to' ] . '_by_' . sanitize_title( $status_transition[ 'action_by' ] ), $this->get_id(), $this );
			}

			if ( ! empty( $status_transition[ 'from' ] ) ) {
				/* translators: 1: old quote request status 2: new quote request status */
				$transition_note = sprintf( __( 'Quote Request status changed from %1$s to %2$s.', 'quote-request-for-woocommerce' ), _qts_get_quote_request_status_name( $status_transition[ 'from' ] ), _qts_get_quote_request_status_name( $status_transition[ 'to' ] ) );
				// Note the transition occurred.
				$this->add_order_note( trim( $status_transition[ 'note' ] . ' ' . $transition_note ), 0, $status_transition[ 'manual' ] );

				/**
				 * Trigger after the quote request status has been updated from and to respective status.
				 * 
				 * @param int $quote_request_id 
				 * @param QTS_Quote_Request $this 
				 * @since 1.0
				 */
				do_action( 'qts_quote_request_status_' . $status_transition[ 'from' ] . '_to_' . $status_transition[ 'to' ], $this->get_id(), $this );

				/**
				 * Trigger after the quote request status has been updated to any status.
				 * 
				 * @param string $from 
				 * @param string $to
				 * @param QTS_Quote_Request $this 
				 * @since 1.0
				 */
				do_action( 'qts_quote_request_status_changed', $this->get_id(), $status_transition[ 'from' ], $status_transition[ 'to' ], $this );
			} else {
				/* translators: %s: new quote request status */
				$transition_note = sprintf( __( 'Quote Request status set to %s.', 'quote-request-for-woocommerce' ), _qts_get_quote_request_status_name( $status_transition[ 'to' ] ) );
				// Note the transition occurred.
				$this->add_order_note( trim( $status_transition[ 'note' ] . ' ' . $transition_note ), 0, $status_transition[ 'manual' ] );
			}
		} catch ( Exception $e ) {
			if ( function_exists( 'wc_get_logger' ) ) {
				$logger = wc_get_logger();
				$logger->error( sprintf( 'Status transition of quote request #%d errored!', $this->get_id() ), array(
					'quote_request' => $this,
					'error'         => $e,
				) );
			}

			$this->add_order_note( __( 'Error during status transition.', 'quote-request-for-woocommerce' ) . ' ' . $e->getMessage() );
		}
	}

	/**
	 * Updates status of quote request immediately.
	 *
	 * @uses QTS_Quote_Request::set_status()
	 * @param string $new_status  Status to change the quote request to. No internal QTS_PREFIX prefix is required.
	 * @param string $note        Optional note to add.
	 * @param bool   $manual      Is this a manual quote request status change?.
	 * @param string $action_by   Is the status update by 'admin'|'customer'?. Default it is 'system'.
	 * @return bool
	 */
	public function update_status( $new_status, $note = '', $manual = false, $action_by = 'system' ) {
		if ( ! $this->get_id() ) { // Quote Request must exist.
			return false;
		}

		try {
			$this->set_status( $new_status, $note, $manual, $action_by );

			// Check with the new status.
			switch ( $this->get_status() ) {
				case 'response_awaitd': // Admin is replied customer needs to response    
					if ( $manual && $this->get_date_expiry( 'edit' ) ) {
						$expiry_date = $this->get_date_expiry( 'edit' )->getTimestamp();
					} else {
						$expiry_date = _qts_prepare_quote_request_expiry_date( $this );
					}

					$this->set_price_negotiations_started( 'yes' );
					$this->schedule_expiry( $expiry_date, 'remind_customer_to_respond_in' );
					break;
				case 'hav_to_respond': // Customer is replied admin needs to response
					$this->set_price_negotiations_started( 'yes' );

					// Cancel all queued jobs since admin don't have any waiting time
					_qts_cancel_all_jobs_from_queue( 'quote_request', $this->get_id() );
					break;
				case 'accepted': // Price negotiations over on both sides send the order invoice to customer to pay for his quote request
				case 'approved': // Admin directly approves the quote request without negotiating the customer requested price 
					if ( $manual && $this->get_date_expiry( 'edit' ) ) {
						$expiry_date = $this->get_date_expiry( 'edit' )->getTimestamp();
					} else {
						$expiry_date = _qts_prepare_quote_request_expiry_date( $this );
					}

					$this->schedule_expiry( $expiry_date, 'remind_customer_to_pay' );
					break;
				case 'completed': // When payment is completed
				case 'expired': // Having no response from the customer
				case 'rejected': // Don't want this offer
					_qts_cancel_all_jobs_from_queue( 'quote_request', $this->get_id() );
					break;
			}

			$this->save();
		} catch ( Exception $e ) {
			if ( function_exists( 'wc_get_logger' ) ) {
				$logger = wc_get_logger();
				$logger->error( sprintf( 'Error updating status for quote request #%d', $this->get_id() ), array(
					'quote_request' => $this,
					'error'         => $e,
				) );
			}

			$this->add_order_note( __( 'Update status event failed.', 'quote-request-for-woocommerce' ) . ' ' . $e->getMessage() );
			return false;
		}
		return true;
	}

	/**
	 * Adds a note (comment) to the quote request. Quote Request must exist.
	 *
	 * @param  string $note              Note to add.
	 * @param  int    $is_customer_note  Is this a note for the customer?.
	 * @param  bool   $added_by_user     Was the note added by a user?.
	 * @param  array  $meta_data         Meta data to add with the note.
	 * @return int                       Comment ID.
	 */
	public function add_order_note( $note, $is_customer_note = 0, $added_by_user = false, $meta_data = array() ) {
		if ( ! $this->get_id() || '' === $note ) {
			return 0;
		}

		if ( is_user_logged_in() && $added_by_user ) {
			$added_by             = current_user_can( 'edit_shop_order', $this->get_id() ) ? 'admin' : 'customer';
			$user                 = get_user_by( 'id', get_current_user_id() );
			$comment_author       = $user->display_name;
			$comment_author_email = $user->user_email;
		} else {
			$added_by             = 'system';
			$comment_author       = __( 'WooCommerce', 'woocommerce' );
			$comment_author_email = strtolower( __( 'WooCommerce', 'woocommerce' ) ) . '@';
			$comment_author_email .= isset( $_SERVER[ 'HTTP_HOST' ] ) ? str_replace( 'www.', '', sanitize_text_field( wp_unslash( $_SERVER[ 'HTTP_HOST' ] ) ) ) : 'noreply.com';
			$comment_author_email = sanitize_email( $comment_author_email );
		}

		$commentdata = array(
			'comment_post_ID'      => $this->get_id(),
			'comment_author'       => $comment_author,
			'comment_author_email' => $comment_author_email,
			'comment_author_url'   => '',
			'comment_content'      => $note,
			'comment_agent'        => 'WooCommerce',
			'comment_type'         => 'order_note',
			'comment_parent'       => 0,
			'comment_approved'     => 1,
		);

		$comment_id = wp_insert_comment( $commentdata );

		// Set it is a quote request note.
		add_comment_meta( $comment_id, 'is_qtsquote_request_note', 1 );

		// Set who added the note.
		add_comment_meta( $comment_id, 'is_qtsquote_request_note_added_by', $added_by );

		// Check if it is added to customer.
		if ( $is_customer_note ) {
			add_comment_meta( $comment_id, 'is_customer_note', 1 );

			/**
			 * Trigger after the new customer note added by someone.
			 * 
			 * @since 1.0
			 */
			do_action( 'qts_quote_request_new_customer_note_added_by_' . $added_by, $comment_id, $commentdata[ 'comment_content' ], $this );

			/**
			 * Trigger after the new customer note added.
			 * 
			 * @since 1.0
			 */
			do_action( 'qts_quote_request_new_customer_note_added', $comment_id, $commentdata[ 'comment_content' ], $this );
		}

		/**
		 * Trigger after the new note added by someone.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_new_quote_request_note_added_by_' . $added_by, $comment_id, $commentdata[ 'comment_content' ], $this );

		/**
		 * Trigger after the new note added.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_new_quote_request_note_added', $comment_id, $commentdata[ 'comment_content' ], $this );
		return $comment_id;
	}

	/**
	 * List available quote request notes excluding conversations.
	 *
	 * @return array
	 */
	public function get_quote_request_notes() {
		$args = array(
			'post_id'    => $this->get_id(),
			'status'     => 'approve',
			'type'       => 'order_note',
			'orderby'    => 'comment_ID',
			'meta_query' => array(
				array(
					'key'     => 'is_customer_note',
					'compare' => 'NOT EXISTS',
				),
		),
		);

		remove_filter( 'the_comments', '_qts_exclude_quote_request_conversations' );
		remove_filter( 'comments_clauses', array( 'WC_Comments', 'exclude_order_comments' ) );
		$comments = get_comments( $args );
		add_filter( 'comments_clauses', array( 'WC_Comments', 'exclude_order_comments' ) );

		return array_filter( array_map( '_qts_get_quote_request_note', $comments ) );
	}

	/**
	 * List available quote request conversation notes.
	 *
	 * @return array
	 */
	public function get_conversation_notes() {
		$args = array(
			'post_id'    => $this->get_id(),
			'status'     => 'approve',
			'type'       => 'order_note',
			'meta_key'   => 'is_customer_note',
			'meta_value' => '1',
			'orderby'    => 'comment_ID',
			'order'      => 'ASC',
		);

		remove_filter( 'the_comments', '_qts_exclude_quote_request_conversations' );
		remove_filter( 'comments_clauses', array( 'WC_Comments', 'exclude_order_comments' ) );
		$comments = get_comments( $args );
		add_filter( 'comments_clauses', array( 'WC_Comments', 'exclude_order_comments' ) );

		return array_filter( array_map( '_qts_get_quote_request_note', $comments ) );
	}

	/**
	 * An alias to get_conversation_notes() method.
	 *
	 * @return array
	 */
	public function get_customer_order_notes() {
		return $this->get_conversation_notes();
	}

	/**
	 * Create quote items.
	 *
	 * @param array $quotes
	 * @param array $types Quote Request item types.
	 */
	public function create_quote_items( $quotes, $types = array( 'line_item' ) ) {
		$types = array_filter( ( array ) $types );

		if ( empty( $types ) ) {
			return;
		}

		foreach ( $types as $type ) {
			switch ( $type ) {
				case 'line_item':
					foreach ( $quotes as $quote_item_key => $values ) {
						$item = new WC_Order_Item_Product();
						$item->set_props(
								array(
									'quantity'     => $values[ 'quantity' ],
									'variation'    => $values[ 'variation' ],
									'subtotal'     => $values[ 'line_subtotal' ],
									'total'        => isset( $values[ 'line_total' ] ) ? $values[ 'line_total' ] : $values[ 'line_subtotal' ],
									'subtotal_tax' => isset( $values[ 'line_subtotal_tax' ] ) ? $values[ 'line_subtotal_tax' ] : 0,
									'total_tax'    => isset( $values[ 'line_tax' ] ) ? $values[ 'line_tax' ] : 0,
									'taxes'        => isset( $values[ 'line_tax_data' ] ) ? $values[ 'line_tax_data' ] : array(),
								)
						);

						$product = $values[ 'data' ];
						if ( $product ) {
							$item->set_props(
									array(
										'name'         => $product->get_name(),
										'tax_class'    => $product->get_tax_class(),
										'product_id'   => $product->is_type( 'variation' ) ? $product->get_parent_id() : $product->get_id(),
										'variation_id' => $product->is_type( 'variation' ) ? $product->get_id() : 0,
									)
							);
						}

						// Set quote item metadata.
						$meta_data   = array();
						$meta_data[] = array(
							'id'    => '',
							'key'   => '_is_qts_quote_request',
							'value' => 'yes',
						);

						// Get the preferred price
						if ( isset( $values[ 'qts_quote' ] ) ) {
							$preferred_price = _qts_calculate_preferred_price( $values[ 'qts_quote' ] );
						} else {
							$preferred_price = _qts_calculate_preferred_price( $values );
						}

						if ( is_numeric( $preferred_price ) ) {
							if ( isset( $values[ 'qts_quote' ] ) ) {
								$values[ 'qts_quote' ][ 'requested_price' ] = $preferred_price;
							} else {
								$values[ 'requested_price' ] = $preferred_price;
							}
						}

						$metakeys = array( 'original_price', 'requested_price', 'requested_price_percent', 'requested_discount_percent' );
						foreach ( $metakeys as $key ) {
							if ( isset( $values[ 'qts_quote' ][ $key ] ) ) {
								$meta_data[] = array(
									'id'    => '',
									'key'   => "_{$key}",
									'value' => $values[ 'qts_quote' ][ $key ],
								);
							} else if ( isset( $values[ $key ] ) ) {
								$meta_data[] = array(
									'id'    => '',
									'key'   => "_{$key}",
									'value' => $values[ $key ],
								);
							}
						}

						$item->set_meta_data( $meta_data );

						/**
						 * Action hook to adjust item before save.
						 * 
						 * @since 1.0
						 */
						do_action( 'qts_create_quote_request_line_item', $item, $quote_item_key, $values, $this );

						// Add item to quote request.
						$this->add_item( $item );
					}
					break;
			}
		}
	}

	/**
	 * Get totals for display on pages and in emails.
	 *
	 * @param string $tax_display Tax to display.
	 * @return array
	 */
	public function get_order_item_totals( $tax_display = '' ) {
		$tax_display = $tax_display ? $tax_display : get_option( 'woocommerce_tax_display_cart' );
		$total_rows  = array();

		$this->add_order_item_totals_subtotal_row( $total_rows, $tax_display );
		$this->add_order_item_totals_discount_row( $total_rows, $tax_display );
		$this->add_order_item_totals_shipping_row( $total_rows, $tax_display );
		$this->add_order_item_totals_fee_rows( $total_rows, $tax_display );
		$this->add_order_item_totals_tax_rows( $total_rows, $tax_display );
		$this->add_order_item_totals_total_row( $total_rows, $tax_display );

		/**
		 * Get the quote request item totals.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_quote_request_item_totals', $total_rows, $this, $tax_display );
	}

	/*
	  |--------------------------------------------------------------------------
	  | Conditionals
	  |--------------------------------------------------------------------------
	  |
	  | Checks if a condition is true or false.
	  |
	 */

	/**
	 * Checks if an quote request can be edited, specifically for use on the Edit Quote Request screen.
	 *
	 * @return bool
	 */
	public function is_editable() {
		/**
		 * Is quote request editable?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_quote_request_is_editable', ( ! $this->has_status( array( 'accepted', 'approved', 'rejected', 'expired', 'completed' ) ) ), $this );
	}

	/**
	 * Checks if the given status can be edited based upon the current quote request status, specifically for use on the Edit Quote Request screen.
	 *
	 * @param string $status Status to check whether it is editable. No internal QTS_PREFIX prefix is required.
	 * @return bool
	 */
	public function is_status_editable( $status ) {
		$status   = _qts_trim_post_status( $status );
		$editable = true;

		switch ( $this->get_status() ) {
			case 'new':
				if ( in_array( $status, array( 'response_awaitd', 'hav_to_respond' ) ) ) {
					$editable = false;
				}
				break;
			case 'on_hold':
				if ( in_array( $status, array( 'new', 'response_awaitd', 'hav_to_respond' ) ) ) {
					$editable = false;
				}
				break;
			case 'accepted':
				if ( in_array( $status, array( 'new', 'on_hold', 'approved', 'response_awaitd', 'hav_to_respond' ) ) ) {
					$editable = false;
				}
				break;
			case 'approved':
				if ( in_array( $status, array( 'new', 'on_hold', 'accepted', 'response_awaitd', 'hav_to_respond' ) ) ) {
					$editable = false;
				}
				break;
			case 'response_awaitd':
				if ( in_array( $status, array( 'new', 'on_hold', 'hav_to_respond' ) ) ) {
					$editable = false;
				}
				break;
			case 'hav_to_respond':
				if ( in_array( $status, array( 'new', 'on_hold', 'response_awaitd' ) ) ) {
					$editable = false;
				}
				break;
			default:
				// Do not exclude the current status.
				if ( ! $this->has_status( $status ) ) {
					$editable = false;
				}
				break;
		}

		if ( 'yes' === $this->get_price_negotiations_started() ) {
			if ( 'approved' === $status ) {
				$editable = false;
			}
		} elseif ( 'accepted' === $status ) {
				$editable = false;
		}

		/**
		 * Is quote request status editable?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_quote_request_is_status_editable', $editable, $status, $this );
	}

	/**
	 * Returns if an quote request has been paid for based on the quote request status.
	 *
	 * @return bool
	 */
	public function is_paid() {
		/**
		 * Is quote request paid?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_quote_request_is_paid', $this->has_status( 'completed' ), $this );
	}

	/**
	 * Checks if an quote request needs payment, based on status and quote request total.
	 *
	 * @return bool
	 */
	public function needs_payment() {
		/**
		 * Is quote request needs payment?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_quote_request_needs_payment', ( $this->has_status( array( 'accepted', 'approved' ) ) && $this->get_total() > 0 ), $this );
	}

	/**
	 * Checks if an quote request needs conversation for customer | admin.
	 *
	 * @return bool
	 */
	public function needs_conversation() {
		/**
		 * Is quote request needs conversation?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_quote_request_needs_conversation', ( ! $this->has_status( array( 'accepted', 'approved', 'rejected', 'expired', 'completed' ) ) ), $this );
	}

	/**
	 * Checks if an quote request is valid to expire, based on status.
	 *
	 * @return bool
	 */
	public function valid_to_expire() {
		/**
		 * Is quote request expiry valid?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_quote_request_is_valid_to_expire', $this->has_status( array( 'response_awaitd', 'accepted', 'approved' ) ), $this );
	}

	/**
	 * Check if quote request has been created via quote, checkout or in another way.
	 * quote - Quote Request submitted via quote page.
	 * checkout - Quote Request submitted via checkout page.
	 * 
	 * @param string $modus Way of creating the quote request.
	 * @return bool
	 */
	public function is_created_via( $modus ) {
		/**
		 * Is quote request created via specific mode?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_quote_request_is_created_via', $modus === $this->get_created_via(), $this, $modus );
	}

	/*
	  |--------------------------------------------------------------------------
	  | URLs and Endpoints
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Generates a URL to view an quote request from the my account page.
	 *
	 * @return string
	 */
	public function get_view_order_url() {
		/**
		 * Get quote request view url.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_view_quote_request_url', wc_get_endpoint_url( 'view-quote', $this->get_id(), wc_get_page_permalink( 'myaccount' ) ), $this );
	}

	/**
	 * Get's the URL to edit the quote request in the backend.
	 *
	 * @return string
	 */
	public function get_edit_order_url() {
		/**
		 * Get quote request edit url.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_edit_quote_request_url', get_admin_url( null, 'post.php?post=' . $this->get_id() . '&action=edit' ), $this );
	}

	/**
	 * Generates a URL so that a customer can pay for their quote request which is accepted/approved.
	 *
	 * @param  bool $on_checkout If on checkout.
	 * @return string
	 */
	public function get_checkout_payment_url( $on_checkout = false ) {
		if ( $this->is_created_via( 'checkout' ) ) {
			$order     = wc_get_order( $this->get_order_id() );
			$order_key = $order ? $order->get_order_key() : 'null';

			$pay_url = add_query_arg( array(
				'pay_for_order'    => 'true',
				'key'              => $order_key,
				'qtsquote_request' => $this->get_order_key(),
					), wc_get_endpoint_url( 'order-pay', $this->get_order_id(), wc_get_checkout_url() ) );
		} else {
			$pay_url = add_query_arg( array(
				'qtscheckout'      => 'true',
				'qtsquote_request' => $this->get_order_key(),
					), wc_get_checkout_url() );
		}

		/**
		 * Get quote request checkout payment url.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_checkout_payment_url', $pay_url, $this );
	}

	/**
	 * Generates a URL so that a customer can accept their quote request.
	 *
	 * @param string $redirect Redirect URL.
	 * @return string
	 */
	public function get_accept_quote_request_url( $redirect = '' ) {
		$url_raw = add_query_arg( array(
			'qtsaction'        => 'accept',
			'qtsquote_request' => $this->get_order_key(),
			'qtsredirect'      => $redirect,
				), _qts_get_home_endpoint() );

		/**
		 * Get quote request accept url.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_accept_quote_request_url', wp_nonce_url( $url_raw, 'accept-qtsquote_request' ) );
	}

	/**
	 * Generates a raw (unescaped & hashed) accept quote request URL for use by emails.
	 *
	 * @param string $redirect Redirect URL.
	 * @return string The unescaped accept quote request URL.
	 */
	public function get_accept_quote_request_url_raw( $redirect = '' ) {
		/**
		 * Get quote request accept url raw.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_accept_quote_request_url_raw', add_query_arg( array(
			'qtsaction'        => 'accept',
			'qtsquote_request' => $this->get_order_key(),
			'qtsredirect'      => $redirect,
			'qtskey'           => hash( 'sha256', $this->get_billing_email() ),
						), _qts_get_home_endpoint() ) );
	}

	/**
	 * Generates a URL so that a customer can reject their quote request.
	 *
	 * @param string $redirect Redirect URL.
	 * @return string
	 */
	public function get_reject_quote_request_url( $redirect = '' ) {
		$url_raw = add_query_arg( array(
			'qtsaction'        => 'reject',
			'qtsquote_request' => $this->get_order_key(),
			'qtsredirect'      => $redirect,
				), _qts_get_home_endpoint() );

		/**
		 * Get quote request reject url.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_reject_quote_request_url', wp_nonce_url( $url_raw, 'reject-qtsquote_request' ) );
	}

	/**
	 * Generates a raw (unescaped & hashed) reject quote request URL for use by emails.
	 *
	 * @param string $redirect Redirect URL.
	 * @return string The unescaped reject quote request URL.
	 */
	public function get_reject_quote_request_url_raw( $redirect = '' ) {
		/**
		 * Get quote request reject url raw.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_reject_quote_request_url_raw', add_query_arg( array(
			'qtsaction'        => 'reject',
			'qtsquote_request' => $this->get_order_key(),
			'qtsredirect'      => $redirect,
			'qtskey'           => hash( 'sha256', $this->get_billing_email() ),
						), _qts_get_home_endpoint() ) );
	}

	/*
	  |--------------------------------------------------------------------------
	  | Getters
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Get the order ID.
	 *
	 * @param  string $context View or edit context.
	 * @return int
	 */
	public function get_order_id( $context = 'view' ) {
		return $this->get_prop( 'order_id', $context );
	}

	/**
	 * Get is price negotiations started ?
	 *
	 * @param  string $context View or edit context.
	 * @return int
	 */
	public function get_price_negotiations_started( $context = 'view' ) {
		return $this->get_prop( 'price_negotiations_started', $context );
	}

	/**
	 * Get the expiry date.
	 *
	 * @param  string $context View or edit context.
	 * @return WC_DateTime|NULL object if the date is set or null if there is no date.
	 */
	public function get_date_expiry( $context = 'view' ) {
		return $this->get_prop( 'date_expiry', $context );
	}

	/**
	 * Get the form fields.
	 *
	 * @param  string $context What the value is for. Valid values are 'view' and 'edit'.
	 * @return array
	 */
	public function get_form_fields( $context = 'view' ) {
		return $this->get_prop( 'form_fields', $context );
	}

	/*
	  |--------------------------------------------------------------------------
	  | Setters
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Set the order ID.
	 *
	 * @param string $value Value to set.
	 */
	public function set_order_id( $value ) {
		$this->set_prop( 'order_id', absint( $value ) );
	}

	/**
	 * Set is price negotiations started ?
	 *
	 * @param string $value Value to set.
	 */
	public function set_price_negotiations_started( $value ) {
		$this->set_prop( 'price_negotiations_started', $value );
	}

	/**
	 * Set date expiry.
	 *
	 * @param  string|integer|null $date UTC timestamp, or ISO 8601 DateTime. If the DateTime string has no timezone or offset, WordPress site timezone will be assumed. Null if there is no date.
	 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
	 */
	public function set_date_expiry( $date ) {
		$this->set_date_prop( 'date_expiry', $date );
	}

	/**
	 * Set the form fields.
	 *
	 * @param string $value Value of the prop.
	 * @throws Exception may be thrown if value is invalid.
	 */
	public function set_form_fields( $value ) {
		$this->set_prop( 'form_fields', is_array( $value ) ? $value : array()  );
	}

	/**
	 * Set quote request status.
	 * Here we can't call the WC_Order::set_status method since they are using 'wc-' prefix to their statuses and so we are doing like this.
	 *
	 * @param string $new_status    Status to change the quote request to. No internal QTS_PREFIX prefix is required.
	 * @param string $note          Optional note to add.
	 * @param bool   $manual_update Is this a manual quote request status change?.
	 * @param string $action_by     Is the status update by 'admin'|'customer'?. Default it is 'system'.
	 * @return array
	 */
	public function set_status( $new_status, $note = '', $manual_update = false, $action_by = 'system' ) {
		$old_status = $this->get_status();
		$new_status = QTS_PREFIX === substr( $new_status, 0, 5 ) ? substr( $new_status, 5 ) : $new_status;

		// If setting the status, ensure it's set to a valid status.
		if ( true === $this->object_read ) {
			// Only allow valid new status.
			if ( ! in_array( QTS_PREFIX . $new_status, $this->get_valid_statuses(), true ) && 'trash' !== $new_status ) {
				$new_status = 'new';
			}

			if ( $old_status ) {
				// If the old status is set but unknown (e.g. draft) assume its new for action usage.
				if ( ! in_array( QTS_PREFIX . $old_status, $this->get_valid_statuses(), true ) && 'trash' !== $old_status ) {
					$old_status = 'new';
				}

				if ( $old_status !== $new_status ) {
					$this->status_transition = array(
						'from'      => ! empty( $this->status_transition[ 'from' ] ) ? $this->status_transition[ 'from' ] : $old_status,
						'to'        => $new_status,
						'note'      => $note,
						'manual'    => ( bool ) $manual_update,
						'action_by' => $action_by,
					);

					if ( $manual_update ) {
						/**
						 * Triggers when the quote request status is manually updated.
						 * 
						 * @since 1.0
						 */
						do_action( 'qts_quote_request_edit_status', $this->get_id(), $new_status );
					}
				}
			}
		}

		$this->set_prop( 'status', $new_status );

		return array(
			'from' => $old_status,
			'to'   => $new_status,
		);
	}

	/**
	 * Schedule the date to expire the quote.
	 * 
	 * @param string|int $date
	 * @param string $hook_to_remind
	 */
	public function schedule_expiry( $date, $hook_to_remind ) {
		_qts_cancel_all_jobs_from_queue( 'quote_request', $this->get_id() );
		_qts_push_job_to_queue( 'quote_request', 'do_expire', $date, $this->get_id() );

		$days_to_remind = array_map( 'trim', explode( ',', get_option( QTS_PREFIX . $hook_to_remind ) ) );
		$days_to_remind = _qts_get_dates( _qts_get_time( 'timestamp' ), $date, $days_to_remind );

		if ( ! empty( $days_to_remind ) ) {
			foreach ( $days_to_remind as $count => $remind_date ) {
				if ( _qts_job_exists_in_queue( 'quote_request', $hook_to_remind, $remind_date, $this->get_id() ) ) {
					continue;
				}

				_qts_push_job_to_queue( 'quote_request', $hook_to_remind, $remind_date, $this->get_id(), array(
					'_reminder_count' => ( 1 + $count ),
					'_total_count'    => count( $days_to_remind ),
				) );
			}
		}
	}
}
