<?php

defined( 'ABSPATH' ) || exit;

/**
 * QTS Quote Totals calculation class.
 *
 * @class QTS_Quote_Totals
 * @package Classes
 */
final class QTS_Quote_Totals {

	/**
	 * Reference to quote object.
	 *
	 * @var QTS_Quote
	 */
	protected $quote;

	/**
	 * Line items to calculate.
	 *
	 * @var array
	 */
	protected $items = array();

	/**
	 * Items tax rates.
	 *
	 * @var array
	 */
	protected $item_tax_rates = array();

	/**
	 * Sets up the items provided, and calculate totals.
	 *
	 * @throws Exception If missing QTS_Quote object.
	 * @param QTS_Quote $quote Quote object to calculate totals for.
	 */
	public function __construct( &$quote = null ) {
		if ( ! is_a( $quote, 'QTS_Quote' ) ) {
			throw new Exception( 'A valid QTS_Quote object is required' );
		}

		$this->quote = $quote;
		$this->calculate_item_subtotals();
	}

	/**
	 * Get default blank set of props used per item.
	 *
	 * @return array
	 */
	protected function get_default_item_props() {
		return ( object ) array(
					'object'                     => null,
					'tax_class'                  => '',
					'taxable'                    => false,
					'quantity'                   => 0,
					'product'                    => false,
					'price_includes_tax'         => false,
					'original_price'             => 0,
					'requested_price'            => null,
					'requested_price_percent'    => null,
					'requested_discount_percent' => null,
					'subtotal'                   => 0,
		);
	}

	/**
	 * Handles a quote object passed in for calculation. 
	 */
	protected function get_items_from_quote() {
		$this->items = array();

		foreach ( $this->quote->get_quote() as $quote_item_key => $quote_item ) {
			$item                             = $this->get_default_item_props();
			$item->key                        = $quote_item_key;
			$item->object                     = $quote_item;
			$item->tax_class                  = $quote_item[ 'data' ]->get_tax_class();
			$item->taxable                    = 'taxable' === $quote_item[ 'data' ]->get_tax_status();
			$item->price_includes_tax         = wc_prices_include_tax();
			$item->quantity                   = $quote_item[ 'quantity' ];
			$item->original_price             = $quote_item[ 'original_price' ];
			$item->requested_price            = $quote_item[ 'requested_price' ];
			$item->requested_price_percent    = isset( $quote_item[ 'requested_price_percent' ] ) ? $quote_item[ 'requested_price_percent' ] : null;
			$item->requested_discount_percent = isset( $quote_item[ 'requested_discount_percent' ] ) ? $quote_item[ 'requested_discount_percent' ] : null;
			$item->preferred_price            = _qts_calculate_preferred_price( $item );
			$item->price                      = is_numeric( $item->preferred_price ) ? $item->preferred_price : $item->original_price;
			$item->total                      = wc_add_number_precision_deep( $item->price * $item->quantity );
			$item->product                    = $quote_item[ 'data' ];
			$item->tax_rates                  = $this->get_item_tax_rates( $item );
			$this->items[ $quote_item_key ]   = $item;
		}
	}

	/**
	 * Get tax rates for an item. Caches rates in class to avoid multiple look ups.
	 *
	 * @param  object $item Item to get tax rates for.
	 * @return array of taxes
	 */
	protected function get_item_tax_rates( $item ) {
		if ( ! wc_tax_enabled() ) {
			return array();
		}

		$tax_class                          = $item->product->get_tax_class();
		$item_tax_rates                     = isset( $this->item_tax_rates[ $tax_class ] ) ? $this->item_tax_rates[ $tax_class ] : $this->item_tax_rates[ $tax_class ] = WC_Tax::get_rates( $item->product->get_tax_class(), WC()->customer );
		return $item_tax_rates;
	}

	/**
	 * Apply rounding to item subtotal before summing.
	 *
	 * @param float $value Item subtotal value.
	 * @return float
	 */
	protected function round_item_subtotal( $value ) {
		return 'yes' !== get_option( 'woocommerce_tax_round_at_subtotal' ) ? round( $value ) : $value;
	}

	/**
	 * Returns array of values for totals calculation.
	 *
	 * @param string $field Field name.
	 * @return array Items object
	 */
	protected function get_values_for_total( $field ) {
		return array_values( wp_list_pluck( $this->items, $field ) );
	}

	/**
	 * Return rounded total based on settings.
	 *
	 * @param array $values Values to round. Should be with precision.
	 * @return float|int
	 */
	public function get_rounded_items_total( $values ) {
		return array_sum( array_map( array( $this, 'round_item_subtotal' ), $values ) );
	}

	/**
	 * Subtotals are costs.
	 */
	protected function calculate_item_subtotals() {
		$this->get_items_from_quote();

		foreach ( $this->items as $item_key => $item ) {
			$item->subtotal = $item->total;

			if ( $item->product->is_taxable() ) {
				if ( $item->price_includes_tax ) {
					$item->subtotal = wc_add_number_precision( wc_get_price_including_tax( $item->product, array( 'qty' => $item->quantity, 'price' => $item->price ) ) );
				} else {
					$item->subtotal = wc_add_number_precision( wc_get_price_excluding_tax( $item->product, array( 'qty' => $item->quantity, 'price' => $item->price ) ) );
				}
			}
			$this->quote->quote_contents[ $item_key ][ 'line_subtotal' ] = wc_remove_number_precision( $item->subtotal );
		}

		$this->quote->set_subtotal( wc_remove_number_precision( round( $this->get_rounded_items_total( $this->get_values_for_total( 'subtotal' ) ) ) ) );
	}
}
