<?php

defined( 'ABSPATH' ) || exit;

/**
 * Register new Payment Gateway id of Quote Request.
 * 
 * @class QTS_Quote_Request_Gateway
 * @package Class
 */
class QTS_Quote_Request_Gateway extends WC_Payment_Gateway {

	/**
	 * QTS_Quote_Request_Gateway constructor.
	 */
	public function __construct() {
		$this->id                 = QTS_PREFIX . 'quote_request';
		$this->method_title       = 'Quote Request' ;
		$this->method_description = 'Pay with Quote Request.';
		$this->has_fields         = true;
		$this->init_form_fields();
		$this->init_settings();
		$this->enabled            = $this->get_option( 'enabled' );
		$this->title              = $this->get_option( 'title' );
		$this->description        = $this->get_option( 'description' );
		$this->supports           = array( 'products', 'qtsquote_request' );

		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
	}

	/**
	 * Admin Settings
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled'     => array(
				'title'   => 'Enable/Disable',
				'type'    => 'checkbox',
				'label'   => ' ',
				'default' => 'no',
			),
			'title'       => array(
				'title'       => 'Title:',
				'type'        => 'text',
				'description' => 'This controls the title which the user see during checkout.',
				'default'     => 'Quote Request Payment Gateway',
			),
			'description' => array(
				'title'    => 'Description', 'quote-request-for-woocommerce',
				'type'     => 'textarea',
				'default'  => 'Use this payment gateway to submit your Quote Request.',
				'desc_tip' => true,
			),
				);
	}

	/**
	 * Check our gateway is available for use.
	 *
	 * @return bool
	 */
	public function is_available() {
		// Check cart class is loaded or abort.
		if ( is_null( WC()->cart ) ) {
			/**
			 * Is quote request payment gateway available by default?
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'qts_is_quote_request_payment_gateway_available_by_default', false );
		}

		return parent::is_available();
	}

	/**
	 * Process Payment.
	 * 
	 * @param int $order_id
	 * @return array
	 * @throws Exception
	 */
	public function process_payment( $order_id ) {

		try {
			$order = wc_get_order( $order_id );

			if ( ! $order ) {
				throw new Exception( __( 'Something went wrong !!', 'quote-request-for-woocommerce' ) );
			}

			$order->update_status( 'qts_new_quote' );

			// Remove cart
			WC()->cart->empty_cart();

			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order ),
					);
		} catch ( Exception $e ) {
			if ( ! empty( $e ) ) {
				wc_add_notice( $e->getMessage(), 'error' );
			}
		}

		// If we reached this point then there were errors
		return array(
			'result'   => 'failure',
			'redirect' => $this->get_return_url( $order ),
				);
	}
}

return new QTS_Quote_Request_Gateway();
