<?php
/**
 * Our Admin Functions
 */
defined( 'ABSPATH' ) || exit;

/**
 * Get our screen ids.
 *
 * @return array
 */
function _qts_get_screen_ids() {
	$wc_screen_id     = sanitize_title( __( 'WooCommerce', 'woocommerce' ) );
	$static_screen_id = 'woocommerce';
	$screen_ids       = array(
		'qts_quote_request',
		'qts_form_field',
		'qts_scheduled_jobs',
		'edit-qts_quote_request',
		'edit-qts_form_field',
		'edit-qts_scheduled_jobs',
		$wc_screen_id . '_page_qts_settings',
		$wc_screen_id . '_page_qts_quote_request',
		$wc_screen_id . '_page_wc-orders--qts_quote_request',
		$static_screen_id . '_page_qts_settings',
		$static_screen_id . '_page_qts_quote_request',
		$static_screen_id . '_page_wc-orders--qts_quote_request',
		'qts_settings',
	);

	/**
	 * Get admin screen ids.
	 * 
	 * @since 1.0
	 */
	return apply_filters( 'qts_screen_ids', $screen_ids );
}

/**
 * Check the screen against the which context.
 * 
 * @param string $screen_id
 * @param string $which
 * @return bool
 */
function _qts_is_screen( $screen_id, $which = 'any' ) {
	if ( in_array( $screen_id, _qts_get_screen_ids(), true ) ) {
		if ( is_array( $which ) ) {
			return in_array( $screen_id, $which );
		} else if ( 'any' !== $which ) {
			$wc_screen_id = sanitize_title( __( 'WooCommerce', 'woocommerce' ) );
			return ( $screen_id === $which || "{$wc_screen_id}_page_{$which}" === $screen_id );
		} else {
			return true;
		}
	}

	return false;
}

/**
 * Create a page and store the ID in an option.
 *
 * @param mixed  $slug Slug for the new page.
 * @param string $option Option name to store the page's ID.
 * @param string $page_title (default: '') Title for the new page.
 * @param string $page_content (default: '') Content for the new page.
 * @param int    $post_parent (default: 0) Parent for the new page.
 * @return int page ID.
 */
function _qts_create_page( $slug, $option = '', $page_title = '', $page_content = '', $post_parent = 0 ) {
	$option_value = '' !== $option ? get_option( $option ) : 0;

	if ( $option_value > 0 ) {
		$page_object = get_post( $option_value );

		if ( $page_object && 'page' === $page_object->post_type && ! in_array( $page_object->post_status, array( 'pending', 'trash', 'future', 'auto-draft' ), true ) ) {
			// Valid page is already in place.
			return $page_object->ID;
		}
	}

	$page_data = array(
		'post_status'    => 'publish',
		'post_type'      => 'page',
		'post_author'    => 1,
		'post_name'      => $slug,
		'post_title'     => $page_title,
		'post_content'   => $page_content,
		'post_parent'    => $post_parent,
		'comment_status' => 'closed',
	);

	$page_id = wp_insert_post( $page_data );

	if ( $option ) {
		update_option( $option, $page_id );
	}

	return $page_id;
}

/**
 * Get WC search field
 * 
 * @param array $args
 * @param bool $echo
 * @return string echo search field
 */
function _qts_search_field( $args = array(), $echo = true ) {

	$args = wp_parse_args( $args, array(
		'class'       => '',
		'id'          => '',
		'name'        => '',
		'type'        => '',
		'action'      => '',
		'placeholder' => '',
		'css'         => 'width: 50%;',
		'multiple'    => true,
		'allow_clear' => true,
		'selected'    => true,
		'options'     => array(),
			) );

	ob_start();
	?>
	<select 
		id="<?php echo esc_attr( $args[ 'id' ] ); ?>" 
		class="<?php echo esc_attr( $args[ 'class' ] ); ?>" 
		name="<?php echo esc_attr( '' !== $args[ 'name' ] ? $args[ 'name' ] : $args[ 'id' ]  ); ?><?php echo ( $args[ 'multiple' ] ) ? '[]' : ''; ?>" 
		data-action="<?php echo esc_attr( $args[ 'action' ] ); ?>" 
		data-placeholder="<?php echo esc_attr( $args[ 'placeholder' ] ); ?>" 
		<?php echo ( $args[ 'allow_clear' ] ) ? 'data-allow_clear="true"' : ''; ?> 
		<?php echo ( $args[ 'multiple' ] ) ? 'multiple="multiple"' : ''; ?> 
		style="<?php echo esc_attr( $args[ 'css' ] ); ?>">
			<?php
			if ( ! is_array( $args[ 'options' ] ) ) {
				$args[ 'options' ] = ( array ) $args[ 'options' ];
			}

			$args[ 'options' ] = array_filter( $args[ 'options' ] );

			foreach ( $args[ 'options' ] as $id ) {
				$option_value = '';

				switch ( $args[ 'type' ] ) {
					case 'product':
						$product = wc_get_product( $id );
						if ( $product ) {
							$option_value = wp_kses_post( $product->get_formatted_name() );
						}
						break;
					case 'customer':
						$user = get_user_by( 'id', $id );
						if ( $user ) {
							$option_value = ( esc_html( $user->display_name ) . '(#' . absint( $user->ID ) . ' &ndash; ' . esc_html( $user->user_email ) . ')' );
						}
						break;
					default:
						$post = get_post( $id );
						if ( $post ) {
							$option_value = sprintf( '(#%s) %s', $post->ID, wp_kses_post( $post->post_title ) );
						}
						break;
				}

				if ( $option_value ) {
					?>
				<option value="<?php echo esc_attr( $id ); ?>" <?php echo ( $args[ 'selected' ] ) ? 'selected="selected"' : ''; ?>><?php echo esc_html( $option_value ); ?></option>
					<?php
				}
			}
			?>
	</select>
	<?php
	if ( $echo ) {
		ob_end_flush();
	} else {
		return ob_get_clean();
	}
}

/**
 * Return the array of categories for the products.
 *
 * @since 3.1.0
 * @return array
 */
function _qts_get_product_terms( $taxonomy = 'product_cat' ) {
	$terms     = array();
	$terms_id  = array();
	$term_name = array();

	$listofterms = get_terms( $taxonomy );
	if ( is_array( $listofterms ) ) {
		foreach ( $listofterms as $term ) {
			$term_name[] = $term->name;
			$terms_id[]  = $term->term_id;
		}
	}

	if ( $terms_id && $term_name ) {
		$terms = array_combine( (array) $terms_id, (array) $term_name );
	}

	return $terms;
}
