<?php

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

defined( 'ABSPATH' ) || exit;

/**
 * QTS_Quote_Request_Gateway_Blocks_Support class.
 *
 * @extends AbstractPaymentMethodType
 */
final class QTS_Quote_Request_Gateway_Blocks_Support extends AbstractPaymentMethodType {

	/**
	 * Payment method name/id/slug (matches id in QTS_Quote_Request_Gateway in core).
	 *
	 * @var string
	 */
	protected $name = '_qts_quote_request';

	/**
	 * Gets the gateway.
	 * 
	 * @var QTS_Quote_Request_Gateway
	 */
	private $gateway;

	/**
	 * Initializes the payment method type.
	 */
	public function initialize() {
		$this->settings = get_option( "woocommerce_{$this->name}_settings", array() );

		if ( ! $this->gateway ) {
			$gateways = WC()->payment_gateways->get_available_payment_gateways();

			if ( isset( $gateways[ $this->name ] ) ) {
				$this->gateway = $gateways[ $this->name ];
			}
		}
	}

	/**
	 * Returns if this payment method should be active. If false, the scripts will not be enqueued.
	 *
	 * @return boolean
	 */
	public function is_active() {
		return filter_var( $this->get_setting( 'enabled', false ), FILTER_VALIDATE_BOOLEAN );
	}

	/**
	 * Returns an array of scripts/handles to be registered for this payment method.
	 *
	 * @return array
	 */
	public function get_payment_method_script_handles() {
		$script_path = 'blocks/frontend/index.js';
		$style_path  = 'blocks/frontend/index.css';

		$script_url = QTS_URL . "/assets/{$script_path}";
		$style_url  = QTS_URL . "/assets/{$style_path}";

		$script_asset_path = QTS_DIR . 'assets/blocks/frontend/index.asset.php';
		$style_asset_path  = QTS_DIR . 'assets/blocks/frontend/index.css';

		$script_asset = file_exists( $script_asset_path ) ? require $script_asset_path : array(
			'dependencies' => array(),
			'version'      => QTS_VERSION,
		);

		wp_enqueue_style(
				'qts-blocks-quote-request-gateway-integration',
				$style_url,
				array(),
				QTS_VERSION
		);

		wp_register_script(
				'qts-blocks-quote-request-gateway-integration',
				$script_url,
				$script_asset[ 'dependencies' ],
				$script_asset[ 'version' ],
				true
		);

		wp_set_script_translations(
				'qts-blocks-quote-request-gateway-integration',
				'quote-request-for-woocommerce'
		);

		return array( 'qts-blocks-quote-request-gateway-integration' );
	}

	/**
	 * Returns an array of key=>value pairs of data made available to the payment methods script.
	 *
	 * @return array
	 */
	public function get_payment_method_data() {
		return array(
			'title'        => $this->get_setting( 'title' ),
			'description'  => $this->gateway ? $this->gateway->get_description() : $this->get_setting( 'description' ),
			'is_available' => $this->gateway ? $this->gateway->is_available() : false,
			'supports'     => $this->get_supported_features(),
		);
	}

	/**
	 * Returns an array of supported features.
	 *
	 * @return string[]
	 */
	public function get_supported_features() {
		$supports = array();

		if ( $this->gateway ) {
			$supports = array_filter( $this->gateway->supports, array( $this->gateway, 'supports' ) );
		}

		return $supports;
	}
}
