<?php

/**
 * Our Templates
 *
 * Functions for the templating system.
 */
defined( 'ABSPATH' ) || exit;

/**
 * Show notice if quote is empty.
 */
function _qts_empty_quote_message() {
	echo '<p class="qts-quote-empty woocommerce-info">' . wp_kses_post( __( 'Your quote is currently empty.', 'quote-request-for-woocommerce' ) ) . '</p>';
}

/**
 * Output the quote request form fields template.
 */
function _qts_quote_request_form() {
	QTS_Form_Fields::enqueues();

	_qts_get_template( 'quote/quote-request-form.php', array(
		'fields' => QTS_Form_Fields::get_available_fields(),
	) );
}

/**
 * Output the proceed to quote button template.
 */
function _qts_proceed_to_quote_button() {
	if ( ! QTS_Cart::asking_for_quote() ) {
		return;
	}

	remove_action( 'woocommerce_proceed_to_checkout', 'woocommerce_button_proceed_to_checkout', 20 );

	_qts_get_template( 'cart/proceed-to-quote-button.php' );
}

/**
 * Remove the quote empty notice when the quote request is created via Mode 1.
 * 
 * @param QTS_Quote_Request $quote_request
 */
function _qts_remove_empty_quote_message( $quote_request ) {
	if ( $quote_request->is_created_via( 'quote' ) ) {
		remove_action( 'qts_quote_is_empty', '_qts_empty_quote_message' );
	}
}

/**
 * My Account > Quote Requests template.
 */
function _qts_my_quote_requests( $current_page ) {
	$current_page = empty( $current_page ) ? 1 : absint( $current_page );

	/**
	 * Get my account quote requests query.
	 * 
	 * @since 1.0
	 */
	$customer_quote_requests = wc_get_orders( apply_filters( 'woocommerce_my_account_my_qtsquote_requests_query', array(
		'customer' => get_current_user_id(),
		'type'     => 'qts_quote_request',
		'status'   => array_keys( _qts_get_quote_request_statuses() ),
		'page'     => $current_page,
		'paginate' => true,
		'limit'    => 5,
			) ) );

	_qts_get_template( 'myaccount/my-quote-requests.php', array(
		'current_page'            => absint( $current_page ),
		'customer_quote_requests' => $customer_quote_requests,
		'has_quote_requests'      => 0 < $customer_quote_requests->total,
	) );
}

/**
 * My Account > Quote Requests > View Quote Request template.
 * 
 * @param int $quote_request_id
 */
function _qts_view_quote_request( $quote_request_id ) {
	$quote_request = _qts_get_quote_request( $quote_request_id );

	if ( ! $quote_request || ! current_user_can( 'view_qtsquote_request', $quote_request_id ) ) {
		echo '<div class="woocommerce-error">' . esc_html__( 'Invalid quote request.', 'quote-request-for-woocommerce' ) . ' <a href="' . esc_url( wc_get_page_permalink( 'myaccount' ) ) . '" class="wc-forward">' . esc_html__( 'My account', 'quote-request-for-woocommerce' ) . '</a></div>';
		return;
	}

	echo '<div class="qts-view-quote-request">';
	_qts_get_template( 'myaccount/view-quote-request.php', array(
		'quote_request'    => $quote_request,
		'quote_request_id' => $quote_request_id,
	) );
	echo '</div>';
}

/**
 * Displays quote request details in a table.
 *
 * @param mixed $quote_request_id
 */
function _qts_quote_request_details_table( $quote_request_id ) {
	$quote_request = _qts_get_quote_request( $quote_request_id );

	if ( ! $quote_request ) {
		return;
	}

	_qts_get_template( 'quote-request/quote-request-details.php', array(
		'quote_request'    => $quote_request,
		'quote_request_id' => $quote_request_id,
			)
	);
}

/**
 * Return a list of allowed tags and attributes for the WC qty input.
 * 
 * @return array
 */
function _qts_kses_wc_qty_input_allowed_html() {
	$allowed_html = array(
		'div'   => array(
			'class' => array(),
		),
		'label' => array(
			'class' => array(),
			'for'   => array(),
		),
		'input' => array(
			'type'      => array(),
			'id'        => array(),
			'class'     => array(),
			'step'      => array(),
			'min'       => array(),
			'max'       => array(),
			'name'      => array(),
			'value'     => array(),
			'title'     => array(),
			'size'      => array(),
			'inputmode' => array(),
		),
	);
	return $allowed_html;
}

/**
 * Return a Quote Basket.
 * 
 * @since 2.7.0
 * @param string $items Menu items.
 * @param array $args Arguments.
 * @return HTML
 */
function _qts_quote_basket( $items, $args ) {
	$menu_ids = maybe_unserialize( get_option( QTS_PREFIX . 'get_quote_basket_menu' ) );
	if ( empty( $menu_ids ) ) {
		return $items;
	}

	$menu_id = 0;
	if ( isset( $args->menu->term_id ) ) {
		$menu_id = $args->menu->term_id;
	} elseif ( isset( $args->term_id ) ) {
		$menu_id = $args->term_id;
	} elseif ( isset( $args->menu ) ) {
		$menu_id = $args->menu;
	}

	$menu_match = in_array( ( string ) $menu_id, ( array ) $menu_ids, true ) ? true : false;
	if ( ! $menu_match ) {
		return $items;
	}

	ob_start();
	_qts_mini_quote();
	$html = ob_get_clean();
	return $items . $html;
}

/**
 * Mini Quote.
 * 
 * @since 2.7.0
 * @param array $args Arguments.
 * @return HTML
 */
function _qts_mini_quote( $args = array() ) {
	$args = wp_parse_args( $args, array(
		'list_class' => '',
			) );

	_qts_get_template( 'quote/mini-quote.php', $args );
}

/**
 * Mini Quote Subtotal.
 * 
 * @since 2.7.0
 * @param array $args Arguments.
 * @return HTML
 */
function _qts_widget_quote_subtotal() {
	echo '<strong>' . esc_html__( 'Subtotal:', 'quote-request-for-woocommerce' ) . '</strong> ' . wp_kses_post( _qts()->quote->get_subtotal() );
}

/**
 * Mini Quote Subtotal button.
 * 
 * @since 2.7.0
 * @param Array $args Arguments.
 * @return HTML
 */
function _qts_widget_quote_button() {
	echo '<a href="' . esc_url( _qts_get_quote_url() ) . '" class="button wc-forward">' . esc_html__( 'View Quote', 'quote-request-for-woocommerce' ) . '</a>';
}

/**
 * Convert to quote mode selection.
 * 
 * @since 3.0.0
 * @return HTML
 */
function _qts_convert_to_quote_mode_selection() {
	if ( 'user-choice' === get_option( QTS_PREFIX . 'quote_request_mode' ) ) {
		_qts_get_template( 'convert-to-quote/quote-mode-selection.php' );
	}
}
