<?php

defined( 'ABSPATH' ) || exit ;

/**
 * Manage order activities for the quote request.
 * 
 * @class QTS_Order_Manager
 * @package Class
 */
class QTS_Order_Manager {

	/**
	 * Data stored in meta keys, prefixed with QTS_PREFIX
	 *
	 * @var array
	 */
	protected static $internal_meta_keys = array(
		'quote_request_id',
		'is_quote_request_order',
		'quote_request_completed',
			) ;

	/**
	 * Init QTS_Order_Manager.
	 */
	public static function init() {
		// Mode 1&2
		add_action( 'woocommerce_order_status_on_hold', __CLASS__ . '::payment_pending', 99 ) ;
		add_action( 'woocommerce_order_status_completed', __CLASS__ . '::complete_the_quote_request', 99 ) ;
		add_action( 'woocommerce_order_status_processing', __CLASS__ . '::complete_the_quote_request', 99 ) ;

		// Mode 1
		add_action( 'woocommerce_checkout_order_processed', __CLASS__ . '::save_quote_request_to_order', 5 ) ;

		// Mode 2
		add_filter( 'woocommerce_cancel_unpaid_order', __CLASS__ . '::prevent_unpaid_order_from_cancel', 99, 2 ) ;
		add_action( 'qts_quote_request_status_accepted', __CLASS__ . '::prepare_to_pay_for_order', -1 ) ;
		add_action( 'qts_quote_request_status_approved', __CLASS__ . '::prepare_to_pay_for_order', -1 ) ;
		add_action( 'qts_quote_request_status_rejected', __CLASS__ . '::quote_request_failed', -1 ) ;
		add_action( 'qts_quote_request_status_expired', __CLASS__ . '::quote_request_failed', -1 ) ;
		add_action( 'woocommerce_thankyou_' . QTS_PREFIX . 'quote_request', __CLASS__ . '::quote_request_submitted_success' ) ;
	}

	/**
	 * Get the meta data for the order.
	 * 
	 * @param int $order_id
	 * @param array $internal_meta_key
	 * @return mixed
	 */
	public static function get_meta( $order_id, $internal_meta_key ) {
		if ( ! in_array( $internal_meta_key, self::$internal_meta_keys ) ) {
			return null ;
		}

		return get_post_meta( $order_id, QTS_PREFIX . $internal_meta_key, true ) ;
	}

	/**
	 * Save the meta data to the order.
	 * 
	 * @param int $order_id
	 * @param array $internal_meta_key
	 * @param mixed $value
	 */
	public static function save_meta( $order_id, $internal_meta_key, $value ) {
		if ( ! in_array( $internal_meta_key, self::$internal_meta_keys ) ) {
			return null ;
		}

		update_post_meta( $order_id, QTS_PREFIX . $internal_meta_key, $value ) ;
	}

	/*
	  |--------------------------------------------------------------------------
	  | Mode 1&2 Combined Funtions
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Notice the admin saying quote request payment is on hold.
	 * 
	 * @param int $order_id
	 */
	public static function payment_pending( $order_id ) {
		$order = wc_get_order( $order_id ) ;

		if ( ! $order ) {
			return ;
		}

		if ( 'yes' === self::get_meta( $order_id, 'quote_request_completed' ) ) {
			return ;
		}

		$quote_request_id = self::get_meta( $order_id, 'quote_request_id' ) ;
		$quote_request    = _qts_get_quote_request( $quote_request_id ) ;

		if ( ! $quote_request ) {
			return ;
		}

		if ( $quote_request->needs_payment() ) {
			/* translators: %s: order id */
			$quote_request->add_order_note( sprintf( __( 'Payment is on hold for the order #%1$s.', 'quote-request-for-woocommerce' ), $order_id ) ) ;
			do_action( 'qts_quote_request_waiting_for_payment', $quote_request, $order_id ) ;
		}
	}

	/**
	 * Complete the quote request only if the order status changed to completed or processing.
	 * 
	 * @param int $order_id
	 */
	public static function complete_the_quote_request( $order_id ) {
		$order = wc_get_order( $order_id ) ;

		if ( ! $order ) {
			return ;
		}

		if ( 'yes' === self::get_meta( $order_id, 'quote_request_completed' ) ) {
			return ;
		}

		$quote_request_id = self::get_meta( $order_id, 'quote_request_id' ) ;
		$quote_request    = _qts_get_quote_request( $quote_request_id ) ;

		if ( ! $quote_request ) {
			return ;
		}

		// Complete the quote request.
		$quote_request->payment_complete() ;
		self::save_meta( $order_id, 'quote_request_completed', 'yes' ) ;
		do_action( 'qts_quote_request_completed', $quote_request, $order_id ) ;
	}

	/*
	  |--------------------------------------------------------------------------
	  | Mode 1 Funtions
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Save the quote request to order meta.
	 * 
	 * @param int $order_id
	 */
	public static function save_quote_request_to_order( $order_id ) {
		if ( is_null( WC()->cart ) ) {
			return ;
		}

		$added_quote = QTS_Cart::awaiting_for_payment() ;

		if ( ! $added_quote || ! is_numeric( $added_quote[ 'payment' ] ) ) {
			return ;
		}

		$quote_request_id = absint( $added_quote[ 'payment' ] ) ;
		$quote_request    = _qts_get_quote_request( $quote_request_id ) ;

		if ( $quote_request && $quote_request->is_created_via( 'quote' ) ) {
			$quote_request->set_order_id( $order_id ) ;
			$quote_request->save() ;

			// Save the quote request in order meta.
			self::save_meta( $order_id, 'is_quote_request_order', 'yes' ) ;
			self::save_meta( $order_id, 'quote_request_id', $quote_request_id ) ;
		}
	}

	/*
	  |--------------------------------------------------------------------------
	  | Mode 2 Funtions
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Prevent the pending order from cancel which is created for quote.
	 * 
	 * @param bool $cancel_order
	 * @param WC_Order $order
	 * @return bool
	 */
	public static function prevent_unpaid_order_from_cancel( $cancel_order, $order ) {
		if ( 'yes' === self::get_meta( $order->get_id(), 'is_quote_request_order' ) ) {
			$cancel_order = false ;
		}

		return $cancel_order ;
	}

	/**
	 * Prepare the order to pay when the quote request accepted/approved for mode 2.
	 * 
	 * @param int $quote_request_id
	 */
	public static function prepare_to_pay_for_order( $quote_request_id ) {
		$quote_request = _qts_get_quote_request( $quote_request_id ) ;

		if ( ! $quote_request ) {
			return ;
		}

		if ( $quote_request->is_created_via( 'checkout' ) ) {
			$order = wc_get_order( $quote_request->get_order_id() ) ;

			//Update the order status from New Quote to Pending Payment.
			if ( $order && $order->has_status( 'qts_new_quote' ) ) {
				$quote_request_items = $quote_request->get_items( 'line_item' ) ;
				$order_items         = $order->get_items( 'line_item' ) ;

				// Update the quote request item latest offer price and quantity in the order.
				if ( $quote_request_items && $order_items ) {
					foreach ( $quote_request_items as $quote_request_item ) {
						$quote_request_product_id = $quote_request_item->get_variation_id() ? $quote_request_item->get_variation_id() : $quote_request_item->get_product_id() ;

						foreach ( $order_items as $order_item ) {
							$order_product_id = $order_item->get_variation_id() ? $order_item->get_variation_id() : $order_item->get_product_id() ;

							if ( $quote_request_product_id === $order_product_id ) {
								$order_item->set_quantity( $quote_request_item->get_quantity() ) ;
								$order_item->set_total( $quote_request_item->get_total() ) ;
							}
						}
					}
				}

				// Calculate the order total.
				$order->calculate_totals() ;
				/* translators: %s: quote request status */
				$order->update_status( 'pending', sprintf( __( 'Quote Request status set to %s. By paying this order quote request gets completed.', 'quote-request-for-woocommerce' ), _qts_get_quote_request_status_name( $quote_request->get_status() ) ) ) ;
			}
		}
	}

	/**
	 * Change the order status to qts_quote_failed when the quote request rejected/expired for mode 2.
	 * 
	 * @param int $quote_request_id
	 */
	public static function quote_request_failed( $quote_request_id ) {
		$quote_request = _qts_get_quote_request( $quote_request_id ) ;

		if ( ! $quote_request ) {
			return ;
		}

		if ( $quote_request->is_created_via( 'checkout' ) ) {
			$order = wc_get_order( $quote_request->get_order_id() ) ;

			//Update the order status from New Quote to Quote Failed.
			if ( $order && $order->has_status( 'qts_new_quote' ) ) {
				/* translators: %s: quote request status */
				$order->update_status( 'qts_quote_failed', sprintf( __( 'Quote Request status set to %s.', 'quote-request-for-woocommerce' ), _qts_get_quote_request_status_name( $quote_request->get_status() ) ) ) ;
			}
		}
	}

	/**
	 * 
	 * Print the quote request submitted notice.
	 *
	 * @param int $order_id
	 */
	public static function quote_request_submitted_success( $order_id ) {
		if ( 'yes' !== self::get_meta( $order_id, 'is_quote_request_order' ) ) {
			return ;
		}

		$quote_request_id = self::get_meta( $order_id, 'quote_request_id' ) ;
		$quote_request    = _qts_get_quote_request( $quote_request_id ) ;

		if ( $quote_request->get_user_id() ) {
			wc_print_notice( trim( str_replace( array( '[quote_request_number]', '[view_quote_request_url]' ), array( esc_html( $quote_request->get_id() ), esc_url( $quote_request->get_view_order_url() ) ), get_option( QTS_PREFIX . 'quote_submitted_success_msg_for_logged_in_users' ) ) ), 'success' ) ;
		} else {
			wc_print_notice( trim( str_replace( '[quote_request_number]', esc_html( $quote_request->get_id() ), get_option( QTS_PREFIX . 'quote_submitted_success_msg_for_guests' ) ) ), 'success' ) ;
		}
	}

}

QTS_Order_Manager::init() ;
