<?php

defined( 'ABSPATH' ) || exit ;

/**
 * QTS Quote session handling class.
 *
 * @class QTS_Quote_Session
 * @package Classes
 */
final class QTS_Quote_Session {

	/**
	 * Reference to Quote object.
	 *
	 * @var QTS_Quote
	 */
	protected $quote ;

	/**
	 * Sets up the items provided, and calculate totals.
	 *
	 * @throws Exception If missing QTS_Quote object.
	 * @param QTS_Quote $quote Quote object to calculate totals for.
	 */
	public function __construct( &$quote ) {
		if ( ! is_a( $quote, 'QTS_Quote' ) ) {
			throw new Exception( 'A valid QTS_Quote object is required' ) ;
		}

		$this->quote = $quote ;
	}

	/**
	 * Init quote session.
	 */
	public function init() {
		// Populate quote from session
		add_action( 'wp_loaded', array( $this, 'get_quote_from_session' ) ) ;

		// Set session
		add_action( 'qts_after_calculate_totals', array( $this, 'set_session' ) ) ;
		add_action( 'qts_quote_loaded_from_session', array( $this, 'set_session' ) ) ;

		// Destroy session
		add_action( 'qts_quote_emptied', array( $this, 'destroy_quote_session' ) ) ;

		// Set our cookie on demand.
		add_action( 'qts_add_to_quote', array( $this, 'maybe_set_quote_cookies' ) ) ;
		add_action( 'wp', array( $this, 'maybe_set_quote_cookies' ), 99 ) ;
		add_action( 'shutdown', array( $this, 'maybe_set_quote_cookies' ), 0 ) ;
	}

	/**
	 * Returns the contents of the quote in an array without the 'data' element.
	 *
	 * @return array contents of the quote
	 */
	public function get_quote_for_session() {
		$quote_session = array() ;
		foreach ( $this->quote->get_quote() as $key => $values ) {
			$quote_session[ $key ] = $values ;
			unset( $quote_session[ $key ][ 'data' ] ) ;
		}

		return $quote_session ;
	}

	/**
	 * Get the quote data from the PHP session and store it in class variables.
	 */
	public function get_quote_from_session() {
		do_action( 'qts_load_quote_from_session' ) ;

		$this->quote->set_totals( WC()->session->get( 'qts_quote_totals', null ) ) ;
		$this->quote->set_removed_quote_contents( WC()->session->get( 'qts_removed_quote_contents', array() ) ) ;

		$update_quote_session = false ;
		$quote                = WC()->session->get( 'qts_quote', null ) ;

		if ( is_null( $quote ) ) {
			$quote                = array() ;
			$update_quote_session = true ;
		}

		$quote_contents = array() ;
		foreach ( $quote as $key => $values ) {
			$product = wc_get_product( $values[ 'variation_id' ] ? $values[ 'variation_id' ] : $values[ 'product_id' ] ) ;

			if ( empty( $product ) || ! $product->exists() || 0 >= $values[ 'quantity' ] || ! QTS_Add_To_Quote::is_quote_enabled( $product ) ) {
				continue ;
			}

			if ( ! $product->is_purchasable() ) {
				$update_quote_session = true ;

				wc_add_notice( trim( get_option( QTS_PREFIX . 'product_not_purchasable_err' ) ), 'error' ) ;
			} else {
				// Check whether the requested price less than the minimum price to ask for quote.
				$min_quote_price = _qts_get_min_quote_price( $product ) ;
				if ( $min_quote_price > 0 && ! is_null( $values[ 'requested_price' ] ) && floatval( $values[ 'requested_price' ] ) < $min_quote_price ) {
					$update_quote_session        = true ;
					$values[ 'requested_price' ] = null ;

					wc_add_notice( trim( str_replace( array( '[product_name]', '[min_quote_price]' ), array( $product->get_name(), wc_price( $min_quote_price ) ), get_option( QTS_PREFIX . 'cannot_request_less_than_min_price_err' ) ) ), 'error' ) ;
				}

				$quote_contents[ $key ] = array_merge( $values, array( 'data' => $product ) ) ;
			}
		}

		if ( ! empty( $quote_contents ) ) {
			$this->quote->set_quote_contents( $quote_contents ) ;
		}

		do_action( 'qts_quote_loaded_from_session', $this->quote ) ;

		if ( $update_quote_session || is_null( WC()->session->get( 'qts_quote_totals', null ) ) ) {
			WC()->session->set( 'qts_quote', $this->get_quote_for_session() ) ;
			$this->quote->calculate_totals() ;
		}
	}

	/**
	 * Sets the php session data for the quote.
	 */
	public function set_session() {
		WC()->session->set( 'qts_quote', $this->get_quote_for_session() ) ;
		WC()->session->set( 'qts_quote_totals', $this->quote->get_totals() ) ;
		WC()->session->set( 'qts_removed_quote_contents', $this->quote->get_removed_quote_contents() ) ;
	}

	/**
	 * Destroy quote session data.
	 */
	public function destroy_quote_session() {
		WC()->session->set( 'qts_quote', null ) ;
		WC()->session->set( 'qts_quote_totals', null ) ;
		WC()->session->set( 'qts_removed_quote_contents', null ) ;
	}

	/**
	 * Set quote cookies on demand.
	 */
	public function maybe_set_quote_cookies() {
		if ( ! headers_sent() && did_action( 'wp_loaded' ) && did_action( 'woocommerce_init' ) ) {
			if ( ! $this->quote->is_empty() ) {
				WC()->session->set_customer_session_cookie( true ) ;
			}
		}
	}

}
