<?php

defined( 'ABSPATH' ) || exit ;

/**
 * Post Types Admin
 * 
 * @class QTS_Admin_Post_Types
 * @package Class
 */
class QTS_Admin_Post_Types {

	/**
	 * Get our post types.
	 * 
	 * @var array 
	 */
	protected static $custom_post_types = array(
		'qts_quote_request'  => 'quote_request',
		'qts_form_field'     => 'form_field',
		'qts_scheduled_jobs' => 'scheduled_jobs',
			) ;

	/**
	 * Init QTS_Admin_Post_Types.
	 */
	public static function init() {

		add_action( 'admin_notices', __CLASS__ . '::output_notices', 99998 ) ;

		foreach ( self::$custom_post_types as $post_type => $meant_for ) {
			add_filter( "manage_{$post_type}_posts_columns", __CLASS__ . "::define_{$meant_for}_columns" ) ;
			add_filter( "manage_edit-{$post_type}_sortable_columns", __CLASS__ . '::define_sortable_columns' ) ;
			add_filter( "bulk_actions-edit-{$post_type}", __CLASS__ . '::define_bulk_actions' ) ;
			add_filter( "handle_bulk_actions-edit-{$post_type}", __CLASS__ . '::handle_bulk_actions', 10, 3 ) ;
			add_action( "manage_{$post_type}_posts_custom_column", __CLASS__ . "::render_{$meant_for}_columns", 10, 2 ) ;
		}

		add_filter( 'manage_shop_order_posts_columns', __CLASS__ . '::define_shop_order_columns', 11 ) ;
		add_action( 'manage_shop_order_posts_custom_column', __CLASS__ . '::render_shop_order_columns', 11, 2 ) ;

		add_filter( 'post_updated_messages', __CLASS__ . '::post_updated_messages' ) ;
		add_filter( 'enter_title_here', __CLASS__ . '::enter_title_here', 1, 2 ) ;
		add_filter( 'post_row_actions', __CLASS__ . '::row_actions', 99, 2 ) ;
		add_filter( 'request', __CLASS__ . '::request_query' ) ;
		add_filter( 'pre_delete_post', __CLASS__ . '::need_post_deletion', 999, 2 ) ;
		add_filter( 'pre_trash_post', __CLASS__ . '::need_post_deletion', 999, 2 ) ;
		add_action( 'admin_init', __CLASS__ . '::admin_action' ) ;
		add_filter( 'get_search_query', __CLASS__ . '::search_label' ) ;
		add_filter( 'query_vars', __CLASS__ . '::add_custom_query_var' ) ;
		add_action( 'parse_query', __CLASS__ . '::search_custom_fields' ) ;
	}

	/**
	 * Render our admin notices.
	 */
	public static function output_notices() {
		global $post_type, $pagenow ;

		switch ( $post_type ) {
			case 'qts_scheduled_jobs':
				if ( isset( $_GET[ 'qts-notice' ] ) ) {
					switch ( $_GET[ 'qts-notice' ] ) {
						case 'job-ran-success':
							if ( isset( $_GET[ 'job-name' ] ) ) {
								echo '<div id="message" class="updated is-dismissible"><p>' . esc_html__( 'Successfully executed: ', 'quote-request-for-woocommerce' ) . '<code>' . esc_html( sanitize_text_field( wp_unslash( $_GET[ 'job-name' ] ) ) ) . '</code></p></div>' ;
							}
							break ;
						case 'job-ran-failed':
							if ( isset( $_GET[ 'job-name' ] ) ) {
								echo '<div id="message" class="error is-dismissible"><p>' . esc_html__( 'Failed to execute: ', 'quote-request-for-woocommerce' ) . '<code>' . esc_html( sanitize_text_field( wp_unslash( $_GET[ 'job-name' ] ) ) ) . '</code></p></div>' ;
							}
							break ;
					}
				}
				break ;
		}

		// Bail out if not on form field list page.
		if ( 'edit.php' !== $pagenow || 'qts_form_field' !== $post_type || ! isset( $_REQUEST[ 'bulk_action' ] ) ) {
			return ;
		}

		$number  = isset( $_REQUEST[ 'changed' ] ) ? absint( $_REQUEST[ 'changed' ] ) : 0 ;
		$action  = wc_clean( wp_unslash( $_REQUEST[ 'bulk_action' ] ) ) ;
		$message = '' ;

		// Check if any post changes happened.
		switch ( $action ) {
			case QTS_PREFIX . 'enabled':
			case QTS_PREFIX . 'disabled':
				/* translators: %d: posts count %s: to post status */
				$message = sprintf( _n( '%1$d field status is changed to %2$s.', '%1$d field statuses are changed to %2$s.', $number, 'quote-request-for-woocommerce' ), number_format_i18n( $number ), _qts_get_form_field_status_name( $action ) ) ;
				break ;
			case 'field_required':
				/* translators: %d: posts count */
				$message = sprintf( _n( '%d field is set to mandatory.', '%d fields are set to mandatory.', $number, 'quote-request-for-woocommerce' ), number_format_i18n( $number ) ) ;
				break ;
			case 'field_optional':
				/* translators: %d: posts count */
				$message = sprintf( _n( '%d field is set to optional.', '%d fields are set to optional.', $number, 'quote-request-for-woocommerce' ), number_format_i18n( $number ) ) ;
				break ;
		}

		if ( ! empty( $message ) ) {
			echo '<div class="updated"><p>' . esc_html( $message ) . '</p></div>' ;
		}
	}

	/**
	 * Define which quote request columns to show on this screen.
	 *
	 * @param array $columns Existing columns.
	 * @return array
	 */
	public static function define_quote_request_columns( $columns ) {
		$columns = array(
			'cb'             => $columns[ 'cb' ],
			'quote_id'       => __( 'Quote ID', 'quote-request-for-woocommerce' ),
			'status'         => __( 'Status', 'quote-request-for-woocommerce' ),
			'requested_date' => __( 'Requested Date', 'quote-request-for-woocommerce' ),
			'mode'           => __( 'Mode', 'quote-request-for-woocommerce' ),
			'submitted_by'   => __( 'Submitted by', 'quote-request-for-woocommerce' ),
			'order_id'       => __( 'Order ID', 'quote-request-for-woocommerce' ),
			'more_info'      => __( 'More Info', 'quote-request-for-woocommerce' ),
				) ;
		return $columns ;
	}

	/**
	 * Define which form field columns to show on this screen.
	 *
	 * @param array $columns Existing columns.
	 * @return array
	 */
	public static function define_form_field_columns( $columns ) {
		$columns = array(
			'cb'       => $columns[ 'cb' ],
			'label'    => __( 'Label', 'quote-request-for-woocommerce' ),
			'type'     => __( 'Type', 'quote-request-for-woocommerce' ),
			'status'   => __( 'Status', 'quote-request-for-woocommerce' ),
			'required' => __( 'Required', 'quote-request-for-woocommerce' ),
				) ;
		return $columns ;
	}

	/**
	 * Define which scheduled job columns to show on this screen.
	 *
	 * @param array $columns Existing columns.
	 * @return array
	 */
	public static function define_scheduled_jobs_columns( $columns ) {
		$columns = array(
			'cb'         => $columns[ 'cb' ],
			'job'        => __( 'Job', 'quote-request-for-woocommerce' ),
			'next_run'   => __( 'Next Run', 'quote-request-for-woocommerce' ),
			'recurrence' => __( 'Recurrence', 'quote-request-for-woocommerce' ),
			'args'       => __( 'Arguments', 'quote-request-for-woocommerce' ),
			'group'      => __( 'Group', 'quote-request-for-woocommerce' ),
			'relation'   => __( 'Relation', 'quote-request-for-woocommerce' ),
				) ;
		return $columns ;
	}

	/**
	 * Define which columns are sortable.
	 *
	 * @param array $columns Existing columns.
	 * @return array
	 */
	public static function define_sortable_columns( $columns ) {
		global $current_screen ;

		if ( ! isset( $current_screen->post_type ) ) {
			return $columns ;
		}

		$columns = array() ;
		switch ( $current_screen->post_type ) {
			case 'qts_quote_request':
				$columns = array(
					'quote_id'       => 'ID',
					'order_id'       => 'order_id',
					'submitted_by'   => 'user_email',
					'requested_date' => 'date',
						) ;
				break ;
			case 'qts_form_field':
				$columns = array(
					'label' => 'post_title',
					'type'  => 'post_excerpt',
						) ;
				break ;
			case 'qts_scheduled_jobs':
				$columns = array(
					'job'      => 'post_title',
					'next_run' => 'next_run',
					'group'    => 'post_excerpt',
						) ;
				break ;
		}

		return wp_parse_args( $columns, $columns ) ;
	}

	/**
	 * Define bulk actions.
	 *
	 * @param array $actions Existing actions.
	 * @return array
	 */
	public static function define_bulk_actions( $actions ) {
		global $current_screen ;

		switch ( $current_screen->post_type ) {
			case 'qts_scheduled_jobs':
				$actions                            = array() ;
				$actions[ 'delete' ]                = __( 'Delete' ) ;
				break ;
			case 'qts_form_field':
				$actions                            = array() ;
				$actions[ QTS_PREFIX . 'enabled' ]  = __( 'Move to Enabled', 'quote-request-for-woocommerce' ) ;
				$actions[ QTS_PREFIX . 'disabled' ] = __( 'Move to Disabled', 'quote-request-for-woocommerce' ) ;
				$actions[ 'field_required' ]        = __( 'Set as Required', 'quote-request-for-woocommerce' ) ;
				$actions[ 'field_optional' ]        = __( 'Set as Optional', 'quote-request-for-woocommerce' ) ;
				$actions[ 'delete' ]                = __( 'Delete' ) ;
				break ;
		}

		return $actions ;
	}

	/**
	 * Handle bulk actions.
	 *
	 * @param  string $redirect_to URL to redirect to.
	 * @param  string $action      Action name.
	 * @param  array  $ids         List of ids.
	 * @return string
	 */
	public static function handle_bulk_actions( $redirect_to, $action, $ids ) {
		$changed = 0 ;

		switch ( $action ) {
			case QTS_PREFIX . 'enabled':
			case QTS_PREFIX . 'disabled':
				$post_type = 'qts_form_field' ;

				foreach ( $ids as $id ) {
					if ( QTS_Form_Fields::is_default_field( $id ) && in_array( QTS_Form_Fields::get_prop( $id, 'name' ), array( 'billing_email', 'billing_first_name' ) ) ) {
						continue ;
					}

					wp_update_post( array_merge( array( 'ID' => $id ), array( 'post_status' => $action ) ) ) ;
					$changed ++ ;
				}
				break ;
			case 'field_required':
				$post_type = 'qts_form_field' ;

				foreach ( $ids as $id ) {
					update_post_meta( $id, '_required', '1' ) ;
					$changed ++ ;
				}
				break ;
			case 'field_optional':
				$post_type = 'qts_form_field' ;

				foreach ( $ids as $id ) {
					if ( QTS_Form_Fields::is_default_field( $id ) && in_array( QTS_Form_Fields::get_prop( $id, 'name' ), array( 'billing_email', 'billing_first_name' ) ) ) {
						continue ;
					}

					delete_post_meta( $id, '_required' ) ;
					$changed ++ ;
				}
				break ;
		}

		if ( $changed ) {
			$redirect_to = add_query_arg( array(
				'post_type'   => $post_type,
				'bulk_action' => $action,
				'changed'     => $changed,
				'ids'         => join( ',', $ids ),
					), $redirect_to ) ;
		}

		return esc_url_raw( $redirect_to ) ;
	}

	/**
	 * Render individual quote request columns.
	 *
	 * @param string $column Column ID to render.
	 * @param int    $post_id Post ID.
	 */
	public static function render_quote_request_columns( $column, $post_id ) {
		$quote_request = new QTS_Quote_Request( $post_id ) ;

		switch ( $column ) {
			case 'quote_id':
				printf( '<a href="%1$s">#%2$s</a>', esc_url( admin_url( "post.php?post={$quote_request->get_id()}&action=edit" ) ), esc_html( $quote_request->get_id() ) ) ;
				break ;
			case 'status':
				echo esc_html( _qts_get_quote_request_status_name( $quote_request->get_status() ) ) ;
				break ;
			case 'requested_date':
				echo wp_kses_post( _qts_format_datetime( $quote_request->get_date_created() ? $quote_request->get_date_created()->getTimestamp() : ''  ) ) ;
				break ;
			case 'mode':
				$quote_request->is_created_via( 'checkout' ) ? esc_html_e( 'Accurate Quote', 'quote-request-for-woocommerce' ) : esc_html_e( 'Approximate Quote', 'quote-request-for-woocommerce' ) ;
				break ;
			case 'submitted_by':
				echo wp_kses_post( $quote_request->get_customer_billing_address_html() ) ;
				break ;
			case 'order_id':
				if ( $quote_request->get_order_id() ) {
					printf( '<a href="%1$s" target="_blank">#%2$s</a>', esc_url( admin_url( "post.php?post={$quote_request->get_order_id()}&action=edit" ) ), esc_html( $quote_request->get_order_id() ) ) ;
				} else {
					echo '-' ;
				}
				break ;
			case 'more_info':
				printf( '<a href="%1$s">%2$s</a>', esc_url( admin_url( "post.php?post={$quote_request->get_id()}&action=edit" ) ), esc_html__( 'View more', 'quote-request-for-woocommerce' ) ) ;
				break ;
		}
	}

	/**
	 * Render individual form field columns.
	 *
	 * @param string $column Column ID to render.
	 * @param int    $post_id Post ID.
	 */
	public static function render_form_field_columns( $column, $post_id ) {
		switch ( $column ) {
			case 'label':
				printf( '<a href="%1$s">%2$s</a>', esc_url( get_admin_url( null, 'post.php?post=' . $post_id . '&action=edit' ) ), esc_html( QTS_Form_Fields::get_prop( $post_id, 'label' ) ) ) ;
				break ;
			case 'type':
				echo esc_html( _qts_get_form_field_type_name( QTS_Form_Fields::get_prop( $post_id, 'type' ) ) ) ;
				break ;
			case 'status':
				echo esc_html( _qts_get_form_field_status_name( QTS_Form_Fields::get_prop( $post_id, 'status' ) ) ) ;
				break ;
			case 'required':
				echo QTS_Form_Fields::get_prop( $post_id, 'required' ) ? esc_html__( 'Yes', 'quote-request-for-woocommerce' ) : esc_html__( 'No', 'quote-request-for-woocommerce' ) ;
				break ;
		}
	}

	/**
	 * Render individual scheduled job columns.
	 *
	 * @param string $column Column ID to render.
	 * @param int    $post_id Post ID.
	 */
	public static function render_scheduled_jobs_columns( $column, $post_id ) {
		$job      = get_post( $post_id ) ;
		$job_meta = get_post_meta( $post_id ) ;

		switch ( $column ) {
			case 'job':
				echo wp_kses_post( $job->post_title ) ;
				break ;
			case 'next_run':
				$scheduled_on  = isset( $job_meta[ '_scheduled_on' ][ 0 ] ) ? $job_meta[ '_scheduled_on' ][ 0 ] : '' ;
				echo empty( $scheduled_on ) ? '--' : wp_kses_post( _qts_prepare_datetime( $scheduled_on ) . nl2br( "\n[" . _qts_get_human_time_diff( $scheduled_on ) . ']' ) ) ;
				break ;
			case 'recurrence':
				$job_schedules = _qts_get_recurring_job_schedules() ;
				echo isset( $job_meta[ '_recurrence' ][ 0 ], $job_schedules[ $job_meta[ '_recurrence' ][ 0 ] ] ) ? esc_html( $job_schedules[ $job_meta[ '_recurrence' ][ 0 ] ][ 'display' ] ) : 'Non-repeating' ;
				break ;
			case 'args':
				echo wp_kses_post( $job->post_content ) ;
				break ;
			case 'group':
				echo wp_kses_post( $job->post_excerpt ) ;
				break ;
			case 'relation':
				if ( $job->post_parent ) {
					if ( 'qts_quote_request' === get_post_type( $job->post_parent ) ) {
						printf( '<a href="%1$s" target="_blank">#%2$s</a> Parent of #%3$s', esc_url( admin_url( "post.php?post={$job->post_parent}&action=edit" ) ), esc_html( $job->post_parent ), esc_html( $job->ID ) ) ;
					} else {
						echo wp_kses_post( "#{$job->post_parent} Parent of #{$job->ID}" ) ;
					}
				} else {
					echo wp_kses_post( "#{$job->ID}" ) ;
				}
				break ;
		}
	}

	/**
	 * Change messages when a post type is updated.
	 *
	 * @param  array $messages Array of messages.
	 * @return array
	 */
	public static function post_updated_messages( $messages ) {
		global $post ;

		$messages[ 'qts_quote_request' ] = array(
			0  => '', // Unused. Messages start at index 1.
			1  => __( 'Quote request updated.', 'quote-request-for-woocommerce' ),
			4  => __( 'Quote request updated.', 'quote-request-for-woocommerce' ),
			6  => __( 'Quote request updated.', 'quote-request-for-woocommerce' ),
			7  => __( 'Quote request saved.', 'quote-request-for-woocommerce' ),
			8  => __( 'Quote request submitted.', 'quote-request-for-woocommerce' ),
			11 => __( 'Quote request updated and sent.', 'quote-request-for-woocommerce' ),
			12 => __( 'Quote request updated and failed to sent.', 'quote-request-for-woocommerce' ),
				) ;

		$messages[ 'qts_form_field' ] = array(
			0  => '', // Unused. Messages start at index 1.
			1  => __( 'Form Field updated.', 'quote-request-for-woocommerce' ),
			2  => __( 'Custom field updated.', 'quote-request-for-woocommerce' ),
			3  => __( 'Custom field deleted.', 'quote-request-for-woocommerce' ),
			4  => __( 'Form Field updated.', 'quote-request-for-woocommerce' ),
			5  => __( 'Revision restored.', 'quote-request-for-woocommerce' ),
			6  => __( 'Form Field updated.', 'quote-request-for-woocommerce' ),
			7  => __( 'Form Field saved.', 'quote-request-for-woocommerce' ),
			8  => __( 'Form Field submitted.', 'quote-request-for-woocommerce' ),
			9  => sprintf(
					/* translators: %s: date */
					__( 'Form Field scheduled for: %s.', 'quote-request-for-woocommerce' ),
						'<strong>' . date_i18n( __( 'M j, Y @ G:i', 'quote-request-for-woocommerce' ), strtotime( $post->post_date ) ) . '</strong>'
			),
			10 => __( 'Form Field draft updated.', 'quote-request-for-woocommerce' ),
				) ;

		return $messages ;
	}

	/**
	 * Change title boxes in admin.
	 *
	 * @param string  $text Text to shown.
	 * @param WP_Post $post Current post object.
	 * @return string
	 */
	public static function enter_title_here( $text, $post ) {
		if ( 'qts_form_field' === $post->post_type ) {
			$text = esc_html__( 'Field label', 'quote-request-for-woocommerce' ) ;
		}

		return $text ;
	}

	/**
	 * Define which WC order columns to show on this screen.
	 *
	 * @param array $existing_columns Existing columns.
	 * @return array
	 */
	public static function define_shop_order_columns( $existing_columns ) {
		$existing_columns[ QTS_PREFIX . 'quote_request_info' ] = __( 'Quote Request Info', 'quote-request-for-woocommerce' ) ;
		return $existing_columns ;
	}

	/**
	 * Render individual WC order columns.
	 *
	 * @param string $column Column ID to render.
	 * @param int    $post_id Post ID.
	 */
	public static function render_shop_order_columns( $column, $post_id ) {

		if ( QTS_PREFIX . 'quote_request_info' === $column ) {
			if ( 'yes' === QTS_Order_Manager::get_meta( $post_id, 'is_quote_request_order' ) ) {
				$quote_request_id = QTS_Order_Manager::get_meta( $post_id, 'quote_request_id' ) ;

				if ( is_numeric( $quote_request_id ) ) {
					/* translators: 1: edit quote request url */
					printf( esc_html__( 'This Order is linked with Quote Request %1$s', 'quote-request-for-woocommerce' ), '<a href="' . esc_url( admin_url( "post.php?post={$quote_request_id}&action=edit" ) ) . '" target="_blank">#' . esc_html( $quote_request_id ) . '</a>' ) ;
				} else {
					echo '--' ;
				}
			} else {
				echo '--' ;
			}
		}
	}

	/**
	 * Set row actions.
	 *
	 * @param array   $actions Array of actions.
	 * @param WP_Post $post Current post object.
	 * @return array
	 */
	public static function row_actions( $actions, $post ) {
		switch ( $post->post_type ) {
			case 'qts_quote_request':
				unset( $actions[ 'inline hide-if-no-js' ], $actions[ 'view' ], $actions[ 'edit' ] ) ;
				break ;
			case 'qts_form_field':
				$actions = array_filter( $actions, function( $key ) {
					return 'edit' === $key ;
				}, ARRAY_FILTER_USE_KEY ) ;

				if ( ! QTS_Form_Fields::is_default_field( $post->ID ) ) {
					$actions[ 'delete' ] = sprintf( '<a href="%s" class="submitdelete" aria-label="Delete">%s</a>', get_delete_post_link( $post->ID, '', true ), __( 'Delete' ) ) ;
				}
				break ;
			case 'qts_scheduled_jobs':
				unset( $actions ) ;

				$actions[ 'run-job' ] = sprintf( '<a href="%s" aria-label="Run Now">%s</a>', wp_nonce_url( add_query_arg( array( 'action' => 'run-job', 'group' => $post->post_excerpt, 'parent' => $post->post_parent ), admin_url( "edit.php?post_type={$post->post_type}&job_id={$post->ID}" ) ), "run-job-{$post->post_excerpt}-{$post->ID}", QTS_PREFIX . 'nonce' ), __( 'Run Now' ) ) ;
				$actions[ 'delete' ]  = sprintf( '<a href="%s" class="submitdelete" aria-label="Delete Job">%s</a>', get_delete_post_link( $post->ID, '', true ), __( 'Delete' ) ) ;
				break ;
		}
		return $actions ;
	}

	/**
	 * Handle any filters.
	 *
	 * @param array $query_vars Query vars.
	 * @return array
	 */
	public static function request_query( $query_vars ) {
		global $typenow ;

		if ( ! in_array( $typenow, array_keys( self::$custom_post_types ) ) ) {
			return $query_vars ;
		}

		// Filter the quote requests by the posted customer.
		if ( ! empty( $_GET[ '_customer_user' ] ) ) {
			$query_vars[ 'meta_query' ] = array(
				array(
					'key'     => '_customer_user',
					'value'   => absint( $_GET[ '_customer_user' ] ),
					'compare' => '=',
				),
					) ;
		}

		//Sorting
		if ( 'qts_form_field' === $typenow ) {
			$query_vars[ 'orderby' ] = 'menu_order' ;
			$query_vars[ 'order' ]   = 'ASC' ;
		}

		if ( empty( $query_vars[ 'orderby' ] ) ) {
			$query_vars[ 'orderby' ] = 'ID' ;
		}

		if ( empty( $query_vars[ 'order' ] ) ) {
			$query_vars[ 'order' ] = 'DESC' ;
		}

		if ( ! empty( $query_vars[ 'orderby' ] ) ) {
			switch ( $query_vars[ 'orderby' ] ) {
				case 'next_run':
					$query_vars[ 'meta_key' ]  = '_scheduled_on' ;
					$query_vars[ 'meta_type' ] = 'DATETIME' ;
					$query_vars[ 'orderby' ]   = 'meta_value' ;
					break ;
				case 'user_email':
				case 'order_id':
					$query_vars[ 'meta_key' ]  = "_{$query_vars[ 'orderby' ]}" ;
					$query_vars[ 'orderby' ]   = 'meta_value' ;
					break ;
				case 'balance':
					$query_vars[ 'meta_key' ]  = "_{$query_vars[ 'orderby' ]}" ;
					$query_vars[ 'orderby' ]   = 'meta_value_num' ;
					break ;
			}
		}

		return $query_vars ;
	}

	/**
	 * Need to prevent the deletion of the post?
	 *
	 * @param bool $bool
	 * @param WP_Post $post
	 * @return bool
	 */
	public static function need_post_deletion( $bool, $post ) {
		if ( 'qts_form_field' === $post->post_type && QTS_Form_Fields::is_default_field( $post->ID ) ) {
			$bool = false ;
		}

		return $bool ;
	}

	/**
	 * Fire our actions performed in admin screen.
	 */
	public static function admin_action() {
		if ( ! isset( $_GET[ 'action' ] ) ) {
			return ;
		}

		$action = sanitize_title( wp_unslash( $_GET[ 'action' ] ) ) ;

		if ( 'run-job' === $action ) {
			if ( isset( $_GET[ QTS_PREFIX . 'nonce' ] ) ) {
				$nonce = sanitize_key( wp_unslash( $_GET[ QTS_PREFIX . 'nonce' ] ) ) ;

				if ( ! isset( $_GET[ 'job_id' ] ) || ! isset( $_GET[ 'group' ] ) || ! isset( $_GET[ 'post_type' ] ) ) {
					return ;
				}

				$job_id    = absint( wp_unslash( $_GET[ 'job_id' ] ) ) ;
				$group     = sanitize_title( wp_unslash( $_GET[ 'group' ] ) ) ;
				$post_type = sanitize_title( wp_unslash( $_GET[ 'post_type' ] ) ) ;

				if ( ! wp_verify_nonce( $nonce, "{$action}-{$group}-{$job_id}" ) ) {
					return ;
				}

				$queue_group = _qts_get_queue( $group ) ;

				if ( ! $queue_group ) {
					return ;
				}

				$queue = $queue_group->get_queue() ;

				if ( empty( $queue ) ) {
					return ;
				}

				$item   = $queue_group->prepare_queue_item( $job_id ) ;
				$status = 'failed' ;

				if ( $queue_group->is_queue_valid( $item ) ) {
					$queue_group->handle( $item ) ;
					$status = 'success' ;
				}

				wp_safe_redirect( esc_url_raw( admin_url( "edit.php?post_type={$post_type}&qts-notice=job-ran-{$status}&job-name={$item->name}" ) ) ) ;
				exit ;
			}
		}

		if ( 'edit' === $action ) {
			if ( isset( $_GET[ 'post' ] ) ) {
				$post_id       = absint( wp_unslash( $_GET[ 'post' ] ) ) ;
				$quote_request = _qts_get_quote_request( $post_id ) ;

				if ( $quote_request && $quote_request->has_status( 'new' ) ) {
					$quote_request->update_status( 'on_hold' ) ;
				}
			}
		}
	}

	/**
	 * Change the label when searching index.
	 *
	 * @param mixed $query Current search query.
	 * @return string
	 */
	public static function search_label( $query ) {
		global $pagenow, $typenow ;

		if ( 'edit.php' !== $pagenow || ! in_array( $typenow, array_keys( self::$custom_post_types ) ) || ! get_query_var( "{$typenow}_search" ) || ! isset( $_GET[ 's' ] ) ) { // WPCS: input var ok.
			return $query ;
		}

		return wc_clean( wp_unslash( $_GET[ 's' ] ) ) ; // WPCS: input var ok, sanitization ok.
	}

	/**
	 * Query vars for custom searches.
	 *
	 * @param mixed $public_query_vars Array of query vars.
	 * @return array
	 */
	public static function add_custom_query_var( $public_query_vars ) {
		return array_merge( $public_query_vars, array_map( function( $type ) {
					return "{$type}_search" ;
		}, array_keys( self::$custom_post_types ) ) ) ;
	}

	/**
	 * Search custom fields as well as content.
	 *
	 * @param WP_Query $wp Query object.
	 */
	public static function search_custom_fields( $wp ) {
		global $pagenow, $wpdb ;

		if ( 'edit.php' !== $pagenow || empty( $wp->query_vars[ 's' ] ) || ! in_array( $wp->query_vars[ 'post_type' ], array_keys( self::$custom_post_types ) ) || ! isset( $_GET[ 's' ] ) ) { // WPCS: input var ok.
			return ;
		}

		$wpdb_ref = &$wpdb ;
		$term     = str_replace( '#', '', wc_clean( wp_unslash( $wp->query_vars[ 's' ] ) ) ) ;
		$post_ids = array() ;

		switch ( $wp->query_vars[ 'post_type' ] ) {
			case 'qts_quote_request':
				$search_fields = array(
					'_billing_address_index',
					'_shipping_address_index',
					'_billing_last_name',
					'_billing_email',
					'_customer_first_name',
					'_customer_last_name',
					'_order_id',
					'_created_via',
						) ;
				break ;
		}

		if ( empty( $search_fields ) ) {
			return ;
		}

		if ( is_numeric( $term ) ) {
			$post_ids[] = absint( $term ) ;
		}

		if ( 'qts_quote_request' === $wp->query_vars[ 'post_type' ] ) {
			$post_ids = array_unique(
					array_merge( $post_ids, $wpdb_ref->get_col(
									$wpdb_ref->prepare(
											"SELECT DISTINCT p1.post_id FROM {$wpdb_ref->postmeta} p1 WHERE p1.meta_value LIKE %s AND p1.meta_key IN ('" . implode( "','", array_map( 'esc_sql', $search_fields ) ) . "')", '%' . $wpdb_ref->esc_like( wc_clean( $term ) ) . '%'
									)
							), $wpdb_ref->get_col(
									$wpdb_ref->prepare(
											"SELECT order_id FROM {$wpdb_ref->prefix}woocommerce_order_items as order_items WHERE order_item_name LIKE %s", '%' . $wpdb_ref->esc_like( wc_clean( $term ) ) . '%'
									)
							)
					) ) ;
		}

		if ( ! empty( $post_ids ) ) {
			// Remove "s" - we don't want to search our post name.
			unset( $wp->query_vars[ 's' ] ) ;

			// so we know we're doing this.
			$wp->query_vars[ "{$wp->query_vars[ 'post_type' ]}_search" ] = true ;

			// Search by found posts.
			$wp->query_vars[ 'post__in' ] = array_merge( $post_ids, array( 0 ) ) ;
		}
	}

}

QTS_Admin_Post_Types::init() ;
