<?php

defined( 'ABSPATH' ) || exit ;

/**
 * Add to Quote Handler.
 *
 * @class QTS_Add_To_Quote
 * @package Classes
 */
class QTS_Add_To_Quote {

	/**
	 * Get the mode of how add to quote is handled.
	 * 
	 * @var string
	 */
	protected static $mode ;

	/**
	 * Get the pages available to render the add to quote.
	 * 
	 * @var array
	 */
	protected static $pages_to_render ;

	/**
	 * Get the add to cart button hide option.
	 * 
	 * @var bool
	 */
	protected static $hide_add_to_cart ;

	/**
	 * Get the product price hide option.
	 * 
	 * @var bool
	 */
	protected static $hide_product_price ;

	/**
	 * Get the quote props.
	 * 
	 * @var array
	 */
	protected static $quote_props = array() ;

	/**
	 * Get the default quote props.
	 * 
	 * @var array
	 */
	protected static $default_quote_props = array(
		'enabled'                    => null,
		'min_quote_price_type'       => null,
		'fixed_min_quote_price'      => null,
		'percentage_min_quote_price' => null,
			) ;

	/**
	 * Hook in methods.
	 */
	public static function init() {
		add_action( 'woocommerce_after_shop_loop_item', array( __CLASS__, 'render_add_to_quote_button' ), 20 ) ;
		add_action( 'woocommerce_after_add_to_cart_button', array( __CLASS__, 'render_add_to_quote_button' ) ) ;
		add_filter( 'woocommerce_available_variation', array( __CLASS__, 'render_single_variation_add_to_quote_button' ), 10, 3 ) ;
		add_filter( 'woocommerce_product_is_in_stock', array( __CLASS__, 'maybe_allow_add_to_quote_for_outofstock' ), 99, 2 ) ;
		add_filter( 'woocommerce_loop_add_to_cart_link', array( __CLASS__, 'maybe_hide_add_to_cart_in_shop' ), 99, 2 ) ;
		add_filter( 'woocommerce_get_price_html', array( __CLASS__, 'maybe_hide_product_price' ), 99, 2 ) ;
		add_filter( 'qts_get_product_props', array( __CLASS__, 'get_product_eligibility' ), 99, 2 ) ;
		add_filter( 'qts_is_non_single_add_to_quote_page', array( __CLASS__, 'is_non_single_add_to_quote_page' ), 20, 2 ) ;
	}

	/**
	 * Getter method.
	 * 
	 * @param int $product_id
	 * @param string $prop
	 * @return mixed
	 */
	public static function _get( $product_id, $prop ) {
		if ( ! self::_isset( $product_id, $prop ) ) {
			return null ;
		}

		return self::$quote_props[ $product_id ][ $prop ] ;
	}

	/**
	 * Isset method.
	 * 
	 * @param int $product_id
	 * @param string $prop
	 * @return bool
	 */
	public static function _isset( $product_id, $prop = null ) {
		if ( is_null( $prop ) ) {
			return isset( self::$quote_props[ $product_id ] ) ;
		}

		return isset( self::$quote_props[ $product_id ][ $prop ] ) ;
	}

	/**
	 * Setter method.
	 * 
	 * @param int $product_id
	 * @param string $prop
	 * @param mixed $value
	 */
	public static function _set( $product_id, $prop, $value ) {
		self::$quote_props[ $product_id ][ $prop ] = $value ;
	}

	/**
	 * Gets the array of props.
	 * 
	 * @param int $product_id
	 * @return array
	 */
	public static function _gets( $product_id ) {
		if ( ! self::_isset( $product_id ) ) {
			return self::get_default_props() ;
		}

		return self::$quote_props[ $product_id ] ;
	}

	/**
	 * Get the mode of how add to quote is handled.
	 * 
	 * @param bool $from_session True only if the mode is chosen by the customer.
	 * @return string
	 */
	public static function get_mode( $from_session = false ) {
		if ( $from_session ) {
			if ( is_null( WC()->session ) ) {
				return '' ;
			}

			return ! is_null( WC()->session->get( QTS_PREFIX . 'add_to_quote_chosen_mode' ) ) ? WC()->session->get( QTS_PREFIX . 'add_to_quote_chosen_mode' ) : '' ;
		} else {
			if ( ! is_null( self::$mode ) ) {
				return self::$mode ;
			}

			self::$mode = get_option( QTS_PREFIX . 'quote_request_mode' ) ;
			return self::$mode ;
		}
	}

	/**
	 * Get the pages available to render the add to quote.
	 * 
	 * @var array
	 */
	public static function get_pages_to_render() {
		if ( ! is_null( self::$pages_to_render ) ) {
			return self::$pages_to_render ;
		}

		self::$pages_to_render = get_option( QTS_PREFIX . 'display_request_quote_page_in', array() ) ;
		return self::$pages_to_render ;
	}

	/**
	 * Check whether to hide the add to cart.
	 *   
	 * @return bool
	 */
	public static function hide_add_to_cart() {
		if ( ! is_null( self::$hide_add_to_cart ) ) {
			return self::$hide_add_to_cart ;
		}

		self::$hide_add_to_cart = 'yes' === get_option( QTS_PREFIX . 'hide_add_to_cart' ) ? true : false ;
		return self::$hide_add_to_cart ;
	}

	/**
	 * Check whether to hide the product price if add to cart button is hidden.
	 * 
	 * @return bool
	 */
	public static function hide_product_price() {
		if ( ! is_null( self::$hide_product_price ) ) {
			return self::$hide_product_price ;
		}

		self::$hide_product_price = 'yes' === get_option( QTS_PREFIX . 'hide_product_price' ) ? true : false ;
		return self::$hide_product_price ;
	}

	/**
	 * Maybe prompt the guest to log in when the customer add to quote button is clicked.
	 */
	public static function prompt_guest_to_log_in() {
		return ! is_user_logged_in() && 'yes' === get_option( QTS_PREFIX . 'display_add_to_quote_for_guests' ) && 'force-login' === get_option( QTS_PREFIX . 'when_guest_clicked_add_to_quote' ) ;
	}

	/**
	 * Maybe get the product instance if available.
	 * 
	 * @param mixed $product
	 * @return WC_Product|bool
	 */
	public static function maybe_get_product_instance( $product ) {
		if ( $product ) {
			if ( is_a( $product, 'WC_Product' ) ) {
				return $product ;
			}

			if ( ! empty( $product->ID ) ) {
				return wc_get_product( $product->ID ) ;
			}

			if ( is_numeric( $product ) ) {
				return wc_get_product( absint( $product ) ) ;
			}
		}

		return false ;
	}

	/**
	 * Check whether the given product is quote enabled product.
	 * 
	 * @param mixed $product
	 * @return bool 
	 */
	public static function is_quote_enabled( $product ) {
		$product = self::maybe_get_product_instance( $product ) ;

		if ( ! $product ) {
			return false ;
		}

		if ( ! self::_isset( $product->get_id() ) ) {
			self::prepare_props( $product ) ;
		}

		return self::_get( $product->get_id(), 'enabled' ) ;
	}

	/**
	 * Return default quote props.
	 * 
	 * @return array
	 */
	public static function get_default_props() {
		return self::$default_quote_props ;
	}

	/**
	 * Prepare the quote props.
	 * 
	 * @param WC_Product|int $product
	 */
	public static function prepare_props( $product ) {
		$product = self::maybe_get_product_instance( $product ) ;
		$props   = array() ;

		if ( 'yes' !== get_option( QTS_PREFIX . 'quote_request_enabled' ) ) {
			unset( self::$quote_props[ $product->get_id() ] ) ;
			return ;
		}

		$props[ 'enabled' ]                      = true ;
		$props[ 'min_quote_price_type' ]         = 'percentage' ;
		$props[ 'fixed_min_quote_price' ]        = 0 ;
		$props[ 'percentage_min_quote_price' ]   = floatval( get_option( QTS_PREFIX . 'min_quote_req_price_percentage' ) ) ;
		self::$quote_props[ $product->get_id() ] = wp_parse_args( ( array ) apply_filters( 'qts_get_product_props', $props, $product ), self::get_default_props() ) ;
	}

	/**
	 * Gets the template or return the template elsewhere to display.s
	 * 
	 * @param WC_Product $product
	 * @param array $args @see self::prepare_add_to_quote_args
	 * @param bool $echo Should this be a flat list or HTML list? 
	 * @return string
	 */
	public static function get_add_to_quote( $product, $args = array(), $echo = true ) {
		if ( empty( $args ) ) {
			$args = self::prepare_add_to_quote_args( $product ) ;
		}

		ob_start() ;

		echo '<div class="qts-add-to-quote-wrapper">' ;
		_qts_get_template( 'add-to-quote.php', $args ) ;
		echo '</div>' ;

		if ( $echo ) {
			ob_end_flush() ;
		} else {
			return ob_get_clean() ;
		}
	}

	/**
	 * Prepare the array of arguments based on the quote request mode for the add to quote button.
	 * 
	 * @param WC_Product $product
	 * @param array $args
	 * @return array
	 */
	public static function prepare_add_to_quote_args( $product, $args = array() ) {
		$non_single_page = apply_filters( 'qts_is_non_single_add_to_quote_page', is_shop(), $product ) ;
		$is_single_page  = apply_filters( 'qts_is_single_add_to_quote_page', is_product() && ! $non_single_page, $product ) ;

		$args = wp_parse_args( $args, array(
			'mode'                => ( 'user-choice' === self::get_mode() && '' !== self::get_mode( true ) ) ? self::get_mode( true ) : self::get_mode(),
			'show_user_options'   => $is_single_page && 'user-choice' === self::get_mode(),
			'ajax_add_to_quote'   => $non_single_page && 'yes' === get_option( QTS_PREFIX . 'ajax_add_to_quote_enabled' ) && 'user-choice' !== self::get_mode(),
			'single_add_to_quote' => $is_single_page
				) ) ;

		$class_name = 'add-to-qtsquote' ;
		if ( 'button' === get_option( QTS_PREFIX . 'request_quote_input_type' ) ) {
			$class_name .= ' button' ;
		}

		if ( $args[ 'ajax_add_to_quote' ] ) {
			$class_name .= ' ajax-add-to-qtsquote' ;
		} else if ( $args[ 'single_add_to_quote' ] ) {
			$class_name .= ' single-add-to-qtsquote' ;
		}

		if ( $non_single_page && 'user-choice' === self::get_mode() ) {
			$add_to_quote_url = _qts_add_to_quote_url( $product, self::get_mode() ) ;
		} else {
			$add_to_quote_url = _qts_add_to_quote_url( $product, $args[ 'mode' ] ) ;
		}

		return apply_filters( 'qts_add_to_quote_args', array(
			'product'            => $product,
			'class_name'         => $class_name,
			'add_to_quote_url'   => $add_to_quote_url,
			'show_user_options'  => $args[ 'show_user_options' ],
			'chosen_mode'        => str_replace( 'user-choice', '', $args[ 'mode' ] ),
			'add_to_quote_label' => get_option( QTS_PREFIX . 'add_to_quote_label' ),
				), $product ) ;
	}

	/**
	 * Check whether the add to quote button is displayed in non single pages. 
	 * 
	 * @global array $woocommerce_loop
	 * @param bool $bool
	 * @param WC_Product $product
	 * @return bool
	 */
	public static function is_non_single_add_to_quote_page( $bool, $product ) {
		global $woocommerce_loop ;
		return is_shop() || ( isset( $woocommerce_loop[ 'name' ] ) && in_array( $woocommerce_loop[ 'name' ], array( 'up-sells', 'related' ) ) ) ;
	}

	/**
	 * Maybe allow add to quote button eligibility when the product is not in stock.
	 * 
	 * @param bool $in_stock
	 * @return bool
	 */
	public static function maybe_allow_add_to_quote_for_outofstock( $in_stock, $product ) {
		if ( $in_stock || ! self::is_quote_enabled( $product ) ) {
			return $in_stock ;
		}

		return true ;
	}

	/**
	 * Hide the add to cart button in Single Product page.
	 * 
	 * @param mixed $product
	 * @return string
	 */
	public static function maybe_hide_add_to_cart_in_single( $product ) {
		$product = self::maybe_get_product_instance( $product ) ;

		if ( ! $product || ! self::is_quote_enabled( $product ) ) {
			return '' ;
		}

		$css = '' ;
		if ( self::hide_add_to_cart() || 'outofstock' === $product->get_stock_status() ) {
			if ( $product->is_type( 'variable' ) ) {
				$css = '.single_variation_wrap .variations_button button{
	                 display:none!important;
	                }' ;
			} else {
				$css = '.cart button.single_add_to_cart_button, .cart a.single_add_to_cart_button{
	                 display:none!important;
	                }' ;
			}
		}

		return $css ;
	}

	/**
	 * Hide the add to cart button in Shop page.
	 * 
	 * @param string $link
	 * @param WC_Product $product
	 * @return string
	 */
	public static function maybe_hide_add_to_cart_in_shop( $link, $product ) {
		$product = self::maybe_get_product_instance( $product ) ;

		if ( ! $product || ! $product->is_type( 'simple' ) || ! self::is_quote_enabled( $product ) ) {
			return $link ;
		}

		if ( self::hide_add_to_cart() || 'outofstock' === $product->get_stock_status() ) {
			$link = '' ;
		}

		return $link ;
	}

	/**
	 * Hide the product price in Shop/Single Product page.
	 * 
	 * @param float $price
	 * @param mixed $product
	 */
	public static function maybe_hide_product_price( $price, $product ) {
		$product = self::maybe_get_product_instance( $product ) ;

		if ( ! $product || ! self::is_quote_enabled( $product ) ) {
			return $price ;
		}

		if ( self::hide_add_to_cart() && self::hide_product_price() ) {
			$price = '' ;
		}

		return $price ;
	}

	/**
	 * Render the add to quote button in Shop and Single product page.
	 */
	public static function render_add_to_quote_button() {
		global $product ;

		$product = self::maybe_get_product_instance( $product ) ;

		if ( ! self::is_quote_enabled( $product ) ) {
			return ;
		}

		$render = true ;
		if ( doing_action( 'woocommerce_after_shop_loop_item' ) ) {
			// Shop page
			if ( $product->is_type( 'variable' ) ) {
				$render = false ;
			} else if ( ! in_array( 'other-pages', self::get_pages_to_render() ) ) {
				$render = false ;
			}
		} else {
			// Single product page
			if ( $product->is_type( 'variable' ) ) {
				$render = false ;
			} else if ( ! in_array( 'single-product', self::get_pages_to_render() ) ) {
				$render = false ;
			}
		}

		if ( $render ) {
			self::get_add_to_quote( $product ) ;
		}
	}

	/**
	 * Render the add to quote button in Single product page for individual variations.
	 * 
	 * @param array $variation_data
	 * @param WC_Product_Variable $variable
	 * @param WC_Product_Variation $variation
	 * @return array
	 */
	public static function render_single_variation_add_to_quote_button( $variation_data, $variable, $variation ) {
		if ( ! self::is_quote_enabled( $variation ) ) {
			return $variation_data ;
		}

		if ( ! in_array( 'single-product', self::get_pages_to_render() ) ) {
			return $variation_data ;
		}

		$variation_data[ QTS_PREFIX . 'add_to_quote' ] = self::get_add_to_quote( $variation, '', false ) ;

		return $variation_data ;
	}

	/**
	 * Check for product eligibility to satisfy as quote enabled product.
	 * 
	 * @param array $props
	 * @param WC_Product $product
	 * @return array
	 */
	public static function get_product_eligibility( $props, $product ) {
		// Product eligibility on quick setup.
		switch ( get_option( QTS_PREFIX . 'get_product_select_type' ) ) {
			case 'include-products':
				$include_products   = ( array ) get_option( QTS_PREFIX . 'get_include_product_selector' ) ;
				$props[ 'enabled' ] = false ;

				if ( in_array( $product->get_parent_id(), $include_products ) || in_array( $product->get_id(), $include_products ) ) {
					$props[ 'enabled' ] = true ;
				}
				break ;
			case 'exclude-products':
				$exclude_products   = ( array ) get_option( QTS_PREFIX . 'get_exclude_product_selector' ) ;
				$props[ 'enabled' ] = true ;

				if ( in_array( $product->get_parent_id(), $exclude_products ) || in_array( $product->get_id(), $exclude_products ) ) {
					$props[ 'enabled' ] = false ;
				}
				break ;
			case 'include-categories':
				$product_id         = $product->get_parent_id() > 0 ? $product->get_parent_id() : $product->get_id() ;
				$terms              = get_the_terms( $product_id, 'product_cat' ) ;
				$include_cat        = ( array ) get_option( QTS_PREFIX . 'get_include_product_category_selector' ) ;
				$props[ 'enabled' ] = false ;

				if ( is_array( $terms ) ) {
					foreach ( $terms as $term ) {
						if ( in_array( $term->term_id, $include_cat ) ) {
							$props[ 'enabled' ] = true ;
							break ;
						}
					}
				}
				break ;
			case 'exclude-categories':
				$product_id         = $product->get_parent_id() > 0 ? $product->get_parent_id() : $product->get_id() ;
				$terms              = get_the_terms( $product_id, 'product_cat' ) ;
				$exclude_cat        = ( array ) get_option( QTS_PREFIX . 'get_exclude_product_category_selector' ) ;
				$props[ 'enabled' ] = true ;

				if ( is_array( $terms ) ) {
					foreach ( $terms as $term ) {
						if ( in_array( $term->term_id, $exclude_cat ) ) {
							$props[ 'enabled' ] = false ;
							break ;
						}
					}
				}
				break ;
		}

		if ( false === $props[ 'enabled' ] ) {
			return $props ;
		}

		// Stock eligibility.
		if ( 'yes' === get_option( QTS_PREFIX . 'display_add_to_quote_for_out_of_stock' ) ) {
			if ( 'yes' === get_option( QTS_PREFIX . 'display_add_to_quote_only_for_out_of_stock' ) ) {
				$props[ 'enabled' ] = 'outofstock' === $product->get_stock_status() ? true : false ;
			}
		} else {
			$props[ 'enabled' ] = 'outofstock' !== $product->get_stock_status() ? true : false ;
		}

		if ( false === $props[ 'enabled' ] ) {
			return $props ;
		}

		// Current user eligibility.
		if ( is_user_logged_in() ) {
			if ( ! current_user_can( 'add_to_qtsquote' ) ) {
				$props[ 'enabled' ] = false ;
			}
		} else {
			if ( 'yes' === get_option( QTS_PREFIX . 'display_add_to_quote_for_guests' ) ) {
				$props[ 'enabled' ] = true ;

				if ( 'allow' !== get_option( QTS_PREFIX . 'when_guest_clicked_add_to_quote' ) ) {
					$props[ 'do_not_add' ] = true ;
				}
			} else {
				$props[ 'enabled' ] = false ;
			}
		}

		return $props ;
	}

}

QTS_Add_To_Quote::init() ;
