<?php
defined( 'ABSPATH' ) || exit ;

/**
 * Privacy/GDPR related functionality which ties into WordPress functionality.
 * 
 * @class QTS_Privacy
 * @package Class
 */
class QTS_Privacy {

	/**
	 * This is a list of exporters.
	 *
	 * @var array
	 */
	protected static $exporters = array() ;

	/**
	 * This is a list of erasers.
	 *
	 * @var array
	 */
	protected static $erasers = array() ;

	/**
	 * Limit background process to number of batches to avoid timeouts
	 *
	 * @var int 
	 */
	protected static $batch_limit = 10 ;

	/**
	 * Force erase personal data from user.
	 *
	 * @var bool 
	 */
	protected static $force_erase_personal_data = false ;

	/**
	 * Init QTS_Privacy.
	 */
	public static function init() {
		self::$force_erase_personal_data = 'yes' === get_option( QTS_PREFIX . 'erasure_request_removes_user_data', 'no' ) ;

		add_action( 'admin_init', __CLASS__ . '::add_privacy_message' ) ;

		self::add_exporter( 'qts-quote-request-users-data', __( 'Quote Request Users Data', 'quote-request-for-woocommerce' ), __CLASS__ . '::quote_request_data_exporter' ) ;
		self::add_eraser( 'qts-quote-request-users-data', __( 'Quote Request Users Data', 'quote-request-for-woocommerce' ), __CLASS__ . '::quote_request_data_eraser' ) ;

		add_filter( 'wp_privacy_personal_data_exporters', __CLASS__ . '::register_exporters', 6 ) ;
		add_filter( 'wp_privacy_personal_data_erasers', __CLASS__ . '::register_erasers' ) ;
	}

	/**
	 * Get plugin name
	 * 
	 * @return string
	 */
	public static function get_plugin_name() {
		$plugin = get_plugin_data( QTS_FILE ) ;
		return $plugin[ 'Name' ] ;
	}

	/**
	 * Adds the privacy message on WC Quote Request privacy page.
	 */
	public static function add_privacy_message() {
		if ( function_exists( 'wp_add_privacy_policy_content' ) ) {
			$content = self::get_privacy_message() ;

			if ( $content ) {
				wp_add_privacy_policy_content( self::get_plugin_name(), $content ) ;
			}
		}
	}

	/**
	 * Integrate this exporter implementation within the WordPress core exporters.
	 *
	 * @param array $exporters List of exporter callbacks.
	 * @return array
	 */
	public static function register_exporters( $exporters = array() ) {
		foreach ( self::$exporters as $id => $exporter ) {
			$exporters[ $id ] = $exporter ;
		}
		return $exporters ;
	}

	/**
	 * Integrate this eraser implementation within the WordPress core erasers.
	 *
	 * @param array $erasers List of eraser callbacks.
	 * @return array
	 */
	public static function register_erasers( $erasers = array() ) {
		foreach ( self::$erasers as $id => $eraser ) {
			$erasers[ $id ] = $eraser ;
		}
		return $erasers ;
	}

	/**
	 * Add exporter to list of exporters.
	 *
	 * @param string $id       ID of the Exporter.
	 * @param string $name     Exporter name.
	 * @param string $callback Exporter callback.
	 */
	public static function add_exporter( $id, $name, $callback ) {
		self::$exporters[ $id ] = array(
			'exporter_friendly_name' => $name,
			'callback'               => $callback,
				) ;
		return self::$exporters ;
	}

	/**
	 * Add eraser to list of erasers.
	 *
	 * @param string $id       ID of the Eraser.
	 * @param string $name     Exporter name.
	 * @param string $callback Exporter callback.
	 */
	public static function add_eraser( $id, $name, $callback ) {
		self::$erasers[ $id ] = array(
			'eraser_friendly_name' => $name,
			'callback'             => $callback,
				) ;
		return self::$erasers ;
	}

	/**
	 * Add privacy policy content for the privacy policy page.
	 */
	public static function get_privacy_message() {
		ob_start() ;
		?>
		<p><?php esc_html_e( 'This includes the basics of what personal data your store may be collecting, storing and sharing. Depending on what settings are enabled and which additional plugins are used, the specific information shared by your store will vary.', 'quote-request-for-woocommerce' ) ; ?></p>
		<h2><?php esc_html_e( 'What the Plugin does', 'quote-request-for-woocommerce' ) ; ?></h2>
		<p><?php esc_html_e( 'Using this plugin, you can allow your users to request for a quote for the products sold on your WooCommerce shop.', 'quote-request-for-woocommerce' ) ; ?></p>
		<h2><?php esc_html_e( 'What we collect and share', 'quote-request-for-woocommerce' ) ; ?></h2>        
		<ul>
			<li><b><?php esc_html_e( 'Email Id, Userid, First Name and Last Name', 'quote-request-for-woocommerce' ) ; ?></b></li>
			<li><?php esc_html_e( '- Used for identifying the Users. ', 'quote-request-for-woocommerce' ) ; ?></li>
			<li><?php esc_html_e( '- Creating Orders for making Quote Payments. ', 'quote-request-for-woocommerce' ) ; ?></li>
		</ul>
		<?php
		return apply_filters( 'qts_privacy_policy_content', ob_get_clean() ) ;
	}

	/**
	 * Finds and exports data which could be used to identify a person from WC Quote Request data associated with an email address.
	 *
	 * Users data are exported in blocks of 10 to avoid timeouts.
	 *
	 * @param string $email_address The user email address.
	 * @param int    $page  Page.
	 * @return array An array of personal data in name value pairs
	 */
	public static function quote_request_data_exporter( $email_address, $page ) {
		$data_to_export = array() ;
		$user           = get_user_by( 'email', $email_address ) ; // Check if user has an ID in the DB to load stored personal data.

		$quote_request_query = array(
			'post_type'   => 'qts_quote_request',
			'post_status' => array_keys( _qts_get_quote_request_statuses() ),
			'limit'       => self::$batch_limit,
			'page'        => absint( $page ),
			'customer'    => array( $email_address ),
				) ;

		if ( $user instanceof WP_User ) {
			$quote_request_query[ 'customer' ][] = ( int ) $user->ID ;
		}

		$quote_requests = wc_get_orders( $quote_request_query ) ;

		if ( 0 < count( $quote_requests ) ) {
			foreach ( $quote_requests as $quote_request ) {
				$data_to_export[] = array(
					'group_id'    => 'qts_quote_requests',
					'group_label' => __( 'Quote Requests', 'quote-request-for-woocommerce' ),
					'item_id'     => 'quote_request-' . $quote_request->get_id(),
					'data'        => self::get_quote_request_personal_data( $quote_request ),
						) ;
			}

			$done = 10 > count( $quote_requests ) ;
		} else {
			$done = true ;
		}

		return array(
			'data' => $data_to_export,
			'done' => $done,
				) ;
	}

	/**
	 * Finds and erases data which could be used to identify a person from WC Quote Request data associated with an email address.
	 *
	 * Users data are erased in blocks of 10 to avoid timeouts.
	 *
	 * @param string $email_address The user email address.
	 * @param int    $page  Page.
	 * @return array An array of personal data in name value pairs
	 */
	public static function quote_request_data_eraser( $email_address, $page ) {
		$user     = get_user_by( 'email', $email_address ) ; // Check if user has an ID in the DB to load stored personal data.
		$response = array(
			'items_removed'  => false,
			'items_retained' => false,
			'messages'       => array(),
			'done'           => true,
				) ;

		$quote_request_query = array(
			'post_type'   => 'qts_quote_request',
			'post_status' => array_keys( _qts_get_quote_request_statuses() ),
			'limit'       => self::$batch_limit,
			'page'        => absint( $page ),
			'customer'    => array( $email_address ),
				) ;

		if ( $user instanceof WP_User ) {
			$quote_request_query[ 'customer' ][] = ( int ) $user->ID ;
		}

		$quote_requests = wc_get_orders( $quote_request_query ) ;

		if ( 0 < count( $quote_requests ) ) {
			foreach ( $quote_requests as $quote_request ) {
				if ( apply_filters( 'qts_privacy_erase_quote_request_personal_data', self::$force_erase_personal_data, $quote_request ) ) {
					self::remove_quote_request_personal_data( $quote_request ) ;

					/* Translators: %s Quote Request number. */
					$response[ 'messages' ][]    = sprintf( __( 'Removed personal data from quote request %s.', 'quote-request-for-woocommerce' ), $quote_request->get_order_number() ) ;
					$response[ 'items_removed' ] = true ;
				} else {
					/* Translators: %s Quote Request number. */
					$response[ 'messages' ][]     = sprintf( __( 'Personal data within quote request %s has been retained.', 'quote-request-for-woocommerce' ), $quote_request->get_order_number() ) ;
					$response[ 'items_retained' ] = true ;
				}
			}
			$response[ 'done' ] = 10 > count( $quote_requests ) ;
		} else {
			$response[ 'done' ] = true ;
		}

		return $response ;
	}

	/**
	 * Get WC Quote Request personal data (key/value pairs) for the User.
	 *
	 * @param object $quote_request
	 * @return array
	 */
	public static function get_quote_request_personal_data( $quote_request ) {
		$personal_data   = array() ;
		$props_to_export = apply_filters( 'qts_privacy_export_quote_request_personal_data_props', array(
			'billing_first_name' => __( 'First Name', 'quote-request-for-woocommerce' ),
			'billing_last_name'  => __( 'Last Name', 'quote-request-for-woocommerce' ),
			'billing_email'      => __( 'Email Address', 'quote-request-for-woocommerce' ),
				), $quote_request ) ;

		foreach ( $props_to_export as $prop => $name ) {
			$value = '' ;

			if ( is_callable( array( $quote_request, 'get_' . $prop ) ) ) {
				$value = $quote_request->{"get_$prop"}() ;
			}

			$value = apply_filters( 'qts_privacy_export_quote_request_personal_data_prop', $value, $prop, $quote_request ) ;

			if ( '' !== $value ) {
				$personal_data[] = array(
					'name'  => $name,
					'value' => $value,
						) ;
			}
		}

		/**
		 * Allow extensions to register their own personal data for the export.
		 *
		 * @param array $personal_data Array of name value pairs to expose in the export.
		 * @param object $quote_request
		 */
		$personal_data = apply_filters( 'qts_privacy_export_quote_request_personal_data', $personal_data, $quote_request ) ;

		return $personal_data ;
	}

	/**
	 * Remove WC Quote Request personal data specific to the User.
	 * 
	 * @param object $quote_request
	 */
	public static function remove_quote_request_personal_data( $quote_request ) {
		$anonymized_data = array() ;

		/**
		 * Allow extensions to remove their own personal data first, so user data is still available.
		 */
		do_action( 'qts_privacy_before_remove_quote_request_personal_data', $quote_request ) ;

		/**
		 * Expose props and data types we'll be anonymizing.
		 */
		$props_to_remove = apply_filters( 'qts_privacy_remove_quote_request_personal_data_props', array(
			'billing_first_name' => 'text',
			'billing_last_name'  => 'text',
			'billing_email'      => 'email',
				), $quote_request ) ;

		if ( ! empty( $props_to_remove ) && is_array( $props_to_remove ) ) {
			foreach ( $props_to_remove as $prop => $data_type ) {
				// Get the current value in edit context.
				$value = $quote_request->{"get_$prop"}( 'edit' ) ;

				// If the value is empty, it does not need to be anonymized.
				if ( '' === $value || empty( $data_type ) ) {
					continue ;
				}

				$anon_value = function_exists( 'wp_privacy_anonymize_data' ) ? wp_privacy_anonymize_data( $data_type, $value ) : '' ;

				/**
				 * Expose a way to control the anonymized value of a prop via 3rd party code.
				 */
				$anonymized_data[ $prop ] = apply_filters( 'qts_privacy_remove_quote_request_personal_data_prop_value', $anon_value, $prop, $value, $data_type, $quote_request ) ;
			}
		}

		// Set all new props and persist the new data to the database.
		$quote_request->set_props( $anonymized_data ) ;
		$quote_request->update_meta_data( '_anonymized', 'yes' ) ;
		$quote_request->save() ;

		/**
		 * Allow extensions to remove their own personal data.
		 */
		do_action( 'qts_privacy_remove_quote_request_personal_data', $quote_request ) ;
	}

}

QTS_Privacy::init() ;
