<?php

/**
 * Email Class.
 * 
 * @abstract QTS_Abstract_Email
 * @extends WC_Email
 */
abstract class QTS_Abstract_Email extends WC_Email {

	/**
	 * Email supports.
	 *
	 * @var array Supports
	 */
	public $supports = array( 'mail_to_admin' ) ;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->template_base = QTS_TEMPLATE_PATH ;

		// Call WC_Email constuctor
		parent::__construct() ;
	}

	/**
	 * Check email supports the given type.
	 *
	 * @param string $type
	 * @return bool
	 */
	public function supports( $type ) {
		return in_array( $type, $this->supports ) ;
	}

	/**
	 * Maybe trigger the sending of this email.
	 */
	public function maybe_trigger() {
		if ( ! $this->is_enabled() ) {
			return ;
		}

		$recipient = $this->get_recipient() ;

		if ( $recipient ) {
			$sent = $this->send( $recipient, $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() ) ;

			if ( $sent ) {
				do_action( 'qts_email_sent', $this ) ;
			} else {
				do_action( 'qts_email_failed_to_sent', $this ) ;
			}
		}
	}

	/**
	 * Get valid recipients.
	 *
	 * @return string
	 */
	public function get_recipient() {
		$recipient = '' ;
		if ( $this->supports( 'recipient' ) ) {
			$recipient = $this->get_option( 'recipient', get_option( 'admin_email' ) ) ;
		} else if ( $this->supports( 'mail_to_admin' ) && 'yes' === $this->get_option( 'mail_to_admin' ) ) {
			$recipient = get_option( 'admin_email' ) ;
		}

		if ( is_null( $this->recipient ) || '' === $this->recipient ) {
			$this->recipient = $recipient ;
		} else if ( '' !== $recipient ) {
			$this->recipient .= ', ' ;
			$this->recipient .= $recipient ;
		}

		return parent::get_recipient() ;
	}

	/**
	 * Get email type.
	 *
	 * @return string
	 */
	public function get_email_type() {
		return class_exists( 'DOMDocument' ) ? 'html' : '' ;
	}

	/**
	 * Format date to display.
	 *
	 * @param int|string $date
	 * @return string
	 */
	public function format_date( $date = '' ) {
		return _qts_format_datetime( $date, false ) ;
	}

	/**
	 * Get content args.
	 *
	 * @return array
	 */
	public function get_content_args() {

		return array(
			'blogname'      => $this->get_blogname(),
			'site_url'      => home_url(),
			'email_heading' => $this->get_heading(),
			'sent_to_admin' => false,
			'plain_text'    => false,
			'email'         => $this,
				) ;
	}

	/**
	 * Get content HTMl.
	 *
	 * @return string
	 */
	public function get_content_html() {
		ob_start() ;
		_qts_get_template( $this->template_html, $this->get_content_args() ) ;
		return ob_get_clean() ;
	}

	/**
	 * Get content plain.
	 *
	 * @return string
	 */
	public function get_content_plain() {
		return '' ;
	}

	/**
	 * Display form fields
	 */
	public function init_form_fields() {
		$this->form_fields              = array() ;
		$this->form_fields[ 'enabled' ] = array(
			'title'   => __( 'Enable/Disable', 'quote-request-for-woocommerce' ),
			'type'    => 'checkbox',
			'label'   => __( 'Enable this email notification', 'quote-request-for-woocommerce' ),
			'default' => 'yes'
				) ;

		if ( $this->supports( 'recipient' ) ) {
			$this->form_fields[ 'recipient' ] = array(
				'title'       => __( 'Recipient(s)', 'quote-request-for-woocommerce' ),
				'type'        => 'text',
				/* translators: 1: admin email */
				'description' => sprintf( __( 'Enter recipients (comma separated) for this email. Defaults to %1$s.', 'quote-request-for-woocommerce' ), '<code>' . esc_attr( get_option( 'admin_email' ) ) . '</code>' ),
				'placeholder' => '',
				'default'     => '',
				'desc_tip'    => true,
					) ;
		}

		$this->form_fields[ 'subject' ] = array(
			'title'       => __( 'Email Subject', 'quote-request-for-woocommerce' ),
			'type'        => 'text',
			/* translators: 1: email subject */
			'description' => sprintf( __( 'Defaults to <code>%1$s</code>', 'quote-request-for-woocommerce' ), $this->subject ),
			'placeholder' => '',
			'default'     => ''
				) ;

		$this->form_fields[ 'heading' ] = array(
			'title'       => __( 'Email Heading', 'quote-request-for-woocommerce' ),
			'type'        => 'text',
			/* translators: 1: email heading */
			'description' => sprintf( __( 'Defaults to <code>%1$s</code>', 'quote-request-for-woocommerce' ), $this->heading ),
			'placeholder' => '',
			'default'     => ''
				) ;

		if ( $this->supports( 'mail_to_admin' ) ) {
			$this->form_fields[ 'mail_to_admin' ] = array(
				'title'   => __( 'Send Email to Admin', 'quote-request-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no'
					) ;
		}
	}

}
