<?php
defined( 'ABSPATH' ) || exit ;

/**
 * Handle Admin metaboxes.
 * 
 * @class QTS_Admin_Metaboxes
 * @package Class
 */
class QTS_Admin_Metaboxes {

	/**
	 * Is meta boxes saved once?
	 *
	 * @var bool
	 */
	private static $saved_meta_boxes = false ;

	/**
	 * Meta box error messages.
	 *
	 * @var array
	 */
	protected static $meta_box_errors = array() ;

	/**
	 * Get our registered order type.
	 * 
	 * @var string 
	 */
	protected $our_order_type = 'qts_quote_request' ;

	/**
	 * QTS_Admin_Metaboxes constructor.
	 */
	public function __construct() {
		add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ), 50 ) ;
		add_action( 'add_meta_boxes', array( $this, 'remove_meta_boxes' ), 50 ) ;
		add_action( 'save_post', array( $this, 'save_meta_boxes' ), 5, 2 ) ;

		add_action( 'woocommerce_process_shop_order_meta', array( $this, 'remove_meta_box_save' ), -5, 2 ) ;
		add_action( 'woocommerce_process_shop_order_meta', 'QTS_Meta_Box_Quote_Request_Data::save', 20, 2 ) ;
		add_action( 'woocommerce_process_shop_order_meta', 'QTS_Meta_Box_Quote_Request_Actions::save', 30, 2 ) ;
		add_action( 'qts_process_form_field_meta', 'QTS_Meta_Box_Form_Field_Actions::save', 10, 3 ) ;
		add_action( 'woocommerce_before_save_order_item', array( $this, 'save_quote_request_item' ) ) ;
		add_action( 'woocommerce_saved_order_items', array( $this, 'update_quote_request_totals' ), 10, 2 ) ;

		add_filter( 'woocommerce_hidden_order_itemmeta', array( $this, 'add_hidden_itemmeta' ) ) ;
		add_action( 'woocommerce_admin_order_item_headers', array( $this, 'add_item_columns' ) ) ;
		add_action( 'woocommerce_admin_order_item_values', array( $this, 'add_item_values' ), 10, 2 ) ;
		add_action( 'woocommerce_order_item_add_action_buttons', array( $this, 'add_non_editable_desc' ) ) ;

		// Error handling
		add_action( 'admin_notices', array( $this, 'output_errors' ) ) ;
		add_action( 'shutdown', array( $this, 'save_errors' ) ) ;
	}

	/**
	 * Add QTS Metaboxes.
	 */
	public function add_meta_boxes() {
		$order_type_object = get_post_type_object( $this->our_order_type ) ;
		/* Translators: %s order type name. */
		add_meta_box( 'woocommerce-order-data', sprintf( __( '%s data', 'quote-request-for-woocommerce' ), $order_type_object->labels->singular_name ), 'QTS_Meta_Box_Quote_Request_Data::output', $this->our_order_type, 'normal', 'high' ) ;
		/* Translators: %s order type name. */
		add_meta_box( 'woocommerce-order-actions', sprintf( __( '%s actions', 'quote-request-for-woocommerce' ), $order_type_object->labels->singular_name ), 'QTS_Meta_Box_Quote_Request_Actions::output', $this->our_order_type, 'side', 'high' ) ;
		/* Translators: %s order type name. */
		add_meta_box( 'woocommerce-order-notes', sprintf( __( '%s notes', 'quote-request-for-woocommerce' ), $order_type_object->labels->singular_name ), 'QTS_Meta_Box_Quote_Request_Notes::output', $this->our_order_type, 'side', 'default' ) ;
		/* Translators: %s order type name. */
		add_meta_box( 'qts-quote-request-conversation-notes', sprintf( __( '%s conversations', 'quote-request-for-woocommerce' ), $order_type_object->labels->singular_name ), 'QTS_Meta_Box_Quote_Request_Conversations::output', $this->our_order_type, 'normal', 'low' ) ;

		add_meta_box( QTS_PREFIX . 'form_field_data', __( 'Form Field Data', 'quote-request-for-woocommerce' ), 'QTS_Meta_Box_Form_Field_Data::output', 'qts_form_field', 'normal', 'high' ) ;
		add_meta_box( QTS_PREFIX . 'form_field_actions', __( 'Form Field Actions', 'quote-request-for-woocommerce' ), 'QTS_Meta_Box_Form_Field_Actions::output', 'qts_form_field', 'side', 'high' ) ;
	}

	/**
	 * Remove Metaboxes.
	 */
	public function remove_meta_boxes() {
		remove_meta_box( 'woocommerce-order-downloads', $this->our_order_type, 'normal' ) ;
		remove_meta_box( 'commentsdiv', $this->our_order_type, 'normal' ) ;
		remove_meta_box( 'submitdiv', $this->our_order_type, 'side' ) ;
		remove_meta_box( 'commentsdiv', 'qts_form_field', 'normal' ) ;
		remove_meta_box( 'submitdiv', 'qts_form_field', 'side' ) ;
	}

	/**
	 * Remove meta box data save on order. 
	 * Do not save order data since we have already registered our post type as order type.
	 *
	 * @param int $post_id
	 * @param WP_Post $post
	 */
	public function remove_meta_box_save( $post_id, $post ) {
		if ( $this->our_order_type === $post->post_type ) {
			remove_action( 'woocommerce_process_shop_order_meta', 'WC_Meta_Box_Order_Data::save', 40 ) ;
			remove_action( 'woocommerce_process_shop_order_meta', 'WC_Meta_Box_Order_Actions::save', 50 ) ;
		}
	}

	/**
	 * Add an error message.
	 *
	 * @param string $text Error to add.
	 */
	public static function add_error( $text ) {
		self::$meta_box_errors[] = $text ;
	}

	/**
	 * Save errors to an option.
	 */
	public function save_errors() {
		update_option( 'qts_meta_box_errors', self::$meta_box_errors ) ;
	}

	/**
	 * Show any stored error messages.
	 */
	public function output_errors() {
		$errors = array_filter( ( array ) get_option( 'qts_meta_box_errors' ) ) ;

		if ( ! empty( $errors ) ) {
			echo '<div id="woocommerce_errors" class="error notice is-dismissible">' ;
			foreach ( $errors as $error ) {
				echo '<p>' . wp_kses_post( $error ) . '</p>' ;
			}
			echo '</div>' ;

			// Clear.
			delete_option( 'qts_meta_box_errors' ) ;
		}
	}

	/**
	 * Save the quote request item.
	 * 
	 * @param WC_Order_Item $item
	 */
	public function save_quote_request_item( $item ) {
		global $thequote_request ;

		$posted = $_REQUEST ;
		if ( ! current_user_can( 'edit_shop_orders' ) || ! isset( $posted[ 'items' ] ) ) {
			return ;
		}

		$item_id = $item->get_id() ;
		$items   = array() ;
		parse_str( wp_unslash( $posted[ 'items' ] ), $items ) ;

		if ( 'line_item' === $item->get_type() && isset( $items[ 'qts_offered_price' ][ $item_id ] ) && is_numeric( $items[ 'qts_offered_price' ][ $item_id ] ) ) {
			if ( ! is_object( $thequote_request ) ) {
				$thequote_request = _qts_get_quote_request( $item->get_order_id() ) ;
			}

			if ( ! _qts_is_quote_request( $thequote_request ) ) {
				return ;
			}

			$offered_price = $items[ 'qts_offered_price' ][ $item_id ] ;
			$total         = $offered_price * $item->get_quantity() ;

			$item->update_meta_data( '_offered_price', $offered_price ) ;
			$item->set_subtotal( $total ) ;
			$item->set_total( $total ) ;
			$item->save() ;
		}
	}

	/**
	 * Update quote request totals.
	 * 
	 * @param int $quote_request_id
	 * @param array $items Quote Request items to save.
	 */
	public function update_quote_request_totals( $quote_request_id, $items ) {
		if ( ! isset( $items[ 'qts_offered_price' ] ) ) {
			return ;
		}

		$quote_request = _qts_get_quote_request( $quote_request_id ) ;

		if ( $quote_request ) {
			$quote_request->calculate_totals() ;
		}
	}

	/**
	 * Add our itemmeta which are belongs to hidden.
	 * 
	 * @param array $itemmeta
	 * @return array
	 */
	public function add_hidden_itemmeta( $itemmeta ) {
		$itemmeta[] = '_is_qts_quote_request' ;
		$itemmeta[] = '_original_price' ;
		$itemmeta[] = '_requested_price' ;
		$itemmeta[] = '_offered_price' ;
		return $itemmeta ;
	}

	/**
	 * Render our columns to print the original price, requested price and offered price HTML field
	 * 
	 * @param QTS_Quote_Request $quote_request
	 */
	public function add_item_columns( $quote_request ) {
		if ( _qts_is_quote_request( $quote_request ) ) {
			?>
			<th class="original-price sortable" data-sort="float"><?php esc_html_e( 'Original Price', 'quote-request-for-woocommerce' ) ; ?></th>
			<th class="requested-price sortable" data-sort="float"><?php esc_html_e( 'Requested Price', 'quote-request-for-woocommerce' ) ; ?></th>
			<th class="offered-price sortable" data-sort="float"><?php esc_html_e( 'Offered Price', 'quote-request-for-woocommerce' ) ; ?></th>
			<?php
		}
	}

	/**
	 * Render original price, requested price and offered price HTML field belongs to the quote request item.
	 * 
	 * @param WC_Product $product
	 * @param WC_Order_Item $item
	 */
	public function add_item_values( $product, $item ) {
		global $thequote_request ;

		if ( 'line_item' === $item->get_type() ) {
			if ( ! is_object( $thequote_request ) ) {
				$thequote_request = _qts_get_quote_request( $item->get_order_id() ) ;
			}

			if ( _qts_is_quote_request( $thequote_request ) ) {
				$has_offered_price = is_numeric( $item->get_meta( '_offered_price' ) ) ;
				$offered_price     = $item->get_meta( '_offered_price' ) ;
				?>
				<td class="item_original_price" width="1%" data-sort-value="<?php echo esc_attr( $item->get_meta( '_original_price' ) ) ; ?>">
					<div class="qts-view-non-editable">
						<?php
						echo wp_kses_post( wc_price( $item->get_meta( '_original_price' ), array( 'currency' => $thequote_request->get_currency() ) ) ) ;
						?>
					</div>
				</td>
				<td class="item_requested_price" width="1%" data-sort-value="<?php echo esc_attr( $item->get_meta( '_requested_price' ) ) ; ?>">
					<div class="qts-view-non-editable">
						<?php
						echo is_numeric( $item->get_meta( '_requested_price' ) ) ? wp_kses_post( wc_price( $item->get_meta( '_requested_price' ), array( 'currency' => $thequote_request->get_currency() ) ) ) : '-' ;
						?>
					</div>
				</td>
				<td class="item_offered_price" width="1%" data-sort-value="<?php echo esc_attr( $offered_price ) ; ?>">
					<div class="view">
						<?php
						echo $has_offered_price ? wp_kses_post( wc_price( $offered_price, array( 'currency' => $thequote_request->get_currency() ) ) ) : '-' ;
						?>
					</div>
					<div class="edit" style="display: none;">
						<input type="text" name="qts_offered_price[<?php echo absint( $item->get_id() ) ; ?>]" placeholder="<?php echo esc_attr( wc_format_localized_price( $item->get_meta( '_requested_price' ) ) ) ; ?>" value="<?php echo $has_offered_price ? esc_attr( wc_format_localized_price( $offered_price ) ) : '' ; ?>" class="wc_input_price" data-tip="<?php esc_attr_e( 'Your offer price.', 'quote-request-for-woocommerce' ) ; ?>" data-offered_price="<?php echo $has_offered_price ? esc_attr( wc_format_localized_price( $offered_price ) ) : '' ; ?>" />
					</div>
				</td>
				<?php
			}
		}
	}

	/**
	 * Render description to say quote request is non editable to update the quote request items, based upon status.
	 * 
	 * @param QTS_Quote_Request $quote_request
	 */
	public function add_non_editable_desc( $quote_request ) {
		if ( _qts_is_quote_request( $quote_request ) && ! $quote_request->is_editable() ) {
			?>
			<span class="qts-description"><?php echo wc_help_tip( __( 'Valid quote request statuses to edit are "New","On-hold,"Response Awaited" and "Have to Respond"', 'quote-request-for-woocommerce' ) ) ; ?> <?php esc_html_e( 'This quote request is no longer editable.', 'quote-request-for-woocommerce' ) ; ?></span>
			<?php
		}
	}

	/**
	 * Check if we're saving, the trigger an action based on the post type.
	 *
	 * @param  int    $post_id Post ID.
	 * @param  object $post Post object.
	 */
	public function save_meta_boxes( $post_id, $post ) {
		$post_id = absint( $post_id ) ;

		// $post_id and $post are required
		if ( empty( $post_id ) || empty( $post ) || self::$saved_meta_boxes ) {
			return ;
		}

		// Dont' save meta boxes for revisions or autosaves.
		if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || is_int( wp_is_post_revision( $post ) ) || is_int( wp_is_post_autosave( $post ) ) ) {
			return ;
		}

		// Check the nonce.
		if ( empty( $_POST[ 'qts_save_meta_nonce' ] ) || ! wp_verify_nonce( sanitize_key( wp_unslash( $_POST[ 'qts_save_meta_nonce' ] ) ), 'qts_save_data' ) ) {
			return ;
		}

		// Check the post being saved == the $post_id to prevent triggering this call for other save_post events.
		$posted = $_POST ;
		if ( empty( $posted[ 'post_ID' ] ) || absint( $posted[ 'post_ID' ] ) !== $post_id ) {
			return ;
		}

		// Check user has permission to edit.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return ;
		}

		self::$saved_meta_boxes = true ;

		if ( in_array( $post->post_type, array( 'qts_form_field' ), true ) ) {
			do_action( 'qts_process_form_field_meta', $post_id, $post, $posted ) ;
		}
	}

}

new QTS_Admin_Metaboxes() ;
