<?php

defined( 'ABSPATH' ) || exit ;

/**
 * Check whether the queue is available.
 * 
 * @param QTS_Background_Process::$group_name $name
 * @return bool true on Success
 */
function _qts_is_queue_available( $name ) {
	if ( is_null( _qts()->queue ) ) {
		return false ;
	}

	if ( ! $name || ! isset( _qts()->queue[ $name ] ) ) {
		return false ;
	}

	return true ;
}

/**
 * Return the queue if available.
 * 
 * @param QTS_Background_Process::$group_name $name
 * @return bool|QTS_Queue false on Failure
 */
function _qts_get_queue( $name ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	return _qts()->queue[ $name ] ;
}

/**
 * Format the given datetime in strtotime.
 * 
 * @param string|int $date
 * @return int
 */
function _qts_format_queue_schedule_time( $date ) {
	return _qts_maybe_strtotime( $date ) ;
}

/**
 * Check whether the job scheduled under the group for given datetime in the queue.
 * 
 * @param QTS_Background_Process::$group_name $name
 * @param string $job_name Should be unique and present in array of QTS_Queue::$jobs
 * @param string|int $scheduled_on
 * @param int $parent_id Parent ID of the group. Want to check the job scheduled under the Parent ID ?
 * @return bool|int false on Failure
 */
function _qts_job_exists_in_queue( $name, $job_name, $scheduled_on = null, $parent_id = 0 ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	_qts()->queue[ $name ]->set_parent_id( $parent_id ) ;

	if ( is_null( $scheduled_on ) ) {
		return _qts()->queue[ $name ]->exists( $job_name ) ;
	} else {
		return _qts()->queue[ $name ]->exists( $job_name, _qts_format_queue_schedule_time( $scheduled_on ) ) ;
	}
}

/**
 * Push/Schedule the job under the group in the queue.
 * 
 * @param QTS_Background_Process::$group_name $name Should be unique
 * @param string $job_name Should be unique and present in array of QTS_Queue::$jobs
 * @param string|int $schedule_on
 * @param int $parent_id Parent ID of the group. Want to schedule the job under the Parent ID ? Which will be easy for mapping the jobs you have scheduled for.
 * @param array $args Additional arguments which are passed on to the job hook's callback function
 * @return bool|int false on Failure
 */
function _qts_push_job_to_queue( $name, $job_name, $schedule_on, $parent_id = 0, $args = array() ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	_qts()->queue[ $name ]->set_parent_id( $parent_id ) ;

	return _qts()->queue[ $name ]->push( $job_name, _qts_format_queue_schedule_time( $schedule_on ), null, $args ) ;
}

/**
 * Push/Schedule the recurring job under the group in the queue.
 *
 * @param QTS_Background_Process::$group_name $name Should be unique
 * @param string $job_name Should be unique and present in array of QTS_Queue::$jobs
 * @param string|int $schedule_on
 * @param string $recurrence How often the job should subsequently recur. See _qts_get_recurring_job_schedules() for accepted values.
 * @param array $args Additional arguments which are passed on to the job hook's callback function
 * @return bool|int false on Failure
 */
function _qts_push_recurring_job_to_queue( $name, $job_name, $schedule_on, $recurrence, $args = array() ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	$schedules = _qts_get_recurring_job_schedules() ;

	if ( empty( $schedules ) || ! isset( $schedules[ $recurrence ] ) ) {
		return false ;
	}

	_qts()->queue[ $name ]->set_parent_id( 0 ) ;

	return _qts()->queue[ $name ]->push( $job_name, _qts_format_queue_schedule_time( $schedule_on ), $recurrence, $args ) ;
}

/**
 * Return the array of jobs scheduled under the group in the queue.
 * 
 * @param QTS_Background_Process::$group_name $name
 * @return bool|array false on Failure
 */
function _qts_get_jobs_from_queue( $name ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	return _qts()->queue[ $name ]->get_queue() ;
}

/**
 * Return the array of job's scheduled datetime under the job name meant for the group in the queue.
 * 
 * @param QTS_Background_Process::$group_name $name
 * @param string $job_name If present in array of QTS_Queue::$jobs
 * @param int $parent_id Parent ID of the group. Want to get the job's scheduled datetime under the Parent ID ?
 * @param bool $single Want to return single job's scheduled datetime from the array ?
 * @return bool|array false on Failure
 */
function _qts_get_job_scheduled( $name, $job_name, $parent_id = 0, $single = true ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	_qts()->queue[ $name ]->set_parent_id( $parent_id ) ;

	$scheduled = _qts()->queue[ $name ]->get( $job_name ) ;

	if ( $single && is_array( $scheduled ) ) {
		reset( $scheduled ) ;
		$scheduled = current( $scheduled ) ;
	}

	return $scheduled ;
}

/**
 * Cancel the job under the group from the queue.
 * 
 * @param QTS_Background_Process::$group_name $name
 * @param string $job_name If present in array of QTS_Queue::$jobs
 * @param int $parent_id Parent ID of the group. Want to cancel the job under the Parent ID ?
 * @return bool false on Failure
 */
function _qts_cancel_job_from_queue( $name, $job_name, $parent_id = 0 ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	_qts()->queue[ $name ]->set_parent_id( $parent_id ) ;

	return _qts()->queue[ $name ]->delete( $job_name ) ;
}

/**
 * Cancel every jobs under the group from the queue.
 * 
 * @param QTS_Background_Process::$group_name $name
 * @param int $parent_id Parent ID of the group. Want to cancel the jobs under the Parent ID ?
 * @return bool false on Failure
 */
function _qts_cancel_all_jobs_from_queue( $name, $parent_id = 0 ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	_qts()->queue[ $name ]->set_parent_id( $parent_id ) ;

	return _qts()->queue[ $name ]->delete_all() ;
}

/**
 * Cancel the queue process of the group on demand.
 * 
 * @param QTS_Background_Process::$group_name $name
 * @return bool false on Failure
 */
function _qts_cancel_queue_process( $name ) {
	if ( ! _qts_is_queue_available( $name ) ) {
		return false ;
	}

	_qts()->queue[ $name ]->cancel_process() ;
}

/**
 * Cancel every queue process on demand.
 * 
 * @return bool false on Failure
 */
function _qts_cancel_all_queue_process() {
	if ( empty( _qts()->queue ) ) {
		return false ;
	}

	array_map( '_qts_cancel_queue_process', array_keys( _qts()->queue ) ) ;
}

/**
 * Retrieve supported recurrence job schedules.
 * 
 * @return array
 */
function _qts_get_recurring_job_schedules() {
	$schedules = array(
		'5mins' => array(
			'interval' => 300,
			'display'  => 'Every 5 Minutes',
		),
			) ;

	return array_merge( apply_filters( 'qts_get_recurring_job_schedules', array() ), $schedules ) ;
}
