<?php
/**
 * Our Admin Functions
 */
defined( 'ABSPATH' ) || exit ;

/**
 * Get our screen ids.
 *
 * @return array
 */
function _qts_get_screen_ids() {
	$qts_screen_id = sanitize_title( __( 'Quote Requests', 'quote-request-for-woocommerce' ) ) ;
	$screen_ids    = array(
		'qts_quote_request',
		'qts_form_field',
		'edit-qts_quote_request',
		'edit-qts_form_field',
		$qts_screen_id . '_page_qts_settings',
			) ;

	return apply_filters( 'qts_screen_ids', $screen_ids ) ;
}

/**
 * Create a page and store the ID in an option.
 *
 * @param mixed  $slug Slug for the new page.
 * @param string $option Option name to store the page's ID.
 * @param string $page_title (default: '') Title for the new page.
 * @param string $page_content (default: '') Content for the new page.
 * @param int    $post_parent (default: 0) Parent for the new page.
 * @return int page ID.
 */
function _qts_create_page( $slug, $option = '', $page_title = '', $page_content = '', $post_parent = 0 ) {
	$option_value = '' !== $option ? get_option( $option ) : 0 ;

	if ( $option_value > 0 ) {
		$page_object = get_post( $option_value ) ;

		if ( $page_object && 'page' === $page_object->post_type && ! in_array( $page_object->post_status, array( 'pending', 'trash', 'future', 'auto-draft' ), true ) ) {
			// Valid page is already in place.
			return $page_object->ID ;
		}
	}

	$page_data = array(
		'post_status'    => 'publish',
		'post_type'      => 'page',
		'post_author'    => 1,
		'post_name'      => $slug,
		'post_title'     => $page_title,
		'post_content'   => $page_content,
		'post_parent'    => $post_parent,
		'comment_status' => 'closed',
			) ;

	$page_id = wp_insert_post( $page_data ) ;

	if ( $option ) {
		update_option( $option, $page_id ) ;
	}

	return $page_id ;
}

/**
 * Get WC search field
 * 
 * @param array $args
 * @param bool $echo
 * @return string echo search field
 */
function _qts_search_field( $args = array(), $echo = true ) {

	$args = wp_parse_args( $args, array(
		'class'       => '',
		'id'          => '',
		'name'        => '',
		'type'        => '',
		'action'      => '',
		'placeholder' => '',
		'css'         => 'width: 50%;',
		'multiple'    => true,
		'allow_clear' => true,
		'selected'    => true,
		'options'     => array()
			) ) ;

	ob_start() ;
	?>
	<select 
		id="<?php echo esc_attr( $args[ 'id' ] ) ; ?>" 
		class="<?php echo esc_attr( $args[ 'class' ] ) ; ?>" 
		name="<?php echo esc_attr( '' !== $args[ 'name' ] ? $args[ 'name' ] : $args[ 'id' ]  ) ; ?><?php echo ( $args[ 'multiple' ] ) ? '[]' : '' ; ?>" 
		data-action="<?php echo esc_attr( $args[ 'action' ] ) ; ?>" 
		data-placeholder="<?php echo esc_attr( $args[ 'placeholder' ] ) ; ?>" 
		<?php echo ( $args[ 'allow_clear' ] ) ? 'data-allow_clear="true"' : '' ; ?> 
		<?php echo ( $args[ 'multiple' ] ) ? 'multiple="multiple"' : '' ; ?> 
		style="<?php echo esc_attr( $args[ 'css' ] ) ; ?>">
			<?php
			if ( ! is_array( $args[ 'options' ] ) ) {
				$args[ 'options' ] = ( array ) $args[ 'options' ] ;
			}

			$args[ 'options' ] = array_filter( $args[ 'options' ] ) ;

			foreach ( $args[ 'options' ] as $id ) {
				$option_value = '' ;

				switch ( $args[ 'type' ] ) {
					case 'product':
						$product = wc_get_product( $id ) ;
						if ( $product ) {
							$option_value = wp_kses_post( $product->get_formatted_name() ) ;
						}
						break ;
					case 'customer':
						$user = get_user_by( 'id', $id ) ;
						if ( $user ) {
							$option_value = ( esc_html( $user->display_name ) . '(#' . absint( $user->ID ) . ' &ndash; ' . esc_html( $user->user_email ) . ')' ) ;
						}
						break ;
					default:
						$post = get_post( $id ) ;
						if ( $post ) {
							$option_value = sprintf( '(#%s) %s', $post->ID, wp_kses_post( $post->post_title ) ) ;
						}
						break ;
				}

				if ( $option_value ) {
					?>
				<option value="<?php echo esc_attr( $id ) ; ?>" <?php echo ( $args[ 'selected' ] ) ? 'selected="selected"' : '' ; ?>><?php echo esc_html( $option_value ) ; ?></option>
					<?php
				}
			}
			?>
	</select>
	<?php
	if ( $echo ) {
		ob_end_flush() ;
	} else {
		return ob_get_clean() ;
	}
}

/**
 * Return the array of categories for the products.
 * 
 * @return array
 */
function _qts_get_product_categories() {
	$categories   = array() ;
	$categoryid   = array() ;
	$categoryname = array() ;

	$listcategories = get_terms( 'product_cat' ) ;
	if ( is_array( $listcategories ) ) {
		foreach ( $listcategories as $category ) {
			$categoryname[] = $category->name ;
			$categoryid[]   = $category->term_id ;
		}
	}

	if ( $categoryid && $categoryname ) {
		$categories = array_combine( ( array ) $categoryid, ( array ) $categoryname ) ;
	}

	return $categories ;
}
