<?php

defined( 'ABSPATH' ) || exit ;

/**
 * Handle Quote Request for WooCommerce Ajax Event.
 * 
 * @class QTS_Ajax
 * @package Class
 */
class QTS_Ajax {

	/**
	 * Init QTS_Ajax.
	 */
	public static function init() {
		//Get Ajax Events.
		$prefix      = QTS_PREFIX ;
		$ajax_events = array(
			'quote_mode_selected'            => true,
			'add_to_quote'                   => true,
			'save_customer_note'             => true,
			'add_quote_request_private_note' => false,
			'send_note_to_customer'          => false,
			'send_note_to_admin'             => false,
			'post_ordering'                  => false,
				) ;

		foreach ( $ajax_events as $ajax_event => $nopriv ) {
			add_action( "wp_ajax_{$prefix}{$ajax_event}", __CLASS__ . "::{$ajax_event}" ) ;

			if ( $nopriv ) {
				add_action( "wp_ajax_nopriv_{$prefix}{$ajax_event}", __CLASS__ . "::{$ajax_event}" ) ;
			}
		}

		// Hook in WC Ajax methods on demand.
		add_action( 'wp_ajax_woocommerce_add_order_item', __CLASS__ . '::exclude_quote_request_conversations', 9 ) ;
		add_action( 'wp_ajax_woocommerce_remove_order_item', __CLASS__ . '::exclude_quote_request_conversations', 9 ) ;
		add_action( 'wp_ajax_woocommerce_save_order_items', __CLASS__ . '::exclude_quote_request_conversations', 9 ) ;
		add_action( 'wp_ajax_woocommerce_add_order_item', __CLASS__ . '::include_quote_request_conversations', 11 ) ;
		add_action( 'wp_ajax_woocommerce_remove_order_item', __CLASS__ . '::include_quote_request_conversations', 11 ) ;
		add_action( 'wp_ajax_woocommerce_save_order_items', __CLASS__ . '::include_quote_request_conversations', 11 ) ;
	}

	/**
	 * When the quote mode is selected.
	 */
	public static function quote_mode_selected() {
		check_ajax_referer( 'qts-quote-mode-selected', 'security' ) ;

		try {
			if ( ! isset( $_POST[ 'product_id' ] ) || ! isset( $_POST[ 'chosen_mode' ] ) ) {
				throw new Exception( __( 'Invalid response', 'quote-request-for-woocommerce' ) ) ;
			}

			$product = wc_get_product( absint( wp_unslash( $_POST[ 'product_id' ] ) ) ) ;

			if ( ! $product ) {
				throw new Exception( __( 'Invalid product to add.', 'quote-request-for-woocommerce' ) ) ;
			}

			ob_start() ;
			_qts_get_template( 'add-to-quote.php', QTS_Add_To_Quote::prepare_add_to_quote_args( $product, array(
						'mode'                => sanitize_text_field( wp_unslash( $_POST[ 'chosen_mode' ] ) ),
						'show_user_options'   => 'user-choice' === QTS_Add_To_Quote::get_mode(),
						'ajax_add_to_quote'   => false,
						'single_add_to_quote' => true,
			) ) ) ;
			$html = ob_get_clean() ;

			wp_send_json_success( array(
				'html' => $html,
			) ) ;
		} catch ( Exception $e ) {
			wp_send_json_error( array( 'error' => esc_html( $e->getMessage() ) ) ) ;
		}
	}

	/**
	 * Add the product to either in Quote or Cart page based upon the settings.
	 */
	public static function add_to_quote() {
		check_ajax_referer( 'qts-add-to-quote', 'security' ) ;

		try {
			if ( ! isset( $_POST[ 'product_id' ] ) || ! isset( $_POST[ 'data' ] ) ) {
				throw new Exception( __( 'Invalid response', 'quote-request-for-woocommerce' ) ) ;
			}

			$data                      = wp_parse_args( wc_clean( wp_unslash( $_POST[ 'data' ] ) ) ) ;
			$quantity                  = ! empty( $data[ 'quantity' ] ) ? absint( $data[ 'quantity' ] ) : 1 ;
			$quote_mode_chosen_by_user = ! empty( $data[ 'add-to-qtsquote-mode' ] ) ? true : false ;
			$adding_quote_mode         = $quote_mode_chosen_by_user ? $data[ 'add-to-qtsquote-mode' ] : QTS_Add_To_Quote::get_mode() ;
			$adding_product_id         = ! empty( $data[ 'variation_id' ] ) ? absint( $data[ 'variation_id' ] ) : absint( wp_unslash( $_POST[ 'product_id' ] ) ) ;
			$product                   = wc_get_product( $adding_product_id ) ;

			if ( ! $product ) {
				throw new Exception( __( 'Invalid product to add', 'quote-request-for-woocommerce' ) ) ;
			}

			if ( $quote_mode_chosen_by_user ) {
				if ( '' !== QTS_Add_To_Quote::get_mode( true ) && QTS_Add_To_Quote::get_mode( true ) !== $adding_quote_mode ) {
					_qts_empty_quote( 'approx' ) ;
					_qts_empty_quote( 'accurate' ) ;
				}

				WC()->session->set( QTS_PREFIX . 'add_to_quote_chosen_mode', $adding_quote_mode ) ;
			} else {
				WC()->session->set( QTS_PREFIX . 'add_to_quote_chosen_mode', '' ) ;
			}

			if ( isset( $_POST[ 'ajax_add_to_quote' ] ) ) {
				$quote_added = false ;

				if ( 'approx' === $adding_quote_mode ) {
					$passed_validation = apply_filters( 'qts_add_to_quote_validation', true, $adding_product_id, $quantity ) ;

					if ( $passed_validation ) {
						$quote_added = _qts()->quote->add_to_quote( $adding_product_id, $quantity ) ;
					}
				} else if ( 'accurate' === $adding_quote_mode ) {
					$passed_validation = apply_filters( 'woocommerce_add_to_cart_validation', true, $adding_product_id, $quantity ) ;

					if ( $passed_validation ) {
						$quote_added = WC()->cart->add_to_cart( $adding_product_id, $quantity ) ;
					}
				}

				if ( false === $quote_added ) {
					WC()->session->set( QTS_PREFIX . 'add_to_quote_chosen_mode', '' ) ;
					wp_send_json_error( array(
						'error'        => esc_html__( 'Something went wrong while adding the quote.', 'quote-request-for-woocommerce' ),
						'redirect_url' => get_permalink( $adding_product_id )
					) ) ;
				}

				if ( ! $quote_mode_chosen_by_user ) {
					if ( 'approx' === $adding_quote_mode ) {
						_qts_empty_quote( 'accurate' ) ;
					} else if ( 'accurate' === $adding_quote_mode ) {
						_qts_empty_quote( 'approx' ) ;
					}
				}

				if ( 'yes' !== get_option( QTS_PREFIX . 'quote_redirect_after_add' ) ) {
					wp_send_json_success( array(
						'view_quote_url' => _qts_get_quote_url( $adding_quote_mode )
					) ) ;
				} else {
					wp_send_json_success( array(
						'redirect_url' => _qts_get_quote_url( $adding_quote_mode )
					) ) ;
				}
			} else {
				$add_to_quote_url = _qts_add_to_quote_url( $product, $adding_quote_mode ) ;

				if ( '' === $add_to_quote_url ) {
					throw new Exception( __( 'Something went wrong while preparing to add the quote.', 'quote-request-for-woocommerce' ) ) ;
				}

				wp_send_json_success( array(
					'redirect_url' => add_query_arg( 'quantity', $quantity, $add_to_quote_url )
				) ) ;
			}
		} catch ( Exception $e ) {
			WC()->session->set( QTS_PREFIX . 'add_to_quote_chosen_mode', '' ) ;
			wp_send_json_error( array( 'error' => esc_html( $e->getMessage() ) ) ) ;
		}
	}

	/**
	 * Save/Clear the customer note added in cart/checkout page before submitting the quote.
	 */
	public static function save_customer_note() {
		check_ajax_referer( 'qts-save-customer-note', 'security' ) ;

		try {
			if ( ! isset( $_POST[ 'note' ] ) || ! isset( $_POST[ 'saved' ] ) ) {
				throw new Exception( __( 'Invalid response', 'quote-request-for-woocommerce' ) ) ;
			}

			$saved = ( bool ) wc_clean( wp_unslash( $_POST[ 'saved' ] ) ) ;

			if ( $saved ) {
				$note = wp_kses_post( trim( sanitize_textarea_field( wp_unslash( $_POST[ 'note' ] ) ) ) ) ;
				WC()->session->set( QTS_PREFIX . 'customer_note', $note ) ;
				wp_send_json_success( array( 'note' => $note, 'message' => __( 'Saved!', 'quote-request-for-woocommerce' ) ) ) ;
			} else {
				WC()->session->set( QTS_PREFIX . 'customer_note', '' ) ;
				wp_send_json_success( array( 'message' => __( 'Cleared!', 'quote-request-for-woocommerce' ) ) ) ;
			}
		} catch ( Exception $e ) {
			WC()->session->set( QTS_PREFIX . 'customer_note', '' ) ;
			wp_send_json_error( array( 'error' => esc_html( $e->getMessage() ) ) ) ;
		}
	}

	/**
	 * Add quote request private note by admin.
	 */
	public static function add_quote_request_private_note() {
		check_ajax_referer( 'qts-add-quote-request-private-note', 'security' ) ;

		if ( ! current_user_can( 'edit_shop_orders' ) || ! isset( $_POST[ 'post_id' ], $_POST[ 'note' ] ) ) {
			wp_die( -1 ) ;
		}

		$post_id = absint( wp_unslash( $_POST[ 'post_id' ] ) ) ;
		$note    = wp_kses_post( trim( sanitize_textarea_field( wp_unslash( $_POST[ 'note' ] ) ) ) ) ;

		if ( $post_id > 0 ) {
			$quote_request = _qts_get_quote_request( $post_id ) ;
			$comment_id    = $quote_request->add_order_note( $note, false, true ) ;

			ob_start() ;
			$note      = _qts_get_quote_request_note( $comment_id ) ;
			include 'admin/meta-boxes/views/html-quote-request-note.php' ;
			$note_html = ob_get_clean() ;

			wp_send_json_success( array(
				'note_html' => $note_html,
			) ) ;
		}
		wp_die() ;
	}

	/**
	 * Send note to customer via Edit Quote Request page.
	 */
	public static function send_note_to_customer() {
		check_ajax_referer( 'qts-send-quote-request-note-to-customer', 'security' ) ;

		if ( ! current_user_can( 'edit_shop_orders' ) || ! isset( $_POST[ 'post_id' ], $_POST[ 'note' ] ) ) {
			wp_die( -1 ) ;
		}

		global $post ;
		$post_id = absint( wp_unslash( $_POST[ 'post_id' ] ) ) ;
		$note    = wp_kses_post( trim( sanitize_textarea_field( wp_unslash( $_POST[ 'note' ] ) ) ) ) ;

		if ( $post_id > 0 ) {
			$quote_request_post = &$post ;
			$quote_request      = _qts_get_quote_request( $post_id ) ;

			if ( $note ) {
				$quote_request->update_status( 'response_awaitd' ) ;
			}

			$quote_request->add_order_note( $note, true, true ) ;

			// Return latest data.
			$quote_request      = _qts_get_quote_request( $post_id ) ;
			$quote_request_post = get_post( $post_id ) ;

			QTS_Meta_Box_Quote_Request_Data::init_address_fields() ;

			if ( WC()->payment_gateways() ) {
				$payment_gateways = WC()->payment_gateways->payment_gateways() ;
			} else {
				$payment_gateways = array() ;
			}

			$payment_method  = $quote_request->get_payment_method() ;
			$billing_fields  = QTS_Meta_Box_Quote_Request_Data::get_address_fields( 'billing' ) ;
			$shipping_fields = QTS_Meta_Box_Quote_Request_Data::get_address_fields( 'shipping' ) ;

			// Get HTML to return.
			ob_start() ;
			include 'admin/meta-boxes/views/html-quote-request-data.php' ;
			$html = ob_get_clean() ;

			ob_start() ;
			$notes      = $quote_request->get_quote_request_notes() ;
			include 'admin/meta-boxes/views/html-quote-request-notes.php' ;
			$notes_html = ob_get_clean() ;

			ob_start() ;
			$notes                   = $quote_request->get_conversation_notes() ;
			include 'admin/meta-boxes/views/html-quote-request-conversation-notes.php' ;
			$conversation_notes_html = ob_get_clean() ;

			wp_send_json_success( array(
				'html'                    => $html,
				'notes_html'              => $notes_html,
				'conversation_notes_html' => $conversation_notes_html
			) ) ;
		}
		wp_die() ;
	}

	/**
	 * Send note to admin via My Quote Requests > View Quote Request page.
	 */
	public static function send_note_to_admin() {
		check_ajax_referer( 'qts-send-quote-request-note-to-admin', 'security' ) ;

		if ( ! isset( $_POST[ 'quote_request_id' ], $_POST[ 'note' ] ) ) {
			wp_die( -1 ) ;
		}

		$quote_request_id = absint( wc_clean( wp_unslash( $_POST[ 'quote_request_id' ] ) ) ) ;
		$note             = wp_kses_post( trim( sanitize_textarea_field( wp_unslash( $_POST[ 'note' ] ) ) ) ) ;

		if ( $quote_request_id > 0 ) {
			$quote_request = _qts_get_quote_request( $quote_request_id ) ;

			if ( $note && ! $quote_request->has_status( array( 'new', 'on_hold' ) ) ) {
				$quote_request->update_status( 'hav_to_respond' ) ;
			}

			$quote_request->add_order_note( $note, true, true ) ;

			// Return latest data.
			$quote_request = _qts_get_quote_request( $quote_request_id ) ;

			// Get HTML to return.
			ob_start() ;
			_qts_get_template( 'myaccount/view-quote-request.php', array(
				'quote_request'    => $quote_request,
				'quote_request_id' => $quote_request_id
			) ) ;
			$html = ob_get_clean() ;

			ob_start() ;
			$notes      = $quote_request->get_conversation_notes() ;
			_qts_get_template( 'myaccount/quote-request-conversation-notes.php', array( 'notes' => $notes ) ) ;
			$notes_html = ob_get_clean() ;

			wp_send_json_success( array(
				'html'       => $html,
				'notes_html' => $notes_html
			) ) ;
		}
		wp_die() ;
	}

	/**
	 * Ajax request handling for post ordering.
	 */
	public static function post_ordering() {
		global $wpdb ;

		$posted = $_REQUEST ;
		if ( ! isset( $posted[ 'id' ] ) ) {
			wp_die( -1 ) ;
		}

		$sorting_id  = absint( $posted[ 'id' ] ) ;
		$post_type   = get_post_type( $sorting_id ) ;
		$previd      = absint( isset( $posted[ 'previd' ] ) ? $posted[ 'previd' ] : 0 ) ;
		$nextid      = absint( isset( $posted[ 'nextid' ] ) ? $posted[ 'nextid' ] : 0 ) ;
		$menu_orders = wp_list_pluck( $wpdb->get_results( $wpdb->prepare( "SELECT ID, menu_order FROM {$wpdb->posts} WHERE post_type=%s ORDER BY menu_order ASC, post_title ASC", esc_sql( $post_type ) ) ), 'menu_order', 'ID' ) ;
		$index       = 0 ;

		foreach ( $menu_orders as $id => $menu_order ) {
			$id = absint( $id ) ;

			if ( $sorting_id === $id ) {
				continue ;
			}
			if ( $nextid === $id ) {
				$index ++ ;
			}
			$index ++ ;
			$menu_orders[ $id ] = $index ;
			$wpdb->update( $wpdb->posts, array( 'menu_order' => $index ), array( 'ID' => $id ) ) ;
		}

		if ( isset( $menu_orders[ $previd ] ) ) {
			$menu_orders[ $sorting_id ] = $menu_orders[ $previd ] + 1 ;
		} elseif ( isset( $menu_orders[ $nextid ] ) ) {
			$menu_orders[ $sorting_id ] = $menu_orders[ $nextid ] - 1 ;
		} else {
			$menu_orders[ $sorting_id ] = 0 ;
		}

		$wpdb->update( $wpdb->posts, array( 'menu_order' => $menu_orders[ $sorting_id ] ), array( 'ID' => $sorting_id ) ) ;
		wp_send_json( $menu_orders ) ;
	}

	/**
	 * Exclude quote request conversation comments on demand from queries and RSS.
	 */
	public static function exclude_quote_request_conversations() {
		add_filter( 'the_comments', '_qts_exclude_quote_request_conversations' ) ;
	}

	/**
	 * Include quote request conversation comments on demand from queries and RSS.
	 */
	public static function include_quote_request_conversations() {
		remove_filter( 'the_comments', '_qts_exclude_quote_request_conversations' ) ;
	}

}

QTS_Ajax::init() ;
