<?php

defined( 'ABSPATH' ) || exit ;

/**
 * Emails class.
 * 
 * @class QTS_Emails
 * @package Class
 */
class QTS_Emails {

	/**
	 * Email notification classes
	 *
	 * @var WC_Email[]
	 */
	protected $emails = array() ;

	/**
	 * Available email notification classes to load
	 * 
	 * @var WC_Email::id => WC_Email class
	 */
	protected $email_classes = array(
		'admin_new_quote_request'        => 'QTS_Email_Admin_New_Quote_Request',
		'admin_quote_accepted'           => 'QTS_Email_Admin_Quote_Accepted',
		'admin_quote_rejected'           => 'QTS_Email_Admin_Quote_Rejected',
		'admin_response_awaited'         => 'QTS_Email_Admin_Response_Awaited',
		'customer_quote_acknowledged'    => 'QTS_Email_Customer_Quote_Acknowledged',
		'customer_quote_approved'        => 'QTS_Email_Customer_Quote_Approved',
		'customer_quote_expired'         => 'QTS_Email_Customer_Quote_Expired',
		'customer_quote_rejected'        => 'QTS_Email_Customer_Quote_Rejected',
		'quote_completed'                => 'QTS_Email_Quote_Completed',
		'customer_response_reminder'     => 'QTS_Email_Customer_Response_Reminder',
		'customer_payment_reminder'      => 'QTS_Email_Customer_Payment_Reminder',
		'customer_response_awaited'      => 'QTS_Email_Customer_Response_Awaited',
		'customer_accepted_quote_to_pay' => 'QTS_Email_Customer_Accepted_Quote_To_Pay'
			) ;

	/**
	 * The single instance of the class
	 *
	 * @var QTS_Emails
	 */
	protected static $_instance = null ;

	/**
	 * Main QTS_Emails Instance.
	 * Ensures only one instance of QTS_Emails is loaded or can be loaded.
	 * 
	 * @return QTS_Emails Main instance
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self() ;
		}
		return self::$_instance ;
	}

	/**
	 * Init the email class hooks in all emails that can be sent.
	 */
	public function init() {
		add_filter( 'woocommerce_email_classes', array( $this, 'add_email_classes' ) ) ;
		add_filter( 'woocommerce_template_directory', array( $this, 'set_template_directory' ), 10, 2 ) ;
		add_action( 'admin_init', array( $this, 'hide_plain_text_template' ) ) ;

		// Email content hooks.
		add_action( 'woocommerce_email_qtsquote_request_details', array( $this, 'quote_request_details' ), 10, 4 ) ;

		// Email Send Ack.
		add_action( 'qts_email_sent', array( $this, 'email_sent' ) ) ;
		add_action( 'qts_email_failed_to_sent', array( $this, 'email_failed_to_sent' ) ) ;

		self::init_notifications() ;
	}

	/**
	 * Hook in all our emails to notify.
	 */
	public static function init_notifications() {
		$email_actions = apply_filters( 'qts_email_actions', array(
			'qts_quote_request_created',
			'qts_quote_request_status_approved',
			'qts_quote_request_status_accepted',
			'qts_quote_request_status_expired',
			'qts_quote_request_new_customer_note_added_by_customer',
			'qts_quote_request_new_customer_note_added_by_admin',
			'qts_quote_request_rejected_by_admin',
			'qts_quote_request_accepted',
			'qts_quote_request_rejected',
			'qts_remind_customer_to_pay',
			'qts_remind_customer_to_respond_in',
			'qts_quote_request_payment_complete',
				) ) ;

		foreach ( $email_actions as $action ) {
			add_action( $action, array( __CLASS__, 'send_notification' ), 10, 10 ) ;
		}
	}

	/**
	 * Init the WC mailer instance and call the notifications for the current filter.
	 *
	 * @param array $args Email args (default: []).
	 */
	public static function send_notification( $args = array() ) {
		try {
			WC()->mailer() ;
			$args = func_get_args() ;
			do_action_ref_array( current_filter() . '_notification', $args ) ;
		} catch ( Exception $e ) {
			return ;
		}
	}

	/**
	 * Load our email classes.
	 * 
	 * @param array $emails
	 */
	public function add_email_classes( $emails ) {
		if ( ! empty( $this->emails ) ) {
			return $emails + $this->emails ;
		}

		// Include email classes.
		include_once 'abstracts/abstract-qts-email.php' ;

		foreach ( $this->email_classes as $id => $class ) {
			$file_name = 'class-' . strtolower( str_replace( '_', '-', $class ) ) ;
			$path      = QTS_DIR . "includes/emails/{$file_name}.php" ;

			if ( is_readable( $path ) ) {
				$this->emails[ $class ] = include( $path ) ;
			}
		}

		return $emails + $this->emails ;
	}

	/**
	 * Hide Template - Plain text
	 */
	public function hide_plain_text_template() {
		if ( ! isset( $_GET[ 'section' ] ) ) {
			return ;
		}

		WC()->mailer() ;

		if ( in_array( $_GET[ 'section' ], array_map( 'strtolower', array_keys( $this->emails ) ) ) ) {
			echo '<style>div.template_plain{display:none;}</style>' ;
		}
	}

	/**
	 * Add the quote request note when the email sent successful.
	 * 
	 * @param WC_Email $email
	 */
	public function email_sent( $email ) {
		if ( is_object( $email->object ) ) {
			/* translators: 1: email name 2: email recipients */
			$email->object->add_order_note( sprintf( __( '<code>%1$s</code> - email has been sent to %2$s.', 'quote-request-for-woocommerce' ), $email->title, $email->recipient ) ) ;
		}
	}

	/**
	 * Add the quote request note when the email failed to sent.
	 * 
	 * @param WC_Email $email
	 */
	public function email_failed_to_sent( $email ) {
		if ( is_object( $email->object ) ) {
			/* translators: 1: email name 2: email recipients */
			$email->object->add_order_note( sprintf( __( 'Failed to sent <code>%1$s</code> - email to %2$s.', 'quote-request-for-woocommerce' ), $email->title, $email->recipient ) ) ;
		}
	}

	/**
	 * Show the quote request details table.
	 *
	 * @param QTS_Quote_Request $quote_request 
	 * @param bool $sent_to_admin If should sent to admin.
	 * @param bool $plain_text If is plain text email.
	 * @param WC_Email $email 
	 */
	public function quote_request_details( $quote_request, $sent_to_admin = false, $plain_text = false, $email = '' ) {
		if ( $plain_text ) {
			_qts_get_template( 'emails/plain/email-quote-request-details.php', array(
				'quote_request' => $quote_request,
				'sent_to_admin' => $sent_to_admin,
				'plain_text'    => $plain_text,
				'email'         => $email,
			) ) ;
		} else {
			_qts_get_template( 'emails/email-quote-request-details.php', array(
				'quote_request' => $quote_request,
				'sent_to_admin' => $sent_to_admin,
				'plain_text'    => $plain_text,
				'email'         => $email,
			) ) ;
		}
	}

	/**
	 * Set our email templates directory.
	 * 
	 * @param string $template_directory
	 * @param string $template
	 * @return string
	 */
	public function set_template_directory( $template_directory, $template ) {
		$templates = array_map( array( $this, 'get_template_name' ), array_keys( $this->email_classes ) ) ;

		foreach ( $templates as $name ) {
			if ( in_array( $template, array(
						"emails/{$name}.php",
						"emails/plain/{$name}.php",
					) )
			) {
				return untrailingslashit( QTS_BASENAME_DIR ) ;
			}
		}

		return $template_directory ;
	}

	/**
	 * Get the template name from email ID
	 */
	public function get_template_name( $id ) {
		return str_replace( '_', '-', $id ) ;
	}

	/**
	 * Are emails available ?
	 *
	 * @return WC_Email class
	 */
	public function available() {
		WC()->mailer() ;
		return ! empty( $this->emails ) ? true : false ;
	}

	/**
	 * Return the email class
	 *
	 * @param string $id
	 * @return null|WC_Email class name
	 */
	public function get_email_class( $id ) {
		$id = strtolower( $id ) ;

		if ( false !== stripos( $id, QTS_PREFIX ) ) {
			$id = ltrim( $id, QTS_PREFIX ) ;
		}

		return isset( $this->email_classes[ $id ] ) ? $this->email_classes[ $id ] : null ;
	}

	/**
	 * Return the emails
	 *
	 * @return WC_Email[]
	 */
	public function get_emails() {
		WC()->mailer() ;
		return $this->emails ;
	}

	/**
	 * Return the email
	 *
	 * @param string $id
	 * @return WC_Email
	 */
	public function get_email( $id ) {
		WC()->mailer() ;
		$class = $this->get_email_class( $id ) ;
		return isset( $this->emails[ $class ] ) ? $this->emails[ $class ] : null ;
	}

}
