<?php

/**
 * Handle frontend forms.
 */
defined( 'ABSPATH' ) || exit ;

/**
 * QTS_Form_Handler class.
 */
class QTS_Form_Handler {

	/**
	 * Hook in methods.
	 */
	public static function init() {
		add_action( 'wp_loaded', array( __CLASS__, 'update_quote_action' ), 20 ) ;
		add_action( 'wp_loaded', array( __CLASS__, 'add_to_quote_action' ), 20 ) ;
		add_action( 'wp_loaded', array( __CLASS__, 'submit_quote' ), 20 ) ;
		add_action( 'wp_loaded', array( __CLASS__, 'update_quote_request_via_account_page' ), 20 ) ;
		add_action( 'wp_loaded', array( __CLASS__, 'update_quote_request_via_email' ), 20 ) ;
		add_action( 'wp_loaded', array( __CLASS__, 'pay_via_checkout' ), 20 ) ;

		add_filter( 'woocommerce_add_to_cart_redirect', array( __CLASS__, 'maybe_redirect_to_cart' ), 99 ) ;
		add_action( 'woocommerce_checkout_order_processed', array( __CLASS__, 'checkout_quote' ), 5, 2 ) ;
		add_action( 'before_woocommerce_pay', array( __CLASS__, 'validate_pay_for_order' ), 5 ) ;
	}

	/**
	 * Perform the quote action.
	 */
	public static function update_quote_action() {
		if ( ! isset( $_REQUEST[ QTS_PREFIX . 'nonce' ] ) ) {
			return ;
		}

		if ( ! ( isset( $_REQUEST[ 'remove_item' ] ) || isset( $_REQUEST[ 'undo_item' ] ) || isset( $_REQUEST[ 'update_quote' ] ) ) ) {
			return ;
		}

		wc_nocache_headers() ;
		$nonce = sanitize_key( wp_unslash( $_REQUEST[ QTS_PREFIX . 'nonce' ] ) ) ;

		if ( ! empty( $_REQUEST[ 'remove_item' ] ) && wp_verify_nonce( $nonce, 'qts-quote' ) ) {
			$quote_item_key = sanitize_text_field( wp_unslash( $_REQUEST[ 'remove_item' ] ) ) ;
			$quote_item     = _qts()->quote->get_quote_item( $quote_item_key ) ;

			if ( $quote_item ) {
				_qts()->quote->remove_quote_item( $quote_item_key ) ;

				$product = wc_get_product( $quote_item[ 'product_id' ] ) ;

				/* translators: %s: Item name. */
				$item_removed_title = $product ? sprintf( _x( '&ldquo;%s&rdquo;', 'Item name in quotes', 'quote-request-for-woocommerce' ), $product->get_name() ) : __( 'Item', 'quote-request-for-woocommerce' ) ;

				if ( $product && $product->is_in_stock() && $product->has_enough_stock( $quote_item[ 'quantity' ] ) ) {
					/* Translators: %s Product title. */
					$removed_notice = sprintf( __( '%s removed.', 'quote-request-for-woocommerce' ), $item_removed_title ) ;
					$removed_notice .= ' <a href="' . esc_url( _qts_get_quote_undo_url( $quote_item_key ) ) . '" class="restore-item">' . __( 'Undo?', 'quote-request-for-woocommerce' ) . '</a>' ;
					wc_add_notice( $removed_notice, 'success' ) ;
				} else {
					/* Translators: %s Product title. */
					wc_add_notice( sprintf( __( '%s removed.', 'quote-request-for-woocommerce' ), $item_removed_title ), 'success' ) ;
				}
			}

			wp_safe_redirect( wp_get_referer() ? remove_query_arg( array( 'remove_item', 'add-to-qtsquote', QTS_PREFIX . 'nonce' ), add_query_arg( 'removed_item', '1', wp_get_referer() ) ) : _qts_get_quote_url()  ) ;
			exit ;
		} elseif ( ! empty( $_REQUEST[ 'undo_item' ] ) && wp_verify_nonce( $nonce, 'qts-quote' ) ) {
			$quote_item_key = sanitize_text_field( wp_unslash( $_REQUEST[ 'undo_item' ] ) ) ;

			_qts()->quote->restore_quote_item( $quote_item_key ) ;

			wp_safe_redirect( wp_get_referer() ? remove_query_arg( array( 'undo_item', 'add-to-qtsquote', QTS_PREFIX . 'nonce' ), wp_get_referer() ) : _qts_get_quote_url()  ) ;
			exit ;
		} elseif ( ! empty( $_POST[ 'update_quote' ] ) && wp_verify_nonce( $nonce, 'qts-quote' ) ) {
			$quote_updated = false ;
			$quote_totals  = isset( $_POST[ QTS_PREFIX . 'quote' ] ) ? wc_clean( wp_unslash( $_POST[ QTS_PREFIX . 'quote' ] ) ) : '' ;

			if ( ! _qts()->quote->is_empty() && is_array( $quote_totals ) ) {
				foreach ( _qts()->quote->get_quote() as $quote_item_key => $values ) {
					$_product = $values[ 'data' ] ;

					if ( ! isset( $quote_totals[ $quote_item_key ] ) || ! isset( $quote_totals[ $quote_item_key ][ 'qty' ] ) ) {
						continue ;
					}

					$quantity                   = wc_stock_amount( preg_replace( '/[^0-9\.]/', '', $quote_totals[ $quote_item_key ][ 'qty' ] ) ) ;
					$requested_price            = null ;
					$requested_price_percent    = null ;
					$requested_discount_percent = null ;

					if ( '' === $quantity ) {
						continue ;
					}

					if ( isset( $quote_totals[ $quote_item_key ][ 'requested_price' ] ) && is_numeric( $quote_totals[ $quote_item_key ][ 'requested_price' ] ) ) {
						$requested_price = wc_format_decimal( $quote_totals[ $quote_item_key ][ 'requested_price' ] ) ;

						if ( $requested_price === $values[ 'requested_price' ] && $quantity === $values[ 'quantity' ] ) {
							continue ;
						}
					} else if ( isset( $quote_totals[ $quote_item_key ][ 'requested_price_percent' ] ) && is_numeric( $quote_totals[ $quote_item_key ][ 'requested_price_percent' ] ) ) {
						$requested_price_percent = floatval( $quote_totals[ $quote_item_key ][ 'requested_price_percent' ] ) ;

						if ( isset( $values[ 'requested_price_percent' ] ) && $requested_price_percent === $values[ 'requested_price_percent' ] && $quantity === $values[ 'quantity' ] ) {
							continue ;
						}
					} else if ( isset( $quote_totals[ $quote_item_key ][ 'requested_discount_percent' ] ) && is_numeric( $quote_totals[ $quote_item_key ][ 'requested_discount_percent' ] ) ) {
						$requested_discount_percent = floatval( $quote_totals[ $quote_item_key ][ 'requested_discount_percent' ] ) ;

						if ( isset( $values[ 'requested_discount_percent' ] ) && $requested_discount_percent === $values[ 'requested_discount_percent' ] && $quantity === $values[ 'quantity' ] ) {
							continue ;
						}
					} else {
						if (
								(
								is_null( $values[ 'requested_price' ] ) ||
								( isset( $values[ 'requested_price_percent' ] ) && is_null( $values[ 'requested_price_percent' ] ) ) ||
								( isset( $values[ 'requested_discount_percent' ] ) && is_null( $values[ 'requested_discount_percent' ] ) )
								) &&
								$quantity === $values[ 'quantity' ]
						) {
							continue ;
						}
					}

					// Update quote validation.
					$passed_validation = apply_filters( 'qts_update_quote_validation', true, $quote_item_key, $values, $quantity ) ;

					// Is sold individually.
					if ( $_product->is_sold_individually() && $quantity > 1 ) {
						/* Translators: %s Product title. */
						wc_add_notice( trim( get_option( QTS_PREFIX . 'sold_individual_err_while_adding_quote' ) ), 'error' ) ;
						$passed_validation = false ;
					}

					// Check whether the requested price less than the minimum price to ask for quote.
					if ( is_numeric( $requested_price ) ) {
						$min_quote_price = _qts_get_min_quote_price( $_product ) ;

						if ( $min_quote_price > 0 && floatval( $requested_price ) < $min_quote_price ) {
							wc_add_notice( trim( str_replace( array( '[product_name]', '[minimum_price]' ), array( $_product->get_name(), wc_price( $min_quote_price ) ), get_option( QTS_PREFIX . 'cannot_request_less_than_min_price_err' ) ) ), 'error' ) ;
							$passed_validation = false ;
						}
					} else if ( is_numeric( $requested_price_percent ) ) {
						$min_quote_price_percent = _qts_get_min_quote_price_percent( $_product ) ;

						if ( $min_quote_price_percent > 0 && floatval( $requested_price_percent ) < $min_quote_price_percent ) {
							wc_add_notice( trim( str_replace( array( '[product_name]', '[minimum_price_percent]' ), array( $_product->get_name(), "{$min_quote_price_percent}%" ), get_option( QTS_PREFIX . 'cannot_request_less_than_min_price_percent_err' ) ) ), 'error' ) ;
							$passed_validation = false ;
						}
					} else if ( is_numeric( $requested_discount_percent ) ) {
						if ( floatval( $requested_discount_percent ) < 0 || floatval( $requested_discount_percent ) > 100 ) {
							$passed_validation = false ;
						}
					}

					if ( $passed_validation ) {
						if ( is_numeric( $requested_price ) ) {
							_qts()->quote->set_requested_price( $quote_item_key, $requested_price, false ) ;
						} else if ( is_numeric( $requested_price_percent ) ) {
							_qts()->quote->set_requested_price_percent( $quote_item_key, $requested_price_percent, false ) ;
						} else if ( is_numeric( $requested_discount_percent ) ) {
							_qts()->quote->set_requested_discount_percent( $quote_item_key, $requested_discount_percent, false ) ;
						}

						_qts()->quote->set_quantity( $quote_item_key, $quantity, false ) ;
						$quote_updated = true ;
					}
				}
			}

			if ( $quote_updated ) {
				_qts()->quote->calculate_totals() ;
				wc_add_notice( __( 'Quote updated.', 'quote-request-for-woocommerce' ), 'success' ) ;
				wp_safe_redirect( remove_query_arg( 'add-to-qtsquote', ( wp_get_referer() ? wp_get_referer() : _qts_get_quote_url() ) ) ) ;
				exit ;
			}
		}
	}

	/**
	 * Add to quote action.
	 */
	public static function add_to_quote_action() {
		if ( ! isset( $_REQUEST[ 'add-to-qtsquote' ] ) ) {
			return ;
		}

		$product_id = absint( wp_unslash( $_REQUEST[ 'add-to-qtsquote' ] ) ) ;

		if ( ! $product_id ) {
			return ;
		}

		wc_nocache_headers() ;

		$was_added_to_quote = false ;
		$adding_to_quote    = wc_get_product( $product_id ) ;

		if ( ! $adding_to_quote ) {
			return ;
		}

		if ( ! QTS_Add_To_Quote::is_quote_enabled( $adding_to_quote ) ) {
			wc_add_notice( __( 'Invalid product to request for quote.', 'quote-request-for-woocommerce' ), 'error' ) ;
			return ;
		}

		if ( QTS_Add_To_Quote::_get( $product_id, 'do_not_add' ) ) {
			wc_add_notice( __( 'Sorry you are not eligible to request for quote. Please log in and add to quote.', 'quote-request-for-woocommerce' ), 'error' ) ;
			return ;
		}

		if ( $adding_to_quote->is_type( array( 'variable', 'variation' ) ) ) {
			$was_added_to_quote = self::add_to_quote_handler_variation( $product_id ) ;
		} else {
			$was_added_to_quote = self::add_to_quote_handler_simple( $product_id ) ;
		}

		if ( $was_added_to_quote && 0 === wc_notice_count( 'error' ) ) {
			// Maybe clear the quote requests added from cart session.
			_qts_empty_quote( 'accurate' ) ;

			// If we added the product to the quote we can now optionally do a redirect.
			if ( 'yes' === get_option( QTS_PREFIX . 'quote_redirect_after_add' ) ) {
				wp_safe_redirect( _qts_get_quote_url() ) ;
				exit ;
			}
		}
	}

	/**
	 * Redirect to cart if the quote product is added successful.
	 * And maybe clear the quote session.
	 * 
	 * @param string $url
	 * @return string
	 */
	public static function maybe_redirect_to_cart( $url ) {
		if ( ! isset( $_REQUEST[ 'add-to-cart' ], $_REQUEST[ 'adding_qtsquote' ] ) ) {
			return $url ;
		}

		$adding_to_cart = absint( wp_unslash( $_REQUEST[ 'add-to-cart' ] ) ) ;

		if ( ! QTS_Add_To_Quote::is_quote_enabled( $adding_to_cart ) ) {
			return $url ;
		}

		// Maybe clear the quote requests added from quote session.
		_qts_empty_quote( 'approx' ) ;

		if ( 'yes' !== get_option( QTS_PREFIX . 'quote_redirect_after_add' ) ) {
			return $url ;
		}

		return wc_get_cart_url() ;
	}

	/**
	 * Handle adding simple products to the quote.
	 *
	 * @param int $product_id Product ID 
	 * @return bool success or not
	 */
	private static function add_to_quote_handler_simple( $product_id ) {
		$quantity          = empty( $_REQUEST[ 'quantity' ] ) ? 1 : wc_stock_amount( wp_unslash( $_REQUEST[ 'quantity' ] ) ) ;
		$passed_validation = apply_filters( 'qts_add_to_quote_validation', true, $product_id, $quantity ) ;

		if ( $passed_validation && false !== _qts()->quote->add_to_quote( $product_id, $quantity ) ) {
			_qts_add_to_quote_message( array( $product_id => $quantity ), true ) ;
			return true ;
		}

		return false ;
	}

	/**
	 * Handle adding variation products to the quote.
	 *
	 * @param int $product_id Product ID
	 * @return bool success or not
	 */
	private static function add_to_quote_handler_variation( $product_id ) {
		try {
			$variation_id    = empty( $_REQUEST[ 'variation_id' ] ) ? '' : absint( wp_unslash( $_REQUEST[ 'variation_id' ] ) ) ;
			$quantity        = empty( $_REQUEST[ 'quantity' ] ) ? 1 : wc_stock_amount( wp_unslash( $_REQUEST[ 'quantity' ] ) ) ;
			$adding_to_quote = wc_get_product( $product_id ) ;

			if ( ! $adding_to_quote ) {
				return false ;
			}

			if ( $adding_to_quote->is_type( 'variation' ) ) {
				$variation_id    = $product_id ;
				$product_id      = $adding_to_quote->get_parent_id() ;
				$adding_to_quote = wc_get_product( $product_id ) ;

				if ( ! $adding_to_quote ) {
					return false ;
				}
			}

			if ( empty( $variation_id ) ) {
				throw new Exception( __( 'Please choose product options&hellip;', 'quote-request-for-woocommerce' ) ) ;
			}
		} catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' ) ;
			return false ;
		}

		$passed_validation = apply_filters( 'qts_add_to_quote_validation', true, $product_id, $quantity, $variation_id ) ;

		if ( $passed_validation && false !== _qts()->quote->add_to_quote( $product_id, $quantity, $variation_id ) ) {
			_qts_add_to_quote_message( array( $variation_id => $quantity ), true ) ;
			return true ;
		}

		return false ;
	}

	/**
	 * Create the quote request from quote page.
	 */
	public static function submit_quote() {
		if ( ! isset( $_REQUEST[ QTS_PREFIX . 'nonce' ] ) || ! isset( $_REQUEST[ 'request_quote' ] ) ) {
			return ;
		}

		if ( ! wp_verify_nonce( sanitize_key( wp_unslash( $_REQUEST[ QTS_PREFIX . 'nonce' ] ) ), 'qts-quote-request' ) ) {
			return ;
		}

		if ( _qts()->quote->is_empty() ) {
			return ;
		}

		if ( ! _qts()->quote->check_quote_items( false ) ) {
			return ;
		}

		try {
			$errors = new WP_Error() ;
			$posted = $_REQUEST ;

			// Validate posted data before proceeding.
			QTS_Form_Fields::validate_posted_data( $posted, $errors ) ;

			do_action( 'qts_after_quote_request_validation', $posted, $errors ) ;

			$messages = $errors->get_error_messages() ;

			if ( $messages ) {
				throw new Exception( implode( '<br>', $messages ) ) ;
			}

			// Create the quote request.
			$quote_request = new QTS_Quote_Request() ;

			$form_fields = array() ;
			foreach ( QTS_Form_Fields::get_available_fields() as $field_key => $field ) {
				if ( ! isset( $posted[ $field_key ] ) ) {
					continue ;
				}

				$value = wc_clean( wp_unslash( $posted[ $field_key ] ) ) ;

				if ( 'billing_email' === $field_key ) {
					$quote_request->set_billing_email( $value ) ;
				} else if ( 'qts_customer_note' === $field_key ) {
					$quote_request->set_customer_note( $value ) ;
				} else {
					if ( in_array( $field_key, array( 'billing_first_name', 'billing_last_name' ) ) ) {
						$metakey = '_customer_' . str_replace( 'billing_', '', $field_key ) ;
						$quote_request->add_meta_data( $metakey, $value ) ;
					}
				}

				$form_fields[ $field_key ]            = $field ;
				$form_fields[ $field_key ][ 'value' ] = $value ;
			}

			$quote_request->set_created_via( 'quote' ) ;
			$quote_request->set_form_fields( $form_fields ) ;
			$quote_request->set_customer_id( get_current_user_id() ) ;
			$quote_request->set_customer_ip_address( WC_Geolocation::get_ip_address() ) ;
			$quote_request->set_customer_user_agent( wc_get_user_agent() ) ;
			$quote_request->set_currency( get_woocommerce_currency() ) ;
			$quote_request->set_total( _qts()->quote->get_subtotal( 'edit' ) ) ;
			$quote_request->create_quote_items( _qts()->quote->get_quote() ) ;
			$quote_request->save() ;

			do_action( 'qts_quote_request_created', $quote_request ) ;

			if ( $quote_request->get_user_id() ) {
				wc_add_notice( trim( str_replace( array( '[quote_request_number]', '[view_quote_request_url]' ), array( esc_html( $quote_request->get_id() ), esc_url( $quote_request->get_view_order_url() ) ), get_option( QTS_PREFIX . 'quote_submitted_success_msg_for_logged_in_users' ) ) ), 'success' ) ;
			} else {
				wc_add_notice( trim( str_replace( '[quote_request_number]', esc_html( $quote_request->get_id() ), get_option( QTS_PREFIX . 'quote_submitted_success_msg_for_guests' ) ) ), 'success' ) ;
			}
		} catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' ) ;
		}
	}

	/**
	 * Create the quote request from checkout page.
	 * 
	 * @param int $order_id 
	 * @param array $posted 
	 */
	public static function checkout_quote( $order_id, $posted ) {
		if ( is_null( WC()->cart ) || ! QTS_Cart::asking_for_quote() ) {
			return ;
		}

		try {
			$available_gateways = WC()->payment_gateways->get_available_payment_gateways() ;

			// Create the quote request.
			$quote_request = new QTS_Quote_Request() ;

			foreach ( $posted as $key => $value ) {
				if ( is_callable( array( $quote_request, "set_{$key}" ) ) ) {
					$quote_request->{"set_{$key}"}( $value ) ;
				}
			}

			$quote_request->set_order_id( $order_id ) ;
			$quote_request->set_created_via( 'checkout' ) ;
			$quote_request->set_customer_id( get_current_user_id() ) ;
			$quote_request->set_prices_include_tax( 'yes' === get_option( 'woocommerce_prices_include_tax' ) ) ;
			$quote_request->add_meta_data( 'is_vat_exempt', ( WC()->cart->get_customer()->get_is_vat_exempt() ? 'yes' : 'no' ), true ) ;
			$quote_request->set_customer_ip_address( WC_Geolocation::get_ip_address() ) ;
			$quote_request->set_customer_user_agent( wc_get_user_agent() ) ;
			$quote_request->set_currency( get_woocommerce_currency() ) ;
			$quote_request->set_customer_note( isset( $posted[ 'qts_customer_note' ] ) ? $posted[ 'qts_customer_note' ] : ''  ) ;
			$quote_request->set_payment_method( isset( $available_gateways[ $posted[ 'payment_method' ] ] ) ? $available_gateways[ $posted[ 'payment_method' ] ] : $posted[ 'payment_method' ]  ) ;
			$quote_request->set_shipping_total( WC()->cart->get_shipping_total() ) ;
			$quote_request->set_discount_total( WC()->cart->get_discount_total() ) ;
			$quote_request->set_discount_tax( WC()->cart->get_discount_tax() ) ;
			$quote_request->set_cart_tax( WC()->cart->get_cart_contents_tax() + WC()->cart->get_fee_tax() ) ;
			$quote_request->set_shipping_tax( WC()->cart->get_shipping_tax() ) ;
			$quote_request->set_total( WC()->cart->get_total( 'edit' ) ) ;
			$quote_request->create_quote_items( WC()->cart->get_cart() ) ;
			WC()->checkout->create_order_fee_lines( $quote_request, WC()->cart ) ;
			WC()->checkout->create_order_shipping_lines( $quote_request, WC()->session->get( 'chosen_shipping_methods' ), WC()->shipping()->get_packages() ) ;
			WC()->checkout->create_order_tax_lines( $quote_request, WC()->cart ) ;
			WC()->checkout->create_order_coupon_lines( $quote_request, WC()->cart ) ;
			$quote_request->save() ;

			// Save the quote request in order meta.
			add_post_meta( $order_id, QTS_PREFIX . 'is_quote_request_order', 'yes' ) ;
			add_post_meta( $order_id, QTS_PREFIX . 'quote_request_id', $quote_request->get_id() ) ;

			// Clear cache.
			WC()->session->set( QTS_PREFIX . 'customer_note', '' ) ;

			do_action( 'qts_quote_request_created', $quote_request ) ;
		} catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' ) ;
		}
	}

	/**
	 * Validate pay for order page and print the notice to customer whether he is eligible to pay his quote request.
	 */
	public static function validate_pay_for_order() {
		global $wp ;

		$order_id = ! empty( $wp->query_vars[ 'order-pay' ] ) ? absint( $wp->query_vars[ 'order-pay' ] ) : 0 ;

		if ( isset( $_GET[ 'pay_for_order' ], $_GET[ 'qtsquote_request' ], $_GET[ 'key' ] ) ) {
			$order = wc_get_order( $order_id ) ;

			if ( ! $order ) {
				return ;
			}

			try {
				$quote_request_key = wc_clean( wp_unslash( $_GET[ 'qtsquote_request' ] ) ) ;
				$quote_request_id  = absint( _qts_get_quote_request_id_by_quote_request_key( $quote_request_key ) ) ;
				$quote_request     = _qts_get_quote_request( $quote_request_id ) ;

				if ( ! $quote_request || $quote_request->is_created_via( 'quote' ) || $order_id !== $quote_request->get_order_id() ) {
					throw new Exception( __( 'Sorry, this quote request is invalid and you cannot pay for this order.', 'quote-request-for-woocommerce' ) ) ;
				}

				if ( ! $order->has_status( 'pending' ) ) {
					throw new Exception( __( 'Sorry, you cannot pay for this quote request.', 'quote-request-for-woocommerce' ) ) ;
				}

				// Check order items stock
				if ( ! _qts_allow_unlimited_qty() || 'yes' !== get_option( QTS_PREFIX . 'make_payment_when_not_having_enough_stock', 'no' ) ) {
					$out_of_stock_products = array() ;

					foreach ( $order->get_items( 'line_item' ) as $item ) {
						$product = $item->get_product() ;

						if ( $product && ! $product->has_enough_stock( $item->get_quantity() ) ) {
							$out_of_stock_products[] = $product->get_name() ;
						}
					}

					if ( ! empty( $out_of_stock_products ) ) {
						/* Translators: %s Product title(s). */
						throw new Exception( sprintf( __( 'Sorry, we do not have enough "%s" in stock to fulfill your order. We apologize for any inconvenience caused.', 'quote-request-for-woocommerce' ), implode( ', ', $out_of_stock_products ) ) ) ;
					}
				}

				wc_print_notice( trim( str_replace( '[quote_request_number]', $quote_request_id, get_option( QTS_PREFIX . 'complete_payment_msg' ) ) ), 'success' ) ;
			} catch ( Exception $e ) {
				add_filter( 'woocommerce_order_needs_payment', '__return_false', 99 ) ;
				wc_print_notice( $e->getMessage(), 'error' ) ;
			}
		}
	}

	/**
	 * Update the quote request based upon status either Accept || Reject via my account.
	 */
	public static function update_quote_request_via_account_page() {
		if ( ! isset( $_GET[ 'qtsaction' ] ) || ! isset( $_GET[ 'qtsquote_request' ] ) || ! isset( $_GET[ '_wpnonce' ] ) ) {
			return ;
		}

		$action = wc_clean( wp_unslash( $_GET[ 'qtsaction' ] ) ) ;

		if ( ! wp_verify_nonce( sanitize_key( wp_unslash( $_GET[ '_wpnonce' ] ) ), "{$action}-qtsquote_request" ) ) {
			return ;
		}

		wc_nocache_headers() ;

		$quote_request_key = wc_clean( wp_unslash( $_GET[ 'qtsquote_request' ] ) ) ;
		$quote_request_id  = absint( _qts_get_quote_request_id_by_quote_request_key( $quote_request_key ) ) ;
		$quote_request     = _qts_get_quote_request( $quote_request_id ) ;
		$user_can          = current_user_can( "{$action}_qtsquote_request", $quote_request_id ) ;
		$redirect          = isset( $_GET[ 'qtsredirect' ] ) ? esc_url_raw( wp_unslash( $_GET[ 'qtsredirect' ] ) ) : '' ;

		// Make sure the quote request is valid.
		if ( $user_can && $quote_request->has_status( 'response_awaitd' ) && $quote_request->get_id() === $quote_request_id && hash_equals( $quote_request->get_order_key(), $quote_request_key ) ) {
			if ( 'accept' === $action ) {
				$quote_request->update_status( 'accepted', __( 'Quote accepted by customer.', 'quote-request-for-woocommerce' ) ) ;

				wc_add_notice( __( 'Your quote request was accepted.', 'quote-request-for-woocommerce' ), 'notice' ) ;
				do_action( 'qts_quote_request_accepted', $quote_request->get_id() ) ;
			} else if ( 'reject' === $action ) {
				$quote_request->update_status( 'rejected', __( 'Quote rejected by customer.', 'quote-request-for-woocommerce' ) ) ;

				wc_add_notice( __( 'Your quote request was rejected.', 'quote-request-for-woocommerce' ), 'notice' ) ;
				do_action( 'qts_quote_request_rejected', $quote_request->get_id() ) ;
			}
		} elseif ( $user_can && ! $quote_request->has_status( 'response_awaitd' ) ) {
			if ( 'accept' === $action ) {
				wc_add_notice( __( 'Your quote request can no longer be accepted. Please contact us if you need assistance.', 'quote-request-for-woocommerce' ), 'error' ) ;
			} else if ( 'reject' === $action ) {
				wc_add_notice( __( 'Your quote request can no longer be rejected. Please contact us if you need assistance.', 'quote-request-for-woocommerce' ), 'error' ) ;
			}
		} else {
			wc_add_notice( __( 'Invalid quote request.', 'quote-request-for-woocommerce' ), 'error' ) ;
		}

		if ( $redirect ) {
			wp_safe_redirect( $redirect ) ;
			exit ;
		}
	}

	/**
	 * Update the quote request based upon status either Accept || Reject via email.
	 */
	public static function update_quote_request_via_email() {
		if ( ! isset( $_GET[ 'qtsaction' ] ) || ! isset( $_GET[ 'qtsquote_request' ] ) || ! isset( $_GET[ 'qtskey' ] ) ) {
			return ;
		}

		wc_nocache_headers() ;

		$action            = wc_clean( wp_unslash( $_GET[ 'qtsaction' ] ) ) ;
		$quote_request_key = wc_clean( wp_unslash( $_GET[ 'qtsquote_request' ] ) ) ;
		$quote_request_id  = absint( _qts_get_quote_request_id_by_quote_request_key( $quote_request_key ) ) ;
		$quote_request     = _qts_get_quote_request( $quote_request_id ) ;
		$user_can          = current_user_can( "{$action}_qtsquote_request", $quote_request_id ) ;
		$redirect          = isset( $_GET[ 'qtsredirect' ] ) ? esc_url_raw( wp_unslash( $_GET[ 'qtsredirect' ] ) ) : '' ;
		$email_hash        = $quote_request ? hash( 'sha256', $quote_request->get_billing_email() ) : '' ;

		// Make sure the user is valid to access/update quote request.
		if ( $user_can && hash_equals( wp_unslash( $_GET[ 'qtskey' ] ), $email_hash ) ) {
			// Make sure the quote request is valid.
			if ( $quote_request->has_status( 'response_awaitd' ) && $quote_request->get_id() === $quote_request_id && hash_equals( $quote_request->get_order_key(), $quote_request_key ) ) {
				if ( 'accept' === $action ) {
					$quote_request->update_status( 'accepted', __( 'Quote accepted by customer.', 'quote-request-for-woocommerce' ) ) ;

					wc_add_notice( __( 'Your quote request was accepted.', 'quote-request-for-woocommerce' ), 'notice' ) ;
					do_action( 'qts_quote_request_accepted', $quote_request->get_id() ) ;
				} else if ( 'reject' === $action ) {
					$quote_request->update_status( 'rejected', __( 'Quote rejected by customer.', 'quote-request-for-woocommerce' ) ) ;

					wc_add_notice( __( 'Your quote request was rejected.', 'quote-request-for-woocommerce' ), 'notice' ) ;
					do_action( 'qts_quote_request_rejected', $quote_request->get_id() ) ;
				}
			} elseif ( ! $quote_request->has_status( 'response_awaitd' ) ) {
				if ( 'accept' === $action ) {
					wc_add_notice( __( 'Your quote request can no longer be accepted. Please contact us if you need assistance.', 'quote-request-for-woocommerce' ), 'error' ) ;
				} else if ( 'reject' === $action ) {
					wc_add_notice( __( 'Your quote request can no longer be rejected. Please contact us if you need assistance.', 'quote-request-for-woocommerce' ), 'error' ) ;
				}
			} else {
				wc_add_notice( __( 'Invalid quote request.', 'quote-request-for-woocommerce' ), 'error' ) ;
			}
		} else {
			/* translators: %s: quote request action */
			wc_add_notice( sprintf( __( 'Invalid user to %s quote request.', 'quote-request-for-woocommerce' ), $action ), 'error' ) ;
		}

		if ( $redirect ) {
			wp_safe_redirect( $redirect ) ;
			exit ;
		}
	}

	/**
	 * Prepare the payment process via checkout.
	 * 
	 * @throws Exception When validate fails.
	 */
	public static function pay_via_checkout() {
		if ( ! isset( $_GET[ 'qtscheckout' ] ) || ! isset( $_GET[ 'qtsquote_request' ] ) ) {
			return ;
		}

		try {
			wc_nocache_headers() ;

			$quote_request_key = wc_clean( wp_unslash( $_GET[ 'qtsquote_request' ] ) ) ;
			$quote_request_id  = absint( _qts_get_quote_request_id_by_quote_request_key( $quote_request_key ) ) ;
			$quote_request     = _qts_get_quote_request( $quote_request_id ) ;

			// Quote Request or payment link is invalid.
			if ( ! $quote_request || $quote_request->get_id() !== $quote_request_id || ! hash_equals( $quote_request->get_order_key(), $quote_request_key ) || ! $quote_request->is_created_via( 'quote' ) ) {
				throw new Exception( __( 'Sorry, this quote request is invalid and cannot be paid for.', 'quote-request-for-woocommerce' ) ) ;
			}

			// Logged out customer does not have permission to pay for this quote request.
			if ( $quote_request->get_user_id() && ! is_user_logged_in() ) {
				echo '<div class="woocommerce-info">' . esc_html__( 'Please log in to your account below to continue to the payment for quote request.', 'quote-request-for-woocommerce' ) . '</div>' ;
				woocommerce_login_form( array(
					'redirect' => $quote_request->get_checkout_payment_url(),
						)
				) ;
				return ;
			}

			// Does not need payment.
			if ( ! $quote_request->needs_payment() ) {
				/* translators: %s: quote request status */
				throw new Exception( sprintf( __( 'This quote request&rsquo;s status is &ldquo;%s&rdquo;&mdash;it cannot be paid for. Please contact us if you need assistance.', 'quote-request-for-woocommerce' ), _qts_get_quote_request_status_name( $quote_request->get_status() ) ) ) ;
			}

			// Force redirect to checkout when the cart is already awating for payment.
			$added_quote = QTS_Cart::awaiting_for_payment() ;
			if ( $added_quote && ( absint( $added_quote[ 'payment' ] ) === $quote_request_id ) ) {
				wp_safe_redirect( wc_get_checkout_url() ) ;
				exit ;
			}

			if ( ! WC()->cart->is_empty() ) {
				WC()->cart->empty_cart() ;
				wc_add_notice( trim( get_option( QTS_PREFIX . 'cannot_purchase_normal_products_while_making_payment_notice' ) ), 'notice' ) ;
			}

			$quote_request_items = $quote_request->get_items( 'line_item' ) ;
			if ( empty( $quote_request_items ) ) {
				throw new Exception( __( 'Sorry, this quote request items are missing and it cannot be paid for. Please contact us if you need assistance.', 'quote-request-for-woocommerce' ) ) ;
			}

			$was_added_to_cart = array() ;
			foreach ( $quote_request_items as $item ) {
				$product = $item->get_product() ;

				if ( ! $product ) {
					continue ;
				}

				$was_added_to_cart[] = WC()->cart->add_to_cart( $product->get_id(), $item->get_quantity(), 0, array(), array(
					'qts_quote' => array(
						'payment'         => $quote_request_id,
						'original_price'  => is_numeric( $item->get_meta( '_original_price' ) ) ? ( float ) $item->get_meta( '_original_price' ) : 0,
						'requested_price' => is_numeric( $item->get_meta( '_requested_price' ) ) ? ( float ) $item->get_meta( '_requested_price' ) : 0,
						'offered_price'   => $item->get_total() / $item->get_quantity(),
						'qty'             => $item->get_quantity()
					) ) ) ;
			}

			$was_added_to_cart = array_filter( $was_added_to_cart ) ;
			if ( empty( $was_added_to_cart ) ) {
				throw new Exception( __( 'Error while processing the payment. Please contact us if you need assistance.', 'quote-request-for-woocommerce' ) ) ;
			}

			//redirect to checkout page.
			wp_safe_redirect( wc_get_checkout_url() ) ;
			exit ;
		} catch ( Exception $e ) {
			if ( QTS_Cart::awaiting_for_payment() ) {
				WC()->cart->empty_cart() ;
			}
			wc_add_notice( $e->getMessage(), 'error' ) ;
		}
	}

}

QTS_Form_Handler::init() ;
