<?php

defined( 'ABSPATH' ) || exit ;

/**
 * Quote Request Data Store CPT
 * 
 * @class QTS_Quote_Request_Data_Store_CPT
 * @package Class
 */
class QTS_Quote_Request_Data_Store_CPT extends WC_Order_Data_Store_CPT implements WC_Object_Data_Store_Interface, WC_Order_Data_Store_Interface {

	/**
	 * Data stored in meta keys, but not considered "meta" for an quote request.
	 *
	 * @var array
	 */
	protected $quote_request_internal_meta_key_to_props = array(
		'_order_id'                   => 'order_id',
		'_price_negotiations_started' => 'price_negotiations_started',
		'_date_expiry'                => 'date_expiry',
		'_form_fields'                => 'form_fields'
			) ;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->internal_meta_keys = array_merge( $this->internal_meta_keys, array_keys( $this->quote_request_internal_meta_key_to_props ) ) ;
	}

	/*
	  |--------------------------------------------------------------------------
	  | CRUD Methods
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Method to create a new quote request in the database.
	 *
	 * @param QTS_Quote_Request $quote_request object.
	 */
	public function create( &$quote_request ) {
		$quote_request->set_version( QTS_VERSION ) ;
		$quote_request->set_date_created( time() ) ;
		$quote_request->set_currency( $quote_request->get_currency() ? $quote_request->get_currency() : get_woocommerce_currency()  ) ;
		$quote_request->set_order_key( _qts_generate_key() ) ;

		$id = wp_insert_post( apply_filters( 'qts_new_quote_request_data', array(
			'post_date'     => gmdate( 'Y-m-d H:i:s', $quote_request->get_date_created( 'edit' )->getOffsetTimestamp() ),
			'post_date_gmt' => gmdate( 'Y-m-d H:i:s', $quote_request->get_date_created( 'edit' )->getTimestamp() ),
			'post_type'     => $quote_request->get_type( 'edit' ),
			'post_status'   => $this->get_post_status( $quote_request ),
			'ping_status'   => 'closed',
			'post_author'   => 1,
			'post_title'    => $this->get_post_title(),
			'post_password' => $quote_request->get_order_key( 'edit' ),
			'post_parent'   => $quote_request->get_parent_id( 'edit' ),
			'post_excerpt'  => $this->get_post_excerpt( $quote_request ),
				) ), true ) ;

		if ( $id && ! is_wp_error( $id ) ) {
			$quote_request->set_id( $id ) ;
			$this->update_post_meta( $quote_request ) ;
			$quote_request->save_meta_data() ;
			$quote_request->apply_changes() ;
			$this->clear_caches( $quote_request ) ;

			do_action( 'qts_new_quote_request', $quote_request->get_id(), $quote_request ) ;
		}
	}

	/**
	 * Method to read an quote request from the database.
	 *
	 * @param QTS_Quote_Request $quote_request object.
	 * @throws Exception If passed quote request is invalid.
	 */
	public function read( &$quote_request ) {
		$quote_request->set_defaults() ;
		$post_object = get_post( $quote_request->get_id() ) ;

		if ( ! $quote_request->get_id() || ! $post_object || ! in_array( $post_object->post_type, wc_get_order_types(), true ) ) {
			throw new Exception( __( 'Invalid quote request.', 'quote-request-for-woocommerce' ) ) ;
		}

		$quote_request->set_props(
				array(
					'parent_id'     => $post_object->post_parent,
					'date_created'  => 0 < $post_object->post_date_gmt ? _qts_maybe_strtotime( $post_object->post_date_gmt ) : null,
					'date_modified' => 0 < $post_object->post_modified_gmt ? _qts_maybe_strtotime( $post_object->post_modified_gmt ) : null,
					'status'        => $post_object->post_status,
				)
		) ;

		$this->read_order_data( $quote_request, $post_object ) ;
		$quote_request->read_meta_data() ;
		$quote_request->set_object_read( true ) ;
	}

	/**
	 * Method to update an quote request in the database.
	 *
	 * @param QTS_Quote_Request $quote_request object.
	 */
	public function update( &$quote_request ) {
		$quote_request->save_meta_data() ;
		$quote_request->set_version( QTS_VERSION ) ;

		if ( null === $quote_request->get_date_created( 'edit' ) ) {
			$quote_request->set_date_created( time() ) ;
		}

		$changes = $quote_request->get_changes() ;

		// Only update the post when the post data changes.
		if ( array_intersect( array( 'date_created', 'date_modified', 'status', 'parent_id', 'post_excerpt' ), array_keys( $changes ) ) ) {
			$post_data = array(
				'post_date'         => gmdate( 'Y-m-d H:i:s', $quote_request->get_date_created( 'edit' )->getOffsetTimestamp() ),
				'post_date_gmt'     => gmdate( 'Y-m-d H:i:s', $quote_request->get_date_created( 'edit' )->getTimestamp() ),
				'post_status'       => $this->get_post_status( $quote_request ),
				'post_parent'       => $quote_request->get_parent_id( 'edit' ),
				'post_excerpt'      => $this->get_post_excerpt( $quote_request ),
				'post_modified'     => isset( $changes[ 'date_modified' ] ) ? gmdate( 'Y-m-d H:i:s', $quote_request->get_date_modified( 'edit' )->getOffsetTimestamp() ) : current_time( 'mysql' ),
				'post_modified_gmt' => isset( $changes[ 'date_modified' ] ) ? gmdate( 'Y-m-d H:i:s', $quote_request->get_date_modified( 'edit' )->getTimestamp() ) : current_time( 'mysql', 1 ),
					) ;

			/**
			 * When updating this object, to prevent infinite loops, use $wpdb
			 * to update data, since wp_update_post spawns more calls to the
			 * save_post action.
			 *
			 * This ensures hooks are fired by either WP itself (admin screen save),
			 * or an update purely from CRUD.
			 */
			if ( doing_action( 'save_post' ) ) {
				$GLOBALS[ 'wpdb' ]->update( $GLOBALS[ 'wpdb' ]->posts, $post_data, array( 'ID' => $quote_request->get_id() ) ) ;
				clean_post_cache( $quote_request->get_id() ) ;
			} else {
				wp_update_post( array_merge( array( 'ID' => $quote_request->get_id() ), $post_data ) ) ;
			}
			$quote_request->read_meta_data( true ) ; // Refresh internal meta data, in case things were hooked into `save_post` or another WP hook.
		}

		$this->update_post_meta( $quote_request ) ;
		$quote_request->apply_changes() ;
		$this->clear_caches( $quote_request ) ;
	}

	/**
	 * Method to delete an quote request from the database.
	 *
	 * @param QTS_Quote_Request $quote_request object.
	 * @param array $args Array of args to pass to the delete method.
	 */
	public function delete( &$quote_request, $args = array() ) {
		$id   = $quote_request->get_id() ;
		$args = wp_parse_args( $args, array(
			'force_delete' => false,
				) ) ;

		if ( ! $id ) {
			return ;
		}

		if ( $args[ 'force_delete' ] ) {
			wp_delete_post( $id ) ;
			$quote_request->set_id( 0 ) ;
			do_action( 'qts_delete_quote_request', $id ) ;
		} else {
			wp_trash_post( $id ) ;
			$quote_request->set_status( 'trash' ) ;
			do_action( 'qts_trash_quote_request', $id ) ;
		}
	}

	/*
	  |--------------------------------------------------------------------------
	  | Additional Methods
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Get the quote request status to save to the post object.
	 *
	 * @param  QTS_Quote_Request $quote_request object.
	 * @return string
	 */
	protected function get_post_status( $quote_request ) {
		$post_status = $quote_request->get_status( 'edit' ) ;

		if ( ! $post_status ) {
			$post_status = apply_filters( 'qts_default_quote_request_status', 'new' ) ;
		}

		if ( in_array( QTS_PREFIX . $post_status, $quote_request->get_valid_statuses() ) ) {
			$post_status = QTS_PREFIX . $post_status ;
		}

		return $post_status ;
	}

	/**
	 * Excerpt for post.
	 *
	 * @param  QTS_Quote_Request $quote_request object.
	 * @return string
	 */
	protected function get_post_excerpt( $quote_request ) {
		return $quote_request->get_customer_note() ;
	}

	/**
	 * Add line items to the quote request.
	 *
	 * @param QTS_Quote_Request $quote_request
	 * @param array $quotes
	 */
	public function create_line_items( &$quote_request, $quotes ) {
		foreach ( $quotes as $quote_item_key => $values ) {

			$item = new WC_Order_Item_Product() ;
			$item->set_props(
					array(
						'quantity'     => $values[ 'quantity' ],
						'variation'    => $values[ 'variation' ],
						'subtotal'     => $values[ 'line_subtotal' ],
						'total'        => isset( $values[ 'line_total' ] ) ? $values[ 'line_total' ] : $values[ 'line_subtotal' ],
						'subtotal_tax' => isset( $values[ 'line_subtotal_tax' ] ) ? $values[ 'line_subtotal_tax' ] : 0,
						'total_tax'    => isset( $values[ 'line_tax' ] ) ? $values[ 'line_tax' ] : 0,
						'taxes'        => isset( $values[ 'line_tax_data' ] ) ? $values[ 'line_tax_data' ] : array(),
					)
			) ;

			$product = $values[ 'data' ] ;
			if ( $product ) {
				$item->set_props(
						array(
							'name'         => $product->get_name(),
							'tax_class'    => $product->get_tax_class(),
							'product_id'   => $product->is_type( 'variation' ) ? $product->get_parent_id() : $product->get_id(),
							'variation_id' => $product->is_type( 'variation' ) ? $product->get_id() : 0,
						)
				) ;
			}

			// Set quote item metadata.
			$meta_data   = array() ;
			$meta_data[] = array(
				'id'    => '',
				'key'   => '_is_qts_quote_request',
				'value' => 'yes'
					) ;

			// Get the preferred price
			if ( isset( $values[ 'qts_quote' ] ) ) {
				$preferred_price = _qts_calculate_preferred_price( $values[ 'qts_quote' ] ) ;
			} else {
				$preferred_price = _qts_calculate_preferred_price( $values ) ;
			}

			if ( is_numeric( $preferred_price ) ) {
				if ( isset( $values[ 'qts_quote' ] ) ) {
					$values[ 'qts_quote' ][ 'requested_price' ] = $preferred_price ;
				} else {
					$values[ 'requested_price' ] = $preferred_price ;
				}
			}

			$metakeys = array( 'original_price', 'requested_price', 'requested_price_percent', 'requested_discount_percent' ) ;
			foreach ( $metakeys as $key ) {
				if ( isset( $values[ 'qts_quote' ][ $key ] ) ) {
					$meta_data[] = array(
						'id'    => '',
						'key'   => "_{$key}",
						'value' => $values[ 'qts_quote' ][ $key ],
							) ;
				} else if ( isset( $values[ $key ] ) ) {
					$meta_data[] = array(
						'id'    => '',
						'key'   => "_{$key}",
						'value' => $values[ $key ],
							) ;
				}
			}

			$item->set_meta_data( $meta_data ) ;

			/**
			 * Action hook to adjust item before save.
			 */
			do_action( 'qts_create_quote_request_line_item', $item, $quote_item_key, $values, $quote_request ) ;

			// Add item to quote request.
			$quote_request->add_item( $item ) ;
		}
	}

	/**
	 * Read quote request data.
	 *
	 * @param QTS_Quote_Request $quote_request object.
	 * @param object   $post_object Post object.
	 */
	protected function read_order_data( &$quote_request, $post_object ) {
		parent::read_order_data( $quote_request, $post_object ) ;

		foreach ( $this->quote_request_internal_meta_key_to_props as $meta_key => $prop ) {
			$setter = "set_$prop" ;
			if ( is_callable( array( $quote_request, $setter ) ) ) {
				$quote_request->{$setter}( get_post_meta( $quote_request->get_id(), "$meta_key", true ) ) ;
			}
		}
	}

	/**
	 * Update meta data in, or delete it from, the database.
	 * As WC defined this method @since 3.6.0 and so we reuse this method here to make it compatible with v3.5.x too.
	 *
	 * @param WC_Data $object The WP_Data object
	 * @param string  $meta_key Meta key to update.
	 * @param mixed   $meta_value Value to save.
	 * @return bool True if updated/deleted.
	 */
	protected function update_or_delete_post_meta( $object, $meta_key, $meta_value ) {
		if ( is_callable( array( get_parent_class( $this ), 'update_or_delete_post_meta' ) ) ) {
			$updated = parent::update_or_delete_post_meta( $object, $meta_key, $meta_value ) ;
		} else {
			if ( in_array( $meta_value, array( array(), '' ), true ) ) {
				$updated = delete_post_meta( $object->get_id(), $meta_key ) ;
			} else {
				$updated = update_post_meta( $object->get_id(), $meta_key, $meta_value ) ;
			}
		}

		return ( bool ) $updated ;
	}

	/**
	 * Updates all the post meta for an quote request based on it's settings in the WC_Order class.
	 *
	 * @param QTS_Quote_Request $quote_request object.
	 */
	protected function update_post_meta( &$quote_request ) {
		parent::update_post_meta( $quote_request ) ;

		$updated_props   = array() ;
		$props_to_update = $this->get_props_to_update( $quote_request, $this->quote_request_internal_meta_key_to_props ) ;

		foreach ( $props_to_update as $meta_key => $prop ) {
			$getter = "get_$prop" ;
			$value  = is_callable( array( $quote_request, $getter ) ) ? $quote_request->{$getter}( 'edit' ) : '' ;
			$value  = is_string( $value ) ? wp_slash( $value ) : $value ;

			switch ( $prop ) {
				case 'date_expiry':
					$value = ! is_null( $value ) ? $value->getTimestamp() : '' ;
					break ;
			}

			$updated = $this->update_or_delete_post_meta( $quote_request, $meta_key, $value ) ;

			if ( $updated ) {
				$updated_props[] = $prop ;
			}
		}

		do_action( 'qts_quote_request_object_updated_props', $quote_request, $updated_props ) ;
	}

	/**
	 * Return count of quote request with a specific status.
	 *
	 * @param  string $status Quote Request status.
	 * @return int
	 */
	public function get_order_count( $status ) {
		global $wpdb ;
		return absint( $wpdb->get_var( $wpdb->prepare( "SELECT COUNT( * ) FROM {$wpdb->posts} WHERE post_type = 'qts_quote_request' AND post_status = %s", $status ) ) ) ;
	}

}
