<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Manage Quote Request products in WC Cart.
 * 
 * @class QTS_Cart
 */
class QTS_Cart {

	/**
	 * The single instance of the class.
	 */
	protected static $instance = null;

	/**
	 * Error handling codes.
	 */
	const CANNOT_ADD_PRODUCTS_WHILE_ASKING_FOR_QUOTE_IN_CART    = 501;
	const CANNOT_ADD_PRODUCTS_WHILE_PAYMENT_IS_AWAITING_IN_CART = 502;
	const CANNOT_ADD_QUOTE_WHILE_PRODUCTS_IN_CART               = 503;
	const CANNOT_ADD_QUOTE_WHILE_PAYMENT_IS_AWAITING_IN_CART    = 504;
	const INVALID_QUOTE_PRODUCTS_REMOVED_FROM_CART              = 505;
	const FORCE_LOGIN_TO_ADD_TO_QUOTE_FOR_QUEST_USERS           = 506;

	/**
	 * Create instance for QTS_Cart.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Construct the QTS_Cart.
	 */
	public function __construct() {
		// Manages Localization & HTML fieldset.
		if ( 'yes' === get_option( QTS_PREFIX . 'display_request_price_field' ) ) {
			add_filter( 'woocommerce_cart_item_price', array( __CLASS__, 'render_ask_for_price_field' ), 10, 3 );
		}

		add_action( 'woocommerce_cart_collaterals', array( __CLASS__, 'render_customer_note_field' ) );
		add_action( 'woocommerce_after_checkout_billing_form', array( __CLASS__, 'render_customer_note_field' ) );
		add_filter( 'woocommerce_checkout_fields', array( __CLASS__, 'add_customer_note_fieldset_to_checkout' ) );
		add_action( 'woocommerce_before_cart_table', array( __CLASS__, 'print_notice' ), 5 );
		add_action( 'woocommerce_checkout_before_customer_details', array( __CLASS__, 'print_notice' ), 5 );
		add_filter( 'woocommerce_order_button_text', array( __CLASS__, 'get_place_order_text' ), 20 );

		// Manage Checkout Fields.
		add_filter( 'woocommerce_checkout_fields', array( __CLASS__, 'prepare_to_replace_billing_email_input_field' ), 99 );

		// Manages Item Qty.
		add_filter( 'woocommerce_product_backorders_allowed', array( __CLASS__, 'maybe_allow_backorders' ), 999 );
		add_filter( 'woocommerce_quantity_input_max', array( __CLASS__, 'maybe_do_not_limit_qty' ), 999 );
		add_filter( 'woocommerce_cart_item_quantity', array( __CLASS__, 'set_qty_input_as_non_editable' ), 999, 2 );

		// Manages Item Removal.
		add_action( 'woocommerce_cart_item_removed', array( __CLASS__, 'when_an_item_is_removed' ), 999 );

		// Manages Cart Validation.
		add_filter( 'woocommerce_add_to_cart_validation', array( __CLASS__, 'validate_add_to_cart' ), 999, 4 );
		add_action( 'woocommerce_cart_loaded_from_session', array( __CLASS__, 'validate_session' ), 999 );

		// Manages Quote in Cart Session
		add_filter( 'woocommerce_add_cart_item_data', array( __CLASS__, 'add_quote_item_data' ), 99, 4 );
		add_action( 'woocommerce_cart_loaded_from_session', array( __CLASS__, 'get_quote_for_session' ), 99999 );
		add_filter( 'woocommerce_coupon_is_valid', array( __CLASS__, 'restrict_coupon_usage' ), 99 );
	}

	/**
	 * Check if the item added as quote is in the cart and return quote data.
	 * 
	 * @param mixed $item Cart item array or Product ID.
	 * @return boolean|array False on failure
	 */
	public static function find_item_added_as_quote( $item ) {
		if ( is_array( $item ) ) {
			if ( isset( $item[ 'qts_quote' ] ) && ! empty( $item[ 'qts_quote' ] ) ) {
				return $item[ 'qts_quote' ];
			}
		} else if ( is_string( $item ) ) {
			if ( isset( WC()->cart->cart_contents[ $item ][ 'qts_quote' ] ) && ! empty( WC()->cart->cart_contents[ $item ][ 'qts_quote' ] ) ) {
				return WC()->cart->cart_contents[ $item ][ 'qts_quote' ];
			}
		} else {
			$product_id = false;
			if ( is_callable( array( $item, 'get_id' ) ) ) {
				$product_id = $item->get_id();
			} else if ( is_numeric( $item ) ) {
				$product_id = $item;
			}

			if ( $product_id && ! empty( WC()->cart->cart_contents ) ) {
				foreach ( WC()->cart->cart_contents as $cart_item ) {
					if (
							isset( $cart_item[ 'qts_quote' ] ) &&
							( $cart_item[ 'variation_id' ] > 0 ? $cart_item[ 'variation_id' ] : $cart_item[ 'product_id' ] ) == $product_id
					) {
						return $cart_item[ 'qts_quote' ];
					}
				}
			}
		}

		return false;
	}

	/**
	 * Check whether the cart is prepared and it is asking for quote.
	 * 
	 * @return boolean
	 */
	public static function asking_for_quote() {
		if ( ! empty( WC()->cart->cart_contents ) ) {
			foreach ( WC()->cart->cart_contents as $cart_item ) {
				$added_quote = self::find_item_added_as_quote( $cart_item );

				if ( $added_quote && ! isset( $added_quote[ 'payment' ] ) ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Check whether the cart is prepared and it is awaiting for payment via checkout for accepted/approved quote request.
	 * 
	 * @return bool|int False on failure
	 */
	public static function awaiting_for_payment() {
		if ( ! empty( WC()->cart->cart_contents ) ) {
			foreach ( WC()->cart->cart_contents as $cart_item ) {
				$added_quote = self::find_item_added_as_quote( $cart_item );

				if ( $added_quote && isset( $added_quote[ 'payment' ] ) ) {
					return $added_quote;
				}
			}
		}

		return false;
	}

	/**
	 * Browse the cart and get the added quote list.
	 * 
	 * @param string $context
	 * @param mixed $value
	 * @return array quote data
	 */
	public static function browse_added_quote_list( $context = null, $value = null ) {
		$browsed_list = array();

		if ( ! empty( WC()->cart->cart_contents ) ) {
			foreach ( WC()->cart->cart_contents as $item_key => $cart_item ) {
				$added_quote = self::find_item_added_as_quote( $cart_item );

				if ( $added_quote ) {
					if ( ! is_null( $context ) ) {
						if ( isset( $added_quote[ $context ] ) && $value === $added_quote[ $context ] ) {
							$browsed_list[ $item_key ] = $added_quote;
						}
					} else {
						$browsed_list[ $item_key ] = $added_quote;
					}
				}
			}
		}

		return $browsed_list;
	}

	/**
	 * Add the error notice to customer.
	 * 
	 * @param int $code
	 */
	public static function add_error_notice( $code ) {
		switch ( $code ) {
			case 501:
				wc_add_notice( trim( get_option( QTS_PREFIX . 'cannot_add_product_while_getting_a_quote_err' ) ), 'error' );
				break;
			case 502:
				wc_add_notice( trim( get_option( QTS_PREFIX . 'cannot_add_product_while_making_payment_err' ) ), 'error' );
				break;
			case 503:
				wc_add_notice( trim( get_option( QTS_PREFIX . 'cannot_add_quote_while_normal_products_in_cart_err' ) ), 'error' );
				break;
			case 504:
				wc_add_notice( trim( get_option( QTS_PREFIX . 'cannot_add_quote_while_making_payment_err' ) ), 'error' );
				break;
			case 505:
				wc_add_notice( trim( get_option( QTS_PREFIX . 'invalid_quote_products_removed_err' ) ), 'error' );
				break;
			case 506:
				wc_add_notice( trim( __( 'Sorry you are not eligible to request for quote. Please log in and add to quote.', 'quote-request-for-woocommerce' ) ), 'error' );
				break;
		}
	}

	/**
	 * Return the array of requested.
	 * 
	 * @return array
	 */
	public static function get_request() {
		$adding_quote = null;

		if ( isset( $_REQUEST[ 'adding_qtsquote' ] ) ) {
			$adding_quote = wc_clean( wp_unslash( $_REQUEST[ 'adding_qtsquote' ] ) );
		}

		return array(
			'adding_quote' => $adding_quote,
		);
	}

	/**
	 * Allow as backorder on demand.
	 * 
	 * @param bool $bool
	 * @return bool
	 */
	public static function maybe_allow_backorders( $bool ) {
		if ( ! _qts_allow_unlimited_qty() ) {
			return $bool;
		}

		if ( isset( $_REQUEST[ 'adding_qtsquote' ] ) || isset( $_REQUEST[ 'add-to-qtsquote' ] ) ) {
			$bool = true;
		}

		if ( 'yes' === get_option( QTS_PREFIX . 'make_payment_when_not_having_enough_stock', 'no' ) && isset( $_GET[ 'qtscheckout' ], $_GET[ 'qtsquote_request' ] ) ) {
			$bool = true;
		}

		if ( is_cart() || is_checkout() ) {
			if ( self::asking_for_quote() ) {
				$bool = true;
			} else if ( 'yes' === get_option( QTS_PREFIX . 'make_payment_when_not_having_enough_stock', 'no' ) && self::awaiting_for_payment() ) {
				$bool = true;
			}
		}

		return $bool;
	}

	/**
	 * Do not limit quantities in cart page only.
	 * 
	 * @param int $qty
	 * @return int
	 */
	public static function maybe_do_not_limit_qty( $qty ) {
		if ( ( is_cart() || is_checkout() ) && _qts_allow_unlimited_qty() && self::asking_for_quote() ) {
			$qty = -1;
		}

		return $qty;
	}

	/**
	 * Set the WC quantity input as non editable fields when the quote is awaiting for payment.
	 * 
	 * @param string $quantity
	 * @param string $cart_item_key
	 * @return string
	 */
	public static function set_qty_input_as_non_editable( $quantity, $cart_item_key ) {
		$added_quote = self::find_item_added_as_quote( $cart_item_key );

		if ( $added_quote && isset( $added_quote[ 'payment' ] ) ) {
			$requested_qty = is_numeric( $added_quote[ 'qty' ] ) ? absint( $added_quote[ 'qty' ] ) : 1;
			$quantity      = sprintf( '%1$s <input type="hidden" name="cart[%2$s][qty]" value="%3$s" />', $requested_qty, $cart_item_key, $requested_qty );
		}

		return $quantity;
	}

	/**
	 * Ensure whether the cart is awaiting for payment if so, empty the cart when single item gets removed by the user.
	 * 
	 * @param string $cart_item_key
	 */
	public static function when_an_item_is_removed( $cart_item_key ) {
		if ( self::awaiting_for_payment() ) {
			WC()->cart->empty_cart();
		}
	}

	/**
	 * Render the ask for price input field.
	 * 
	 * @param string $price
	 * @param array $cart_item
	 * @param string $item_key
	 * @return string
	 */
	public static function render_ask_for_price_field( $price, $cart_item, $item_key ) {
		$added_quote = self::find_item_added_as_quote( $cart_item );

		if ( $added_quote && ! isset( $added_quote[ 'payment' ] ) ) {
			$original_price             = is_numeric( $added_quote[ 'original_price' ] ) ? wc_format_decimal( $added_quote[ 'original_price' ], wc_get_price_decimals() ) : '0';
			$requested_price            = is_numeric( $added_quote[ 'requested_price' ] ) ? wc_format_decimal( $added_quote[ 'requested_price' ], wc_get_price_decimals() ) : '';
			$requested_price_percent    = isset( $added_quote[ 'requested_price_percent' ] ) && is_numeric( $added_quote[ 'requested_price_percent' ] ) ? floatval( $added_quote[ 'requested_price_percent' ] ) : '';
			$requested_discount_percent = isset( $added_quote[ 'requested_discount_percent' ] ) && is_numeric( $added_quote[ 'requested_discount_percent' ] ) ? floatval( $added_quote[ 'requested_discount_percent' ] ) : '';
			$request_price_type         = get_option( QTS_PREFIX . 'request_price_field_type', 'fixed' );
			$step                       = _qts_get_number_input_step_value();

			$field = wc_price( $original_price );
			$field .= '<p>';

			if ( 'percent' === $request_price_type ) {
				$field .= "<label for='preferred-price'>";
				$field .= __( 'Your Preferred Price: ', 'quote-request-for-woocommerce' );
				$field .= '</label>';
				$field .= "<input type='number' id='qts_requested_price_percent' name='cart[{$item_key}][qts_requested_price_percent]' value='{$requested_price_percent}' min='0' max='100' step='{$step}'>%";
			} else if ( 'percent-discount' === $request_price_type ) {
				$field .= "<label for='preferred-discount'>";
				$field .= __( 'Your Preferred Discount: ', 'quote-request-for-woocommerce' );
				$field .= '</label>';
				$field .= "<input type='number' id='qts_requested_discount_percent' name='cart[{$item_key}][qts_requested_discount_percent]' value='{$requested_discount_percent}' min='0' max='100' step='{$step}'>%";
			} else {
				$field .= "<label for='preferred-price'>";
				$field .= __( 'Your Preferred Price: ', 'quote-request-for-woocommerce' );
				$field .= '</label>';
				$field .= "<input type='number' id='qts_requested_price' name='cart[{$item_key}][qts_requested_price]' value='{$requested_price}' min='0' max='{$original_price}' step='{$step}'>";
			}

			$field .= '</p>';
			$price = $field;
		}

		return $price;
	}

	/**
	 * Return the array of fieldset to be used for customer quote note.
	 */
	public static function get_customer_note_fieldset() {
		$fieldset                = array();
		$fieldset[ 'field_key' ] = 'qts_customer_note';
		$fieldset[ 'field' ]     = array(
			'type'        => $fieldset[ 'field_key' ],
			'class'       => array( 'form-row-wide' ),
			'label'       => __( 'Customer Note', 'quote-request-for-woocommerce' ),
			'placeholder' => esc_attr__( 'Notes about your quote.', 'quote-request-for-woocommerce' ),
			'default'     => WC()->session->get( QTS_PREFIX . 'customer_note', '' ),
		);

		return $fieldset;
	}

	/**
	 * Include the prepared customer quote note fieldset in the array of checkout fields.
	 * 
	 * @param array $fields
	 * @return array
	 */
	public static function add_customer_note_fieldset_to_checkout( $fields ) {
		if ( self::asking_for_quote() ) {
			$note_fieldset                                                 = self::get_customer_note_fieldset();
			$fields[ 'qtsquote_request' ][ $note_fieldset[ 'field_key' ] ] = $note_fieldset[ 'field' ];
		}

		return $fields;
	}

	/**
	 * Render the customer note to quote input field in cart/checkout page.
	 */
	public static function render_customer_note_field() {
		/**
		 * Need customer note field?
		 * 
		 * @since 1.0
		 */
		if ( self::asking_for_quote() && apply_filters( 'qts_show_customer_note_field', true ) ) {
			$note_fieldset = self::get_customer_note_fieldset();
			woocommerce_form_field( $note_fieldset[ 'field_key' ], $note_fieldset[ 'field' ], QTS_Form_Fields::get_value( $note_fieldset[ 'field_key' ] ) );
		}
	}

	/**
	 * Notice customer to pay the quote via checkout.
	 */
	public static function print_notice() {
		if ( is_cart() || is_checkout() ) {
			$added_quote = self::awaiting_for_payment();

			if ( $added_quote && is_numeric( $added_quote[ 'payment' ] ) ) {
				wc_print_notice( trim( str_replace( '[quote_request_number]', absint( $added_quote[ 'payment' ] ), get_option( QTS_PREFIX . 'complete_payment_msg' ) ) ), 'success' );
			}
		}
	}

	/**
	 * Localized place order button when the cart contains quote requests.
	 * 
	 * @param string $order_button_text
	 * @return string
	 */
	public static function get_place_order_text( $order_button_text ) {
		if ( self::asking_for_quote() ) {
			$order_button_text = get_option( QTS_PREFIX . 'submit_quote_label', esc_html__( 'Submit Quote', 'quote-request-for-woocommerce' ) );
		}

		return $order_button_text;
	}

	/**
	 * Prepare to replace the billing email input field with the email text in checkout.
	 * Email will be replaced by the one which is used for their quote request submission.
	 * 
	 * @param array $fields
	 * @return array
	 */
	public static function prepare_to_replace_billing_email_input_field( $fields ) {
		if ( isset( $fields[ 'billing' ][ 'billing_email' ] ) ) {
			$added_quote = self::awaiting_for_payment();

			if ( $added_quote && is_numeric( $added_quote[ 'payment' ] ) ) {
				$quote_request_id = absint( $added_quote[ 'payment' ] );
				$quote_request    = _qts_get_quote_request( $quote_request_id );

				// Set the email adddress which is used for their quote request submission.
				$fields[ 'billing' ][ 'billing_email' ][ 'qtsquote_request' ][ 'email' ] = $quote_request->get_billing_email();
			}
		}

		return $fields;
	}

	/**
	 * Validate while adding the product to cart.
	 * 
	 * @param bool $bool
	 * @param int $product_id
	 * @param int $quantity
	 * @param int $variation_id
	 * @param array $variations
	 * @param array $cart_item_data
	 * @return boolean
	 */
	public static function validate_add_to_cart( $bool, $product_id, $quantity, $variation_id = null ) {
		$add_to_quote_product = $variation_id ? $variation_id : $product_id;
		$requested            = self::get_request();

		if ( is_null( $requested[ 'adding_quote' ] ) ) {
			if ( self::awaiting_for_payment() ) {
				self::add_error_notice( self::CANNOT_ADD_PRODUCTS_WHILE_PAYMENT_IS_AWAITING_IN_CART );
				return false;
			}

			if ( self::asking_for_quote() ) {
				self::add_error_notice( self::CANNOT_ADD_PRODUCTS_WHILE_ASKING_FOR_QUOTE_IN_CART );
				return false;
			}
		} else {
			QTS_Add_To_Quote::prepare_props( $add_to_quote_product );

			if ( QTS_Add_To_Quote::_get( $add_to_quote_product, 'enabled' ) ) {
				if ( QTS_Add_To_Quote::_get( $add_to_quote_product, 'do_not_add' ) ) {
					self::add_error_notice( self::FORCE_LOGIN_TO_ADD_TO_QUOTE_FOR_QUEST_USERS );
					return false;
				}
			}

			if ( empty( WC()->cart->cart_contents ) ) {
				return $bool;
			}

			remove_action( 'woocommerce_cart_loaded_from_session', array( __CLASS__, 'validate_session' ), 999 );

			if ( QTS_Add_To_Quote::_get( $add_to_quote_product, 'enabled' ) ) {
				if ( self::awaiting_for_payment() ) {
					self::add_error_notice( self::CANNOT_ADD_QUOTE_WHILE_PAYMENT_IS_AWAITING_IN_CART );
					return false;
				}

				if ( ! self::asking_for_quote() ) {
					self::add_error_notice( self::CANNOT_ADD_QUOTE_WHILE_PRODUCTS_IN_CART );
					return false;
				}
			} else {
				if ( self::awaiting_for_payment() ) {
					self::add_error_notice( self::CANNOT_ADD_PRODUCTS_WHILE_PAYMENT_IS_AWAITING_IN_CART );
					return false;
				}

				if ( self::asking_for_quote() ) {
					self::add_error_notice( self::CANNOT_ADD_PRODUCTS_WHILE_ASKING_FOR_QUOTE_IN_CART );
					return false;
				}
			}

			add_action( 'woocommerce_cart_loaded_from_session', array( __CLASS__, 'validate_session' ), 999 );
		}

		return $bool;
	}

	/**
	 * Validate the quote requests in the cart session.
	 */
	public static function validate_session() {
		if ( empty( WC()->cart->cart_contents ) ) {
			return;
		}

		$browsed_list = array_keys( self::browse_added_quote_list() );

		if ( ! empty( $browsed_list ) && count( WC()->cart->cart_contents ) !== count( $browsed_list ) ) {
			array_map( array( WC()->cart, 'remove_cart_item' ), $browsed_list );
			self::add_error_notice( self::INVALID_QUOTE_PRODUCTS_REMOVED_FROM_CART );
		}
	}

	/**
	 * Add the quoted item data to the cart item.
	 * 
	 * @param array $cart_item_data
	 * @param int $product_id
	 * @param int $variation_id
	 * @param string $quantity
	 * @return array
	 */
	public static function add_quote_item_data( $cart_item_data, $product_id, $variation_id, $quantity ) {
		$add_to_quote_product = $variation_id ? $variation_id : $product_id;
		$requested            = self::get_request();

		if ( is_null( $requested[ 'adding_quote' ] ) || ! QTS_Add_To_Quote::_get( $add_to_quote_product, 'enabled' ) ) {
			return $cart_item_data;
		}

		$product = wc_get_product( $add_to_quote_product );
		if ( ! $product ) {
			return $cart_item_data;
		}

		/**
		 * Get the quote item data while request for quote.
		 * 
		 * @since 1.0
		 */
		$cart_item_data[ 'qts_quote' ] = apply_filters( 'qts_add_quote_item_data_to_cart', array(
			'original_price'             => _qts_get_original_price( $product ),
			'requested_price'            => null,
			'requested_price_percent'    => null,
			'requested_discount_percent' => null,
			'product_options'            => QTS_Add_To_Quote::_gets( $product->get_id() ),
				), $cart_item_data, $product_id, $variation_id, $quantity );

		return $cart_item_data;
	}

	/**
	 * Get the quote data from the cart session before the cart calculating totals.
	 */
	public static function get_quote_for_session() {
		$quote_totals = isset( $_REQUEST[ 'cart' ] ) ? wc_clean( wp_unslash( $_REQUEST[ 'cart' ] ) ) : '';

		foreach ( WC()->cart->get_cart() as $item_key => $cart_item ) {
			if ( empty( $cart_item[ 'qts_quote' ] ) ) {
				continue;
			}

			$product = wc_get_product( $cart_item[ 'variation_id' ] > 0 ? $cart_item[ 'variation_id' ] : $cart_item[ 'product_id' ] );
			if ( ! $product ) {
				continue;
			}

			// Set admin offered price if it is awaiting for payment.
			if ( isset( $cart_item[ 'qts_quote' ][ 'payment' ] ) ) {
				WC()->cart->cart_contents[ $item_key ][ 'data' ]->set_price( $cart_item[ 'qts_quote' ][ 'offered_price' ] );
				continue;
			}

			if ( ! QTS_Add_To_Quote::is_quote_enabled( $product ) ) {
				unset( WC()->cart->cart_contents[ $item_key ][ 'qts_quote' ] );
				continue;
			}

			// Check for exsiting values
			$requested_price            = $cart_item[ 'qts_quote' ][ 'requested_price' ];
			$requested_price_percent    = isset( $cart_item[ 'qts_quote' ][ 'requested_price_percent' ] ) ? $cart_item[ 'qts_quote' ][ 'requested_price_percent' ] : null;
			$requested_discount_percent = isset( $cart_item[ 'qts_quote' ][ 'requested_discount_percent' ] ) ? $cart_item[ 'qts_quote' ][ 'requested_discount_percent' ] : null;

			// Check for posted values
			if ( ! empty( $quote_totals ) ) {
				if ( isset( $quote_totals[ $item_key ][ 'qts_requested_price' ] ) && is_numeric( $quote_totals[ $item_key ][ 'qts_requested_price' ] ) ) {
					$requested_price = wc_format_decimal( $quote_totals[ $item_key ][ 'qts_requested_price' ] );
				} else if ( isset( $quote_totals[ $item_key ][ 'qts_requested_price_percent' ] ) && is_numeric( $quote_totals[ $item_key ][ 'qts_requested_price_percent' ] ) ) {
					$requested_price_percent = floatval( $quote_totals[ $item_key ][ 'qts_requested_price_percent' ] );
				} else if ( isset( $quote_totals[ $item_key ][ 'qts_requested_discount_percent' ] ) && is_numeric( $quote_totals[ $item_key ][ 'qts_requested_discount_percent' ] ) ) {
					$requested_discount_percent = floatval( $quote_totals[ $item_key ][ 'qts_requested_discount_percent' ] );
				}
			}

			/**
			 * Update quote validation.
			 * 
			 * @since 1.0
			 */
			$passed_validation = apply_filters( 'qts_update_quote_validation', true, $item_key, $cart_item, $cart_item[ 'quantity' ] );

			// Check whether the requested price less than the minimum price to ask for quote.
			if ( is_numeric( $requested_price ) ) {
				$min_quote_price = _qts_get_min_quote_price( $product );

				if ( $min_quote_price > 0 && floatval( $requested_price ) < $min_quote_price ) {
					wc_add_notice( trim( str_replace( array( '[product_name]', '[minimum_price]' ), array( $product->get_name(), wc_price( $min_quote_price ) ), get_option( QTS_PREFIX . 'cannot_request_less_than_min_price_err' ) ) ), 'error' );
					$passed_validation = false;
				}
			} else if ( is_numeric( $requested_price_percent ) ) {
				$min_quote_price_percent = _qts_get_min_quote_price_percent( $product );

				if ( $min_quote_price_percent > 0 && floatval( $requested_price_percent ) < $min_quote_price_percent ) {
					wc_add_notice( trim( str_replace( array( '[product_name]', '[minimum_price_percent]' ), array( $product->get_name(), "{$min_quote_price_percent}%" ), get_option( QTS_PREFIX . 'cannot_request_less_than_min_price_percent_err' ) ) ), 'error' );
					$passed_validation = false;
				}
			} else if ( is_numeric( $requested_discount_percent ) ) {
				if ( floatval( $requested_discount_percent ) < 0 || floatval( $requested_discount_percent ) > 100 ) {
					$passed_validation = false;
				}
			}

			if ( ! $passed_validation ) {
				continue;
			}

			// Prepare the prices.
			WC()->cart->cart_contents[ $item_key ][ 'qts_quote' ] = array(
				'original_price'             => _qts_get_original_price( $product ),
				'requested_price'            => $requested_price,
				'requested_price_percent'    => $requested_price_percent,
				'requested_discount_percent' => $requested_discount_percent,
				'product_options'            => QTS_Add_To_Quote::_gets( $product->get_id() ),
			);

			$preferred_price = _qts_calculate_preferred_price( WC()->cart->cart_contents[ $item_key ][ 'qts_quote' ] );
			if ( is_numeric( $preferred_price ) ) {
				WC()->cart->cart_contents[ $item_key ][ 'data' ]->set_price( $preferred_price );
			} else {
				WC()->cart->cart_contents[ $item_key ][ 'data' ]->set_price( WC()->cart->cart_contents[ $item_key ][ 'qts_quote' ][ 'original_price' ] );
			}
		}
	}

	/**
	 * Restrict coupon usage when the cart is requesting for quote.
	 * 
	 * @param bool $bool
	 * @return bool
	 */
	public static function restrict_coupon_usage( $bool ) {
		if ( self::asking_for_quote() ) {
			throw new Exception( trim( get_option( QTS_PREFIX . 'cannot_use_coupon_while_getting_a_quote_err' ) ), 100 );
		}

		return $bool;
	}

}

QTS_Cart::instance();
