<?php

defined( 'ABSPATH' ) || exit;

include_once( 'class-qts-quote-session.php' );
include_once( 'class-qts-quote-totals.php' );

/**
 * QTS Quote
 *
 * @class QTS_Quote
 * @package Classes
 */
class QTS_Quote {

	/**
	 * Contains an array of quote items.
	 *
	 * @var array
	 */
	public $quote_contents = array();

	/**
	 * Contains an array of removed quote items so we can restore them if needed.
	 *
	 * @var array
	 */
	public $removed_quote_contents = array();

	/**
	 * Total defaults used to reset.
	 *
	 * @var array
	 */
	protected $default_totals = array(
		'quote_contents_total' => 0,
		'subtotal'             => 0,
	);

	/**
	 * Calculated totals.
	 *
	 * @var array
	 */
	protected $totals = array();

	/**
	 * Reference to the quote session handling class.
	 *
	 * @var QTS_Quote_Session
	 * @uses WC Session
	 */
	protected $session;

	/**
	 * Constructor for the quote class
	 */
	public function __construct() {
		$this->session = new QTS_Quote_Session( $this );
		$this->session->init();

		add_action( 'qts_add_to_quote', array( $this, 'calculate_totals' ), 20, 0 );
		add_action( 'qts_quote_item_removed', array( $this, 'calculate_totals' ), 20, 0 );
		add_action( 'qts_quote_item_restored', array( $this, 'calculate_totals' ), 20, 0 );
		add_action( 'qts_quote_request_created', array( $this, 'empty_quote' ) );
	}

	/*
	  |--------------------------------------------------------------------------
	  | Setters and Getters.
	  |--------------------------------------------------------------------------
	  |
	  | Methods to retrieve class properties and avoid direct access.
	 */

	/**
	 * Gets quote contents.
	 *
	 * @return array of quote items
	 */
	public function get_quote_contents() {
		/**
		 * Get quote contents.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_get_quote_contents', ( array ) $this->quote_contents );
	}

	/**
	 * Return items removed from the quote.
	 *
	 * @return array
	 */
	public function get_removed_quote_contents() {
		return ( array ) $this->removed_quote_contents;
	}

	/**
	 * Return all calculated totals.
	 *
	 * @return array
	 */
	public function get_totals() {
		return empty( $this->totals ) ? $this->default_totals : $this->totals;
	}

	/**
	 * Get a total.
	 *
	 * @param string $key Key of element in $totals array.
	 * @return mixed
	 */
	protected function get_totals_var( $key ) {
		return isset( $this->totals[ $key ] ) ? $this->totals[ $key ] : $this->default_totals[ $key ];
	}

	/**
	 * Gets quote total. This is the total of items in the quote.
	 *
	 * @return float
	 */
	public function get_quote_contents_total() {
		return $this->get_totals_var( 'quote_contents_total' );
	}

	/**
	 * Get subtotal.
	 *
	 * @param string $context
	 * @return float
	 */
	public function get_subtotal( $context = 'view' ) {
		$subtotal = $this->get_totals_var( 'subtotal' );

		return 'view' === $context ? wc_price( $subtotal ) : $subtotal;
	}

	/**
	 * Sets the contents of the quote.
	 *
	 * @param array $value Quote array.
	 */
	public function set_quote_contents( $value ) {
		$this->quote_contents = ( array ) $value;
	}

	/**
	 * Set items removed from the quote.
	 *
	 * @param array $value Item array.
	 */
	public function set_removed_quote_contents( $value = array() ) {
		$this->removed_quote_contents = ( array ) $value;
	}

	/**
	 * Set all calculated totals.
	 *
	 * @param array $value Value to set.
	 */
	public function set_totals( $value = array() ) {
		$this->totals = wp_parse_args( $value, $this->default_totals );
	}

	/**
	 * Set quote contents_total.
	 *
	 * @param string $value Value to set.
	 */
	public function set_quote_contents_total( $value ) {
		$this->totals[ 'quote_contents_total' ] = wc_format_decimal( $value, wc_get_price_decimals() );
	}

	/**
	 * Set subtotal.
	 *
	 * @param string $value Value to set.
	 */
	public function set_subtotal( $value ) {
		$this->totals[ 'subtotal' ] = wc_format_decimal( $value, wc_get_price_decimals() );
	}

	/**
	 * Reset quote totals to the defaults. Useful before running calculations.
	 */
	private function reset_totals() {
		$this->totals = $this->default_totals;
	}

	/**
	 * Returns the contents of the quote in an array.
	 *
	 * @return array contents of the quote
	 */
	public function get_quote() {
		if ( ! did_action( 'wp_loaded' ) ) {
			return;
		}

		if ( ! did_action( 'qts_load_quote_from_session' ) ) {
			$this->session->get_quote_from_session();
		}

		return array_filter( $this->get_quote_contents() );
	}

	/**
	 * Returns a specific item in the quote.
	 *
	 * @param string $item_key Quote item key.
	 * @return array Item data
	 */
	public function get_quote_item( $item_key ) {
		return isset( $this->quote_contents[ $item_key ] ) ? $this->quote_contents[ $item_key ] : array();
	}

	/**
	 * Get quote items quantities.
	 *
	 * @return array
	 */
	public function get_quote_item_quantities() {
		$quantities = array();

		foreach ( $this->get_quote() as $values ) {
			$product                                           = $values[ 'data' ];
			$quantities[ $product->get_stock_managed_by_id() ] = isset( $quantities[ $product->get_stock_managed_by_id() ] ) ? $quantities[ $product->get_stock_managed_by_id() ] + $values[ 'quantity' ] : $values[ 'quantity' ];
		}

		return $quantities;
	}

	/**
	 * Get the item row subtotal.
	 *
	 * @param string $item_key Quote item key.
	 * @return string formatted price
	 */
	public function get_item_subtotal( $item_key ) {
		$quote_item = $this->get_quote_item( $item_key );

		if ( ! empty( $quote_item ) ) {
			$preferred_price = _qts_calculate_preferred_price( $quote_item );

			if ( is_numeric( $preferred_price ) ) {
				$item_subtotal = wc_price( $preferred_price * $quote_item[ 'quantity' ] );
			} else {
				$item_subtotal = wc_price( $quote_item[ 'original_price' ] * $quote_item[ 'quantity' ] );
			}
		} else {
			$item_subtotal = wc_price( 0 );
		}

		/**
		 * Get quote item subtotal.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_quote_item_subtotal', $item_subtotal, $item_key, $this );
	}

	/**
	 * Check all quote items for errors.
	 */
	public function check_quote_items( $display_error = true ) {
		$product_qty_in_quote = $this->get_quote_item_quantities();

		// Looks through quote items and checks the posts are not trashed or deleted and check each item is in stock.
		foreach ( $this->get_quote() as $quote_item_key => $values ) {
			$product = $values[ 'data' ];

			if ( ! $product || ! $product->exists() || 'trash' === $product->get_status() || ! QTS_Add_To_Quote::is_quote_enabled( $product ) ) {
				$this->set_quantity( $quote_item_key, 0 );

				if ( $display_error ) {
					wc_add_notice( __( 'An item which is no longer available was removed from your quote.', 'quote-request-for-woocommerce' ), 'error' );
				}
				return false;
			}

			if ( ! $product->is_in_stock() ) {
				if ( $display_error ) {
					wc_add_notice( trim( str_replace( '[product_name]', $product->get_name(), get_option( QTS_PREFIX . 'product_not_in_stock_err' ) ) ), 'error' );
				}
				return false;
			}

			// We only need to check products managing stock, with a limited stock qty.
			if ( ! $product->managing_stock() || $product->backorders_allowed() || _qts_allow_unlimited_qty() ) {
				continue;
			}

			// Check stock based on all items in the quote.
			$required_stock = $product_qty_in_quote[ $product->get_stock_managed_by_id() ];

			if ( $product->get_stock_quantity() < $required_stock ) {
				if ( $display_error ) {
					wc_add_notice( trim( str_replace( '[product_name]', $product->get_name(), get_option( QTS_PREFIX . 'insufficient_stock_while_making_payment_err' ) ) ), 'error' );
				}
				return false;
			}
		}

		return true;
	}

	/**
	 * Checks if the quote is empty.
	 *
	 * @return bool
	 */
	public function is_empty() {
		return 0 === count( $this->get_quote() );
	}

	/**
	 * Check if product is in the quote and return quote item key.
	 *
	 * @param mixed $quote_id id of product to find in the quote.
	 * @return string quote item key
	 */
	public function find_product_in_quote( $quote_id = false ) {
		if ( false !== $quote_id && is_array( $this->quote_contents ) && isset( $this->quote_contents[ $quote_id ] ) ) {
			return $quote_id;
		}

		return '';
	}

	/**
	 * Empties the quote.
	 */
	public function empty_quote() {
		$this->quote_contents         = array();
		$this->removed_quote_contents = array();
		$this->totals                 = $this->default_totals;

		/**
		 * Triggers after quote is emptied.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_quote_emptied' );
	}

	/**
	 * Add a product to the quote.
	 *
	 * @param int   $product_id contains the id of the product to add to the quote.
	 * @param int   $quantity contains the quantity of the item to add.
	 * @param int   $variation_id ID of the variation being added to the quote.
	 * @param array $variation attribute values.
	 * @param array $quote_item_data extra quote item data we want to pass into the item.
	 * @return string|bool $quote_item_key
	 */
	public function add_to_quote( $product_id = 0, $quantity = 1, $variation_id = 0, $variation = array(), $quote_item_data = array() ) {
		try {
			$product_id   = absint( $product_id );
			$variation_id = absint( $variation_id );

			if ( 'product_variation' === get_post_type( $product_id ) ) {
				$variation_id = $product_id;
				$product_id   = wp_get_post_parent_id( $variation_id );
			}

			$product_data = wc_get_product( $variation_id ? $variation_id : $product_id );

			/**
			 * Get the quantity to add.
			 * 
			 * @since 1.0
			 */
			$quantity = apply_filters( 'qts_add_to_quote_quantity', $quantity, $product_id );
			if ( $quantity <= 0 || ! $product_data || 'trash' === $product_data->get_status() || ! QTS_Add_To_Quote::is_quote_enabled( $product_data ) ) {
				return false;
			}

			/**
			 * Load quote item data - may be added by other plugins.
			 * 
			 * @since 1.0
			 */
			$quote_item_data = ( array ) apply_filters( 'qts_add_quote_item_data', $quote_item_data, $product_id, $variation_id, $quantity );
			$quote_id        = $this->generate_quote_id( $product_id, $variation_id, $variation, $quote_item_data );
			$quote_item_key  = $this->find_product_in_quote( $quote_id );

			if ( $product_data->is_sold_individually() ) {
				$quantity = 1;

				if ( $quote_item_key && $this->quote_contents[ $quote_item_key ][ 'quantity' ] > 0 ) {
					/* translators: %s: product name */
					throw new Exception( sprintf( '<a href="%s" class="button wc-forward">%s</a> %s', _qts_get_quote_url(), __( 'View quote', 'quote-request-for-woocommerce' ), sprintf( __( 'You cannot add another "%s" to your quote.', 'quote-request-for-woocommerce' ), $product_data->get_name() ) ) );
				}
			}

			if ( ! $product_data->is_purchasable() ) {
				throw new Exception( __( 'Sorry, this product cannot be purchased.', 'quote-request-for-woocommerce' ) );
			}

			if ( ! $product_data->is_in_stock() ) {
				/* translators: %s: product name */
				throw new Exception( sprintf( __( 'You cannot add &quot;%s&quot; to the quote because the product is out of stock.', 'quote-request-for-woocommerce' ), $product_data->get_name() ) );
			}

			if ( ! $product_data->has_enough_stock( $quantity ) ) {
				/* translators: 1: product name 2: quantity in stock */
				throw new Exception( sprintf( __( 'You cannot add that amount of &quot;%1$s&quot; to the quote because there is not enough stock (%2$s remaining).', 'quote-request-for-woocommerce' ), $product_data->get_name(), wc_format_stock_quantity_for_display( $product_data->get_stock_quantity(), $product_data ) ) );
			}

			// Stock check.
			if ( $product_data->managing_stock() ) {
				$products_qty_in_quote = $this->get_quote_item_quantities();

				if ( isset( $products_qty_in_quote[ $product_data->get_stock_managed_by_id() ] ) && ! $product_data->has_enough_stock( $products_qty_in_quote[ $product_data->get_stock_managed_by_id() ] + $quantity ) ) {
					throw new Exception( sprintf( '<a href="%s" class="button wc-forward">%s</a> %s', _qts_get_quote_url(), __( 'View quote', 'quote-request-for-woocommerce' ),
											/* translators: 1: quantity in stock 2: current quantity */ sprintf( __( 'You cannot add that amount to the quote &mdash; we have %1$s in stock and you already have %2$s in your quote.', 'quote-request-for-woocommerce' ), wc_format_stock_quantity_for_display( $product_data->get_stock_quantity(), $product_data ), wc_format_stock_quantity_for_display( $products_qty_in_quote[ $product_data->get_stock_managed_by_id() ], $product_data ) )
									) );
				}
			}

			if ( ! $quote_item_key ) {
				$quote_item_key = $quote_id;

				/**
				 * Get quote item to add.
				 * 
				 * @since 1.0
				 */
				$this->quote_contents[ $quote_item_key ] = apply_filters( 'qts_add_quote_item', array_merge( $quote_item_data, array(
					'key'                        => $quote_item_key,
					'product_id'                 => $product_id,
					'variation_id'               => $variation_id,
					'variation'                  => $variation,
					'quantity'                   => $quantity,
					'original_price'             => _qts_get_original_price( $product_data ),
					'requested_price'            => null,
					'requested_price_percent'    => null,
					'requested_discount_percent' => null,
					'data'                       => $product_data,
						) ), $quote_item_key );
			} else {
				$new_quantity = $quantity + $this->quote_contents[ $quote_item_key ][ 'quantity' ];
				$this->set_quantity( $quote_item_key, $new_quantity, false );
			}

			/**
			 * Triggers after added to quote.
			 * 
			 * @since 1.0
			 */
			do_action( 'qts_add_to_quote', $quote_item_key, $product_id, $quantity, $variation_id, $variation, $quote_item_data );

			return $quote_item_key;
		} catch ( Exception $e ) {
			if ( $e->getMessage() ) {
				wc_add_notice( $e->getMessage(), 'error' );
			}
			return false;
		}
	}

	/**
	 * Set the original product price for an item in the quote.
	 *
	 * @param string $quote_item_key
	 * @param float  $price
	 * @param bool   $refresh_totals
	 * @return bool
	 */
	public function set_original_price( $quote_item_key, $price = 0, $refresh_totals = true ) {
		if ( $price < 0 || ! is_numeric( $price ) ) {
			$price = 0;
		}

		// Update price.
		$this->quote_contents[ $quote_item_key ][ 'original_price' ] = wc_format_decimal( $price, wc_get_price_decimals() );

		if ( $refresh_totals ) {
			$this->calculate_totals();
		}

		return true;
	}

	/**
	 * Set the customer requested price for an item in the quote.
	 *
	 * @param string $quote_item_key
	 * @param float  $price
	 * @param bool   $refresh_totals
	 * @return bool
	 */
	public function set_requested_price( $quote_item_key, $price = 0, $refresh_totals = true ) {
		if ( 0 === $price || $price < 0 || ! is_numeric( $price ) ) {
			$this->quote_contents[ $quote_item_key ][ 'requested_price' ] = null;
			return false;
		}

		// Update price.
		$this->quote_contents[ $quote_item_key ][ 'requested_price' ] = wc_format_decimal( $price, wc_get_price_decimals() );

		if ( $refresh_totals ) {
			$this->calculate_totals();
		}

		return true;
	}

	/**
	 * Set the customer requested price percent for an item in the quote.
	 *
	 * @param string $quote_item_key
	 * @param float  $price
	 * @param bool   $refresh_totals
	 * @return bool
	 */
	public function set_requested_price_percent( $quote_item_key, $price = 0, $refresh_totals = true ) {
		if ( 0 === $price || $price < 0 || ! is_numeric( $price ) ) {
			$this->quote_contents[ $quote_item_key ][ 'requested_price_percent' ] = null;
			return false;
		}

		// Update price percent.
		$this->quote_contents[ $quote_item_key ][ 'requested_price_percent' ] = floatval( $price );

		if ( $refresh_totals ) {
			$this->calculate_totals();
		}

		return true;
	}

	/**
	 * Set the customer requested discount percent for an item in the quote.
	 *
	 * @param string $quote_item_key
	 * @param float  $discount_percent
	 * @param bool   $refresh_totals
	 * @return bool
	 */
	public function set_requested_discount_percent( $quote_item_key, $discount_percent = 0, $refresh_totals = true ) {
		if ( 0 === $discount_percent || $discount_percent < 0 || $discount_percent > 100 || ! is_numeric( $discount_percent ) ) {
			$this->quote_contents[ $quote_item_key ][ 'requested_discount_percent' ] = null;
			return false;
		}

		// Update discount percent.
		$this->quote_contents[ $quote_item_key ][ 'requested_discount_percent' ] = floatval( $discount_percent );

		if ( $refresh_totals ) {
			$this->calculate_totals();
		}

		return true;
	}

	/**
	 * Set the quantity for an item in the quote.
	 *
	 * @param string $quote_item_key
	 * @param int    $quantity
	 * @param bool   $refresh_totals
	 * @return bool
	 */
	public function set_quantity( $quote_item_key, $quantity = 1, $refresh_totals = true ) {
		if ( 0 === $quantity || $quantity < 0 ) {
			return $this->remove_quote_item( $quote_item_key );
		}

		// Update qty.
		$this->quote_contents[ $quote_item_key ][ 'quantity' ] = $quantity;

		if ( $refresh_totals ) {
			$this->calculate_totals();
		}

		return true;
	}

	/**
	 * Remove a quote item.
	 *
	 * @param  string $quote_item_key Quote item key to remove from the quote.
	 * @return bool
	 */
	public function remove_quote_item( $quote_item_key ) {
		if ( isset( $this->quote_contents[ $quote_item_key ] ) ) {
			$this->removed_quote_contents[ $quote_item_key ] = $this->quote_contents[ $quote_item_key ];
			unset( $this->removed_quote_contents[ $quote_item_key ][ 'data' ] );
			unset( $this->quote_contents[ $quote_item_key ] );

			/**
			 * Triggers when quote item is removed.
			 * 
			 * @since 1.0
			 */
			do_action( 'qts_quote_item_removed', $quote_item_key, $this );
			return true;
		}

		return false;
	}

	/**
	 * Restore a quote item.
	 *
	 * @param  string $quote_item_key Quote item key to restore to the quote.
	 * @return bool
	 */
	public function restore_quote_item( $quote_item_key ) {
		if ( isset( $this->removed_quote_contents[ $quote_item_key ] ) ) {
			$restore_item                                      = $this->removed_quote_contents[ $quote_item_key ];
			$this->quote_contents[ $quote_item_key ]           = $restore_item;
			$this->quote_contents[ $quote_item_key ][ 'data' ] = wc_get_product( $restore_item[ 'variation_id' ] ? $restore_item[ 'variation_id' ] : $restore_item[ 'product_id' ] );
			unset( $this->removed_quote_contents[ $quote_item_key ] );

			/**
			 * Triggers when quote item is restored.
			 * 
			 * @since 1.0
			 */
			do_action( 'qts_quote_item_restored', $quote_item_key, $this );
			return true;
		}

		return false;
	}

	/**
	 * Generate a unique ID for the quote item being added.
	 *
	 * @param int   $product_id - id of the product the key is being generated for.
	 * @param int   $variation_id of the product the key is being generated for.
	 * @param array $variation data for the quote item.
	 * @param array $quote_item_data
	 * @return string quote item key
	 */
	public function generate_quote_id( $product_id, $variation_id = 0, $variation = array(), $quote_item_data = array() ) {
		$id_parts = array( $product_id );

		if ( $variation_id && 0 !== $variation_id ) {
			$id_parts[] = $variation_id;
		}

		if ( is_array( $variation ) && ! empty( $variation ) ) {
			$variation_key = '';
			foreach ( $variation as $key => $value ) {
				$variation_key .= trim( $key ) . trim( $value );
			}

			$id_parts[] = $variation_key;
		}

		if ( is_array( $quote_item_data ) && ! empty( $quote_item_data ) ) {
			$quote_item_data_key = '';
			foreach ( $quote_item_data as $key => $value ) {
				if ( is_array( $value ) || is_object( $value ) ) {
					$value = http_build_query( $value );
				}
				$quote_item_data_key .= trim( $key ) . trim( $value );
			}

			$id_parts[] = $quote_item_data_key;
		}

		return md5( implode( '_', $id_parts ) );
	}

	/**
	 * Calculate totals for the items in the quote.
	 */
	public function calculate_totals() {
		$this->reset_totals();

		if ( $this->is_empty() ) {
			$this->session->set_session();
			return;
		}

		/**
		 * Triggers before calculating quote totals.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_before_calculate_totals', $this );

		new QTS_Quote_Totals( $this );

		/**
		 * Triggers after calculating quote totals.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_after_calculate_totals', $this );
	}

}
