<?php

defined( 'ABSPATH' ) || exit;

/**
 * Handle Admin menus, post types and settings.
 * 
 * @class QTS_Admin_Settings
 * @package Class
 */
class QTS_Admin_Settings {

	/**
	 * Section pages.
	 *
	 * @var array
	 */
	private static $sections = array();

	/**
	 * Init QTS_Admin_Settings.
	 */
	public static function init() {
		add_action( 'qts_reset_options', __CLASS__ . '::reset_options' );
		add_filter( 'woocommerce_account_settings', __CLASS__ . '::add_wc_account_settings' );
		add_action( 'admin_notices', __CLASS__ . '::gateway_needed_notice' );
	}

	/**
	 * Include the section classes.
	 */
	public static function get_sections() {
		if ( empty( self::$sections ) ) {
			self::$sections[] = include( 'settings-page/class-qts-admin-settings-general.php' );
			self::$sections[] = include( 'settings-page/class-qts-admin-settings-message.php' );
			self::$sections[] = include( 'settings-page/class-qts-admin-settings-localization.php' );
		}

		return self::$sections;
	}

	/**
	 * Settings page.
	 *
	 * Handles the display of the main Quote Request for WooCommerce settings page in admin.
	 */
	public static function output() {
		global $current_tab;

		/**
		 * Plugin settings start.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_settings_start' );

		$current_tab = ( empty( $_GET[ 'tab' ] ) ) ? 'general' : urldecode( sanitize_text_field( $_GET[ 'tab' ] ) );

		// Include settings pages
		self::get_sections();

		/**
		 * Add plugin default options based on tab requested.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_add_options_' . $current_tab );

		/**
		 * Add plugin default options.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_add_options' );

		if ( ! empty( $_POST[ 'save' ] ) ) {
			if ( empty( $_REQUEST[ QTS_PREFIX . 'nonce' ] ) || ! wp_verify_nonce( sanitize_key( $_REQUEST[ QTS_PREFIX . 'nonce' ] ), 'qts-settings' ) ) {
				die( esc_html__( 'Action failed. Please refresh the page and retry.', 'quote-request-for-woocommerce' ) );
			}

			/**
			 * Save plugin options when saved based on tab requested.
			 * 
			 * @param mixed $_POST
			 * @since 1.0
			 */
			do_action( 'qts_update_options_' . $current_tab, $_POST );

			/**
			 * Save plugin options when saved.
			 * 
			 * @param mixed $_POST
			 * @since 1.0
			 */
			do_action( 'qts_update_options', $_POST );

			wp_safe_redirect( esc_url_raw( add_query_arg( array( 'saved' => 'true' ) ) ) );
			exit;
		}
		if ( ! empty( $_POST[ 'reset' ] ) || ! empty( $_POST[ 'reset_all' ] ) ) {
			if ( empty( $_REQUEST[ QTS_PREFIX . 'nonce' ] ) || ! wp_verify_nonce( sanitize_key( $_REQUEST[ QTS_PREFIX . 'nonce' ] ), 'qts-reset-settings' ) ) {
				die( esc_html__( 'Action failed. Please refresh the page and retry.', 'quote-request-for-woocommerce' ) );
			}

			/**
			 * Reset plugin to default options based on tab requested.
			 * 
			 * @param mixed $_POST
			 * @since 1.0
			 */
			do_action( 'qts_reset_options_' . $current_tab, $_POST );

			if ( ! empty( $_POST[ 'reset_all' ] ) ) {
				/**
				 * Reset plugin to default options.
				 * 
				 * @param mixed $_POST
				 * @since 1.0
				 */
				do_action( 'qts_reset_options', $_POST );
			}

			wp_safe_redirect( esc_url_raw( add_query_arg( array( 'saved' => 'true' ) ) ) );
			exit;
		}
		// Get any returned messages
		$error   = ( empty( $_GET[ 'wc_error' ] ) ) ? '' : urldecode( stripslashes( sanitize_title( $_GET[ 'wc_error' ] ) ) );
		$message = ( empty( $_GET[ 'wc_message' ] ) ) ? '' : urldecode( stripslashes( sanitize_title( $_GET[ 'wc_message' ] ) ) );

		if ( $error || $message ) {
			if ( $error ) {
				echo '<div id="message" class="error fade"><p><strong>' . esc_html( $error ) . '</strong></p></div>';
			} else {
				echo '<div id="message" class="updated fade"><p><strong>' . esc_html( $message ) . '</strong></p></div>';
			}
		} elseif ( ! empty( $_GET[ 'saved' ] ) ) {
			echo '<div id="message" class="updated fade"><p><strong>' . esc_html__( 'Your settings have been saved.', 'quote-request-for-woocommerce' ) . '</strong></p></div>';
		}

		include 'views/html-admin-settings.php';
	}

	/**
	 * Default options.
	 *
	 * Sets up the default options used on the settings page.
	 */
	public static function save_default_options( $reset_all = false ) {
		if ( empty( self::$sections ) ) {
			self::get_sections();
		}

		foreach ( self::$sections as $tab ) {
			if ( ! isset( $tab->settings ) || ! is_array( $tab->settings ) ) {
				continue;
			}

			$tab->add_options( $reset_all );
		}
	}

	/**
	 * Reset All settings
	 */
	public static function reset_options() {
		self::save_default_options( true );
	}

	/**
	 * Add privacy settings under WooCommerce Privacy
	 *
	 * @param array $settings
	 * @return array
	 */
	public static function add_wc_account_settings( $settings ) {
		$original_settings = $settings;

		$erasure_text = esc_html__( 'account erasure request', 'quote-request-for-woocommerce' );
		if ( current_user_can( 'manage_privacy_options' ) ) {
			if ( version_compare( get_bloginfo( 'version' ), '5.3', '<' ) ) {
				$erasure_text = sprintf( '<a href="%s">%s</a>', esc_url( admin_url( 'tools.php?page=remove_personal_data' ) ), $erasure_text );
			} else {
				$erasure_text = sprintf( '<a href="%s">%s</a>', esc_url( admin_url( 'erase-personal-data.php' ) ), $erasure_text );
			}
		}

		if ( ! empty( $original_settings ) ) {
			$new_settings = array();

			foreach ( $original_settings as $pos => $setting ) {
				if ( ! isset( $setting[ 'id' ] ) ) {
					continue;
				}

				switch ( $setting[ 'id' ] ) {
					case 'woocommerce_erasure_request_removes_order_data':
						$new_settings[ $pos + 1 ] = array(
							'title'         => esc_html__( 'Account erasure requests', 'quote-request-for-woocommerce' ),
							'desc'          => esc_html__( 'Remove personal data from Quote Request for WooCommerce', 'quote-request-for-woocommerce' ),
							/* Translators: %s URL to erasure request screen. */
							'desc_tip'      => sprintf( __( 'When handling an %s, should personal data within Quote Request for WooCommerce be retained or removed?', 'quote-request-for-woocommerce' ), $erasure_text ),
							'id'            => QTS_PREFIX . 'erasure_request_removes_user_data',
							'type'          => 'checkbox',
							'default'       => 'no',
							'checkboxgroup' => '',
							'autoload'      => false,
						);
						break;
				}
			}
			if ( ! empty( $new_settings ) ) {
				foreach ( $new_settings as $pos => $new_setting ) {
					array_splice( $settings, $pos, 0, array( $new_setting ) );
				}
			}
		}
		return $settings;
	}

	/**
	 * Output admin notices when the Quote Request payment gateway is needed in order for the plugin to work.
	 */
	public static function gateway_needed_notice() {
		if ( is_null( WC()->payment_gateways ) ) {
			return;
		}

		if ( 'approx' === get_option( QTS_PREFIX . 'quote_request_mode' ) ) {
			return;
		}

		$gateways = WC()->payment_gateways->payment_gateways();
		$enabled  = false;

		foreach ( $gateways as $gateway ) {
			if ( 'yes' === $gateway->enabled && $gateway->supports( 'qtsquote_request' ) ) {
				add_filter( 'qts_is_quote_request_payment_gateway_available_by_default', '__return_true' );
				$enabled = true;
				remove_filter( 'qts_is_quote_request_payment_gateway_available_by_default', '__return_true' );
				break;
			}
		}

		if ( ! $enabled ) {
			printf( '<div class="error"><p>%s</p></div>', esc_html__( 'Note: "Quote Request" Payment Gateway has to be enabled in order for the user to submit a Quote in "Accurate Quote(Final Amount to be Paid)" mode or "Let the User Decide" mode.', 'quote-request-for-woocommerce' ) );
		}
	}

}

QTS_Admin_Settings::init();
